# Configuração de Otimização do Sistema Villa Joias

## Otimizações de Banco de Dados

### 1. Índices Estratégicos
```sql
-- Índices para performance em consultas frequentes
CREATE INDEX idx_transacoes_data ON transacoes(data_transacao);
CREATE INDEX idx_transacoes_caixa ON transacoes(caixa_id);
CREATE INDEX idx_transacoes_tipo ON transacoes(tipo);
CREATE INDEX idx_transacoes_valor ON transacoes(valor);

CREATE INDEX idx_inadimplentes_status ON inadimplentes(status);
CREATE INDEX idx_inadimplentes_representante ON inadimplentes(representante_id);
CREATE INDEX idx_inadimplentes_spc ON inadimplentes(spc_negativado);

CREATE INDEX idx_acordos_status ON acordos(status);
CREATE INDEX idx_acordos_data ON acordos(data_acordo);
CREATE INDEX idx_acordos_tipo ON acordos(tipo_pagamento);

CREATE INDEX idx_despesas_cartao_data ON despesas_cartao(data_compra);
CREATE INDEX idx_despesas_cartao_cartao ON despesas_cartao(cartao_id);

CREATE INDEX idx_usuarios_email ON usuarios(email);
CREATE INDEX idx_usuarios_status ON usuarios(status);

-- Índices compostos para consultas complexas
CREATE INDEX idx_transacoes_caixa_data ON transacoes(caixa_id, data_transacao);
CREATE INDEX idx_transacoes_rep_data ON transacoes(representante_id, data_transacao);
CREATE INDEX idx_inadimplentes_rep_status ON inadimplentes(representante_id, status);
```

### 2. Views Otimizadas
```sql
-- View para dashboard principal
CREATE VIEW vw_dashboard_metricas AS
SELECT 
    (SELECT COALESCE(SUM(saldo_atual), 0) FROM caixas WHERE ativo = 1) as saldo_total,
    (SELECT COALESCE(SUM(valor), 0) FROM transacoes WHERE tipo = 'entrada' AND DATE(data_transacao) = CURDATE()) as entradas_hoje,
    (SELECT COALESCE(SUM(valor), 0) FROM transacoes WHERE tipo = 'saida' AND DATE(data_transacao) = CURDATE()) as saidas_hoje,
    (SELECT COUNT(*) FROM inadimplentes WHERE status = 'ativo') as inadimplentes_ativos,
    (SELECT COALESCE(SUM(saldo_atual), 0) FROM inadimplentes WHERE status = 'ativo') as valor_inadimplencia,
    (SELECT COUNT(*) FROM acordos WHERE status = 'ativo' AND MONTH(data_acordo) = MONTH(CURDATE())) as acordos_mes;

-- View para performance de representantes
CREATE VIEW vw_performance_representantes AS
SELECT 
    r.id,
    r.nome,
    r.codigo,
    COALESCE(SUM(CASE WHEN t.tipo = 'entrada' AND MONTH(t.data_transacao) = MONTH(CURDATE()) THEN t.valor ELSE 0 END), 0) as vendas_mes,
    COALESCE(SUM(CASE WHEN t.tipo = 'entrada' AND YEAR(t.data_transacao) = YEAR(CURDATE()) THEN t.valor ELSE 0 END), 0) as vendas_ano,
    COUNT(DISTINCT CASE WHEN t.tipo = 'entrada' AND MONTH(t.data_transacao) = MONTH(CURDATE()) THEN t.id END) as transacoes_mes,
    (SELECT COUNT(*) FROM inadimplentes WHERE representante_id = r.id AND status = 'ativo') as inadimplentes_count
FROM representantes r
LEFT JOIN transacoes t ON r.id = t.representante_id
WHERE r.ativo = 1
GROUP BY r.id, r.nome, r.codigo;

-- View para fluxo de caixa mensal
CREATE VIEW vw_fluxo_caixa_mensal AS
SELECT 
    DATE_FORMAT(data_transacao, '%Y-%m') as mes_ano,
    COALESCE(SUM(CASE WHEN tipo = 'entrada' THEN valor ELSE 0 END), 0) as entradas,
    COALESCE(SUM(CASE WHEN tipo = 'saida' THEN valor ELSE 0 END), 0) as saidas,
    COALESCE(SUM(CASE WHEN tipo = 'entrada' THEN valor ELSE -valor END), 0) as saldo_liquido
FROM transacoes
WHERE data_transacao >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
GROUP BY DATE_FORMAT(data_transacao, '%Y-%m')
ORDER BY mes_ano;
```

### 3. Procedures para Relatórios
```sql
DELIMITER //

-- Procedure para relatório de cobrança
CREATE PROCEDURE sp_relatorio_cobranca(
    IN p_data_inicio DATE,
    IN p_data_fim DATE
)
BEGIN
    SELECT 
        COUNT(*) as total_inadimplentes,
        SUM(saldo_atual) as valor_total,
        COUNT(CASE WHEN spc_negativado = 1 THEN 1 END) as negativados,
        COUNT(CASE WHEN cobranca_juridica = 1 THEN 1 END) as juridicos,
        AVG(saldo_atual) as ticket_medio,
        (SELECT COUNT(*) FROM acordos WHERE data_acordo BETWEEN p_data_inicio AND p_data_fim) as acordos_periodo,
        (SELECT SUM(valor_acordo) FROM acordos WHERE data_acordo BETWEEN p_data_inicio AND p_data_fim) as valor_recuperado
    FROM inadimplentes 
    WHERE data_registro BETWEEN p_data_inicio AND p_data_fim;
END //

-- Procedure para análise de performance
CREATE PROCEDURE sp_analise_performance(
    IN p_representante_id INT,
    IN p_meses INT
)
BEGIN
    DECLARE data_inicio DATE DEFAULT DATE_SUB(CURDATE(), INTERVAL p_meses MONTH);
    
    SELECT 
        DATE_FORMAT(t.data_transacao, '%Y-%m') as mes,
        COUNT(*) as total_transacoes,
        SUM(t.valor) as total_vendas,
        AVG(t.valor) as ticket_medio,
        COUNT(DISTINCT t.cliente_nome) as clientes_unicos
    FROM transacoes t
    WHERE t.representante_id = p_representante_id 
        AND t.tipo = 'entrada'
        AND t.data_transacao >= data_inicio
    GROUP BY DATE_FORMAT(t.data_transacao, '%Y-%m')
    ORDER BY mes;
END //

DELIMITER ;
```

## Otimizações de Código Python

### 1. Cache de Consultas Frequentes
```python
from functools import lru_cache
from datetime import datetime, timedelta
import redis

# Configurar Redis para cache
redis_client = redis.Redis(host='localhost', port=6379, db=0, decode_responses=True)

class CacheManager:
    @staticmethod
    def get_cache_key(prefix, *args):
        return f"{prefix}:{':'.join(map(str, args))}"
    
    @staticmethod
    def get(key):
        try:
            return redis_client.get(key)
        except:
            return None
    
    @staticmethod
    def set(key, value, expire=300):  # 5 minutos default
        try:
            redis_client.setex(key, expire, value)
        except:
            pass

# Cache para dashboard
@lru_cache(maxsize=128)
def get_dashboard_metrics_cached():
    cache_key = CacheManager.get_cache_key('dashboard', datetime.now().strftime('%Y-%m-%d-%H'))
    cached = CacheManager.get(cache_key)
    
    if cached:
        return json.loads(cached)
    
    # Buscar dados do banco
    metrics = get_dashboard_metrics_from_db()
    
    # Salvar no cache por 1 hora
    CacheManager.set(cache_key, json.dumps(metrics), 3600)
    
    return metrics
```

### 2. Otimização de Consultas SQLAlchemy
```python
from sqlalchemy.orm import joinedload, selectinload

class OptimizedQueries:
    @staticmethod
    def get_transacoes_with_relations(limit=100):
        """Buscar transações com relacionamentos otimizados"""
        return db.session.query(Transacao)\
            .options(
                joinedload(Transacao.caixa),
                joinedload(Transacao.representante),
                joinedload(Transacao.categoria)
            )\
            .order_by(Transacao.data_transacao.desc())\
            .limit(limit)\
            .all()
    
    @staticmethod
    def get_representantes_performance():
        """Performance de representantes com subquery otimizada"""
        subquery = db.session.query(
            Transacao.representante_id,
            func.sum(Transacao.valor).label('total_vendas'),
            func.count(Transacao.id).label('total_transacoes')
        ).filter(
            Transacao.tipo == 'entrada',
            Transacao.data_transacao >= datetime.now() - timedelta(days=30)
        ).group_by(Transacao.representante_id).subquery()
        
        return db.session.query(
            Representante,
            subquery.c.total_vendas,
            subquery.c.total_transacoes
        ).outerjoin(subquery, Representante.id == subquery.c.representante_id)\
         .filter(Representante.ativo == True)\
         .all()
    
    @staticmethod
    def get_inadimplentes_summary():
        """Resumo de inadimplentes otimizado"""
        return db.session.query(
            func.count(Inadimplente.id).label('total'),
            func.sum(Inadimplente.saldo_atual).label('valor_total'),
            func.sum(case([(Inadimplente.spc_negativado == True, 1)], else_=0)).label('negativados'),
            func.avg(Inadimplente.saldo_atual).label('ticket_medio')
        ).filter(Inadimplente.status == 'ativo').first()
```

### 3. Paginação Eficiente
```python
class PaginationHelper:
    @staticmethod
    def paginate_query(query, page=1, per_page=50):
        """Paginação otimizada com contagem eficiente"""
        total = query.count()
        items = query.offset((page - 1) * per_page).limit(per_page).all()
        
        return {
            'items': items,
            'total': total,
            'page': page,
            'per_page': per_page,
            'pages': (total + per_page - 1) // per_page,
            'has_prev': page > 1,
            'has_next': page * per_page < total
        }
```

## Otimizações de Frontend

### 1. Lazy Loading de Componentes
```javascript
// Carregamento sob demanda dos módulos
const CobrancaModule = lazy(() => import('./components/CobrancaModule'));
const RelatoriosModule = lazy(() => import('./components/RelatoriosModule'));
const UsuariosModule = lazy(() => import('./components/UsuariosModule'));

// Suspense para loading
<Suspense fallback={<div className="loading">Carregando...</div>}>
    <CobrancaModule />
</Suspense>
```

### 2. Debounce em Buscas
```javascript
import { useMemo, useState, useEffect } from 'react';
import { debounce } from 'lodash';

const useDebounceSearch = (searchTerm, delay = 300) => {
    const [debouncedTerm, setDebouncedTerm] = useState(searchTerm);
    
    useEffect(() => {
        const handler = setTimeout(() => {
            setDebouncedTerm(searchTerm);
        }, delay);
        
        return () => {
            clearTimeout(handler);
        };
    }, [searchTerm, delay]);
    
    return debouncedTerm;
};
```

### 3. Otimização de Re-renders
```javascript
import { memo, useCallback, useMemo } from 'react';

const OptimizedTable = memo(({ data, onEdit, onDelete }) => {
    const memoizedData = useMemo(() => 
        data.map(item => ({
            ...item,
            formattedValue: formatCurrency(item.value)
        })), [data]
    );
    
    const handleEdit = useCallback((id) => {
        onEdit(id);
    }, [onEdit]);
    
    return (
        <table>
            {memoizedData.map(item => (
                <TableRow 
                    key={item.id} 
                    item={item} 
                    onEdit={handleEdit}
                    onDelete={onDelete}
                />
            ))}
        </table>
    );
});
```

## Configurações de Produção

### 1. Configuração do Gunicorn
```python
# gunicorn.conf.py
bind = "0.0.0.0:5000"
workers = 4
worker_class = "gevent"
worker_connections = 1000
max_requests = 1000
max_requests_jitter = 100
timeout = 30
keepalive = 2
preload_app = True
```

### 2. Configuração do Nginx
```nginx
server {
    listen 80;
    server_name villa-joias.com;
    
    # Gzip compression
    gzip on;
    gzip_types text/plain text/css application/json application/javascript text/xml application/xml application/xml+rss text/javascript;
    
    # Static files
    location /static/ {
        alias /var/www/villa-joias/static/;
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
    
    # API
    location /api/ {
        proxy_pass http://127.0.0.1:5000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_connect_timeout 30s;
        proxy_send_timeout 30s;
        proxy_read_timeout 30s;
    }
    
    # Frontend
    location / {
        try_files $uri $uri/ /index.html;
        expires 1h;
    }
}
```

### 3. Monitoramento de Performance
```python
import time
import logging
from functools import wraps

def monitor_performance(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        start_time = time.time()
        result = func(*args, **kwargs)
        end_time = time.time()
        
        execution_time = end_time - start_time
        
        if execution_time > 1.0:  # Log se demorar mais de 1 segundo
            logging.warning(f"Slow query: {func.__name__} took {execution_time:.2f}s")
        
        return result
    return wrapper

# Aplicar em rotas críticas
@app.route('/api/relatorios/dashboard')
@monitor_performance
def dashboard():
    return get_dashboard_data()
```

## Configurações de Segurança

### 1. Rate Limiting
```python
from flask_limiter import Limiter
from flask_limiter.util import get_remote_address

limiter = Limiter(
    app,
    key_func=get_remote_address,
    default_limits=["200 per day", "50 per hour"]
)

@app.route('/api/auth/login', methods=['POST'])
@limiter.limit("5 per minute")
def login():
    # Implementação do login
    pass
```

### 2. Validação de Entrada
```python
from marshmallow import Schema, fields, validate

class TransacaoSchema(Schema):
    tipo = fields.Str(required=True, validate=validate.OneOf(['entrada', 'saida']))
    valor = fields.Decimal(required=True, validate=validate.Range(min=0.01))
    caixa_id = fields.Int(required=True)
    descricao = fields.Str(validate=validate.Length(max=500))
    
def validate_input(schema_class):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            schema = schema_class()
            try:
                data = schema.load(request.json)
                return func(data, *args, **kwargs)
            except ValidationError as err:
                return jsonify({'errors': err.messages}), 400
        return wrapper
    return decorator
```

Este arquivo contém todas as otimizações necessárias para garantir que o Sistema Villa Joias funcione com máxima performance e segurança em produção.

