#!/bin/bash

# Script de Configuração SSL - Sistema Villa Joias
# Versão: 2.0 VPS Edition
# Uso: ./configure_ssl.sh seudominio.com.br email@exemplo.com

DOMAIN=$1
EMAIL=$2

if [ -z "$DOMAIN" ] || [ -z "$EMAIL" ]; then
    echo "❌ Erro: Você deve especificar domínio e email"
    echo "Uso: ./configure_ssl.sh seudominio.com.br email@exemplo.com"
    exit 1
fi

if [ "$EUID" -ne 0 ]; then
    echo "❌ Este script deve ser executado como root (sudo)"
    exit 1
fi

echo "🔒 Configurando SSL para: $DOMAIN"
echo "=================================="

# Verificar se Certbot está instalado
if ! command -v certbot &> /dev/null; then
    echo "📦 Instalando Certbot..."
    apt install snapd -y
    snap install core
    snap refresh core
    snap install --classic certbot
    ln -sf /snap/bin/certbot /usr/bin/certbot
fi

# Verificar se o domínio está apontando para o servidor
echo "🔍 Verificando DNS do domínio..."
SERVER_IP=$(curl -s ifconfig.me)
DOMAIN_IP=$(dig +short $DOMAIN)

if [ "$SERVER_IP" != "$DOMAIN_IP" ]; then
    echo "⚠️  AVISO: O domínio $DOMAIN não está apontando para este servidor"
    echo "   IP do servidor: $SERVER_IP"
    echo "   IP do domínio: $DOMAIN_IP"
    echo ""
    read -p "Deseja continuar mesmo assim? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "❌ Configuração SSL cancelada"
        exit 1
    fi
fi

# Verificar se o site está respondendo
echo "🌐 Verificando se o site está acessível..."
if curl -s -o /dev/null -w "%{http_code}" "http://$DOMAIN" | grep -q "200\|301\|302"; then
    echo "✅ Site está acessível"
else
    echo "❌ Site não está acessível em http://$DOMAIN"
    echo "   Verifique se o Virtual Host está configurado corretamente"
    exit 1
fi

# Obter certificado SSL
echo "🔐 Obtendo certificado SSL..."
certbot --apache \
    --non-interactive \
    --agree-tos \
    --email "$EMAIL" \
    --domains "$DOMAIN,www.$DOMAIN" \
    --redirect

if [ $? -eq 0 ]; then
    echo "✅ Certificado SSL configurado com sucesso!"
    
    # Testar renovação automática
    echo "🔄 Testando renovação automática..."
    if certbot renew --dry-run; then
        echo "✅ Renovação automática configurada"
    else
        echo "⚠️  Aviso: Problema na configuração de renovação automática"
    fi
    
    # Atualizar .htaccess para forçar HTTPS
    echo "🔧 Atualizando .htaccess para forçar HTTPS..."
    sed -i 's/# RewriteCond %{HTTPS} off/RewriteCond %{HTTPS} off/' /var/www/html/villa_joias/.htaccess
    sed -i 's/# RewriteRule \^(.*)\$ https:\/\/%{HTTP_HOST}%{REQUEST_URI} \[L,R=301\]/RewriteRule ^(.*)$ https:\/\/%{HTTP_HOST}%{REQUEST_URI} [L,R=301]/' /var/www/html/villa_joias/.htaccess
    
    # Reiniciar Apache
    systemctl restart apache2
    
    echo ""
    echo "🎉 SSL CONFIGURADO COM SUCESSO!"
    echo "==============================="
    echo ""
    echo "🔒 Certificado: Let's Encrypt"
    echo "🌐 Domínio: $DOMAIN"
    echo "📧 Email: $EMAIL"
    echo "🔄 Renovação: Automática"
    echo ""
    echo "✅ Acesse: https://$DOMAIN"
    echo "✅ Teste SSL: https://www.ssllabs.com/ssltest/analyze.html?d=$DOMAIN"
    echo ""
    
else
    echo "❌ Erro ao configurar SSL"
    echo "Possíveis causas:"
    echo "- Domínio não está apontando para este servidor"
    echo "- Firewall bloqueando porta 80/443"
    echo "- Apache não está configurado corretamente"
    echo ""
    echo "Logs do Certbot:"
    tail -20 /var/log/letsencrypt/letsencrypt.log
fi

