-- =====================================================
-- SISTEMA DE CONTROLE DE FLUXO DE CAIXA VILLA JOIAS
-- Estrutura Completa do Banco de Dados
-- Versão: 1.0
-- Data: 19/06/2025
-- =====================================================

SET FOREIGN_KEY_CHECKS = 0;
DROP DATABASE IF EXISTS villa_joias;
CREATE DATABASE villa_joias CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE villa_joias;

-- =====================================================
-- TABELA DE USUÁRIOS E PERMISSÕES
-- =====================================================

CREATE TABLE usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha_hash VARCHAR(255) NOT NULL,
    tipo_usuario ENUM('administrador', 'gerente', 'operador', 'consulta', 'personalizado') DEFAULT 'operador',
    ativo BOOLEAN DEFAULT TRUE,
    ultimo_login TIMESTAMP NULL,
    tentativas_login INT DEFAULT 0,
    bloqueado_ate TIMESTAMP NULL,
    token_2fa VARCHAR(32) NULL,
    ativo_2fa BOOLEAN DEFAULT FALSE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    INDEX idx_email (email),
    INDEX idx_tipo (tipo_usuario),
    INDEX idx_ativo (ativo)
);

CREATE TABLE permissoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    modulo VARCHAR(50) NOT NULL,
    acao VARCHAR(50) NOT NULL,
    permitido BOOLEAN DEFAULT TRUE,
    restricoes JSON NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    UNIQUE KEY unique_permissao (usuario_id, modulo, acao),
    INDEX idx_usuario_modulo (usuario_id, modulo)
);

CREATE TABLE logs_auditoria (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NULL,
    acao VARCHAR(100) NOT NULL,
    tabela_afetada VARCHAR(50) NULL,
    registro_id INT NULL,
    dados_anteriores JSON NULL,
    dados_novos JSON NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_usuario_data (usuario_id, criado_em),
    INDEX idx_tabela_registro (tabela_afetada, registro_id),
    INDEX idx_data (criado_em)
);

-- =====================================================
-- TABELAS DE REPRESENTANTES E CLIENTES
-- =====================================================

CREATE TABLE representantes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    codigo VARCHAR(20) UNIQUE NOT NULL,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) NULL,
    telefone VARCHAR(20) NULL,
    regiao VARCHAR(50) NULL,
    comissao_padrao DECIMAL(5,2) DEFAULT 0.00,
    meta_mensal DECIMAL(15,2) DEFAULT 0.00,
    ativo BOOLEAN DEFAULT TRUE,
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    INDEX idx_codigo (codigo),
    INDEX idx_nome (nome),
    INDEX idx_ativo (ativo),
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL
);

CREATE TABLE clientes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    representante_id INT NOT NULL,
    telefone VARCHAR(20) NULL,
    email VARCHAR(100) NULL,
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (representante_id) REFERENCES representantes(id) ON DELETE RESTRICT,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_nome (nome),
    INDEX idx_representante (representante_id)
);

-- =====================================================
-- TABELAS DE CAIXAS E CONTAS
-- =====================================================

CREATE TABLE caixas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    tipo ENUM('capital_giro', 'reserva', 'conta_corrente', 'fisico', 'outros') NOT NULL,
    descricao TEXT NULL,
    instituicao VARCHAR(100) NULL,
    saldo_inicial DECIMAL(15,2) DEFAULT 0.00,
    saldo_atual DECIMAL(15,2) DEFAULT 0.00,
    limite_cheque_especial DECIMAL(15,2) DEFAULT 0.00,
    taxa_rendimento DECIMAL(8,4) DEFAULT 0.00,
    periodicidade_rendimento ENUM('diario', 'mensal', 'trimestral', 'anual') DEFAULT 'mensal',
    ativo BOOLEAN DEFAULT TRUE,
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_nome (nome),
    INDEX idx_tipo (tipo),
    INDEX idx_ativo (ativo)
);

-- =====================================================
-- TABELAS DE CATEGORIAS E SUBCATEGORIAS
-- =====================================================

CREATE TABLE categorias (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    descricao TEXT NULL,
    cor VARCHAR(7) DEFAULT '#2C3E50',
    icone VARCHAR(50) NULL,
    orcamento_mensal DECIMAL(15,2) DEFAULT 0.00,
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_nome (nome),
    INDEX idx_ativo (ativo)
);

CREATE TABLE subcategorias (
    id INT AUTO_INCREMENT PRIMARY KEY,
    categoria_id INT NOT NULL,
    nome VARCHAR(100) NOT NULL,
    descricao TEXT NULL,
    orcamento_mensal DECIMAL(15,2) DEFAULT 0.00,
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (categoria_id) REFERENCES categorias(id) ON DELETE CASCADE,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_categoria (categoria_id),
    INDEX idx_nome (nome),
    INDEX idx_ativo (ativo)
);

-- =====================================================
-- TABELAS DE CARTÕES DE CRÉDITO
-- =====================================================

CREATE TABLE cartoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    instituicao VARCHAR(100) NULL,
    limite_total DECIMAL(15,2) DEFAULT 0.00,
    limite_disponivel DECIMAL(15,2) DEFAULT 0.00,
    dia_fechamento INT NOT NULL CHECK (dia_fechamento BETWEEN 1 AND 31),
    dia_vencimento INT NOT NULL CHECK (dia_vencimento BETWEEN 1 AND 31),
    ativo BOOLEAN DEFAULT TRUE,
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_nome (nome),
    INDEX idx_ativo (ativo)
);

CREATE TABLE faturas_cartao (
    id INT AUTO_INCREMENT PRIMARY KEY,
    cartao_id INT NOT NULL,
    mes_referencia DATE NOT NULL,
    valor_total DECIMAL(15,2) DEFAULT 0.00,
    valor_pago DECIMAL(15,2) DEFAULT 0.00,
    valor_pendente DECIMAL(15,2) DEFAULT 0.00,
    data_vencimento DATE NOT NULL,
    status ENUM('aberta', 'paga_total', 'paga_parcial', 'vencida') DEFAULT 'aberta',
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (cartao_id) REFERENCES cartoes(id) ON DELETE CASCADE,
    UNIQUE KEY unique_fatura (cartao_id, mes_referencia),
    INDEX idx_cartao_mes (cartao_id, mes_referencia),
    INDEX idx_vencimento (data_vencimento),
    INDEX idx_status (status)
);

-- =====================================================
-- TABELA PRINCIPAL DE TRANSAÇÕES
-- =====================================================

CREATE TABLE transacoes (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    tipo ENUM('entrada', 'saida', 'transferencia', 'rendimento', 'taxa', 'ajuste') NOT NULL,
    subtipo VARCHAR(50) NULL,
    data_transacao DATE NOT NULL,
    valor DECIMAL(15,2) NOT NULL,
    descricao TEXT NOT NULL,
    
    -- Relacionamentos
    caixa_origem_id INT NULL,
    caixa_destino_id INT NULL,
    representante_id INT NULL,
    cliente_id INT NULL,
    categoria_id INT NULL,
    subcategoria_id INT NULL,
    cartao_id INT NULL,
    fatura_cartao_id INT NULL,
    
    -- Dados específicos
    metodo_pagamento ENUM('dinheiro', 'pix', 'transferencia', 'cartao_debito', 'cartao_credito', 'cheque', 'outros') NULL,
    numero_parcelas INT DEFAULT 1,
    parcela_atual INT DEFAULT 1,
    valor_original DECIMAL(15,2) NULL,
    
    -- Controle
    status ENUM('pendente', 'confirmado', 'cancelado', 'estornado') DEFAULT 'confirmado',
    conciliado BOOLEAN DEFAULT FALSE,
    data_conciliacao TIMESTAMP NULL,
    
    -- Auditoria
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    
    -- Chaves estrangeiras
    FOREIGN KEY (caixa_origem_id) REFERENCES caixas(id) ON DELETE RESTRICT,
    FOREIGN KEY (caixa_destino_id) REFERENCES caixas(id) ON DELETE RESTRICT,
    FOREIGN KEY (representante_id) REFERENCES representantes(id) ON DELETE SET NULL,
    FOREIGN KEY (cliente_id) REFERENCES clientes(id) ON DELETE SET NULL,
    FOREIGN KEY (categoria_id) REFERENCES categorias(id) ON DELETE SET NULL,
    FOREIGN KEY (subcategoria_id) REFERENCES subcategorias(id) ON DELETE SET NULL,
    FOREIGN KEY (cartao_id) REFERENCES cartoes(id) ON DELETE SET NULL,
    FOREIGN KEY (fatura_cartao_id) REFERENCES faturas_cartao(id) ON DELETE SET NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    
    -- Índices para performance
    INDEX idx_tipo_data (tipo, data_transacao),
    INDEX idx_caixa_origem (caixa_origem_id),
    INDEX idx_caixa_destino (caixa_destino_id),
    INDEX idx_representante (representante_id),
    INDEX idx_categoria (categoria_id, subcategoria_id),
    INDEX idx_cartao (cartao_id),
    INDEX idx_data (data_transacao),
    INDEX idx_status (status),
    INDEX idx_conciliado (conciliado)
);

-- =====================================================
-- TABELAS DE DÍVIDAS E PARCELAS
-- =====================================================

CREATE TABLE dividas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(200) NOT NULL,
    descricao TEXT NULL,
    valor_total DECIMAL(15,2) NOT NULL,
    valor_pago DECIMAL(15,2) DEFAULT 0.00,
    valor_pendente DECIMAL(15,2) NOT NULL,
    numero_parcelas INT NOT NULL,
    valor_parcela DECIMAL(15,2) NOT NULL,
    data_primeira_parcela DATE NOT NULL,
    periodicidade ENUM('semanal', 'quinzenal', 'mensal', 'bimestral', 'trimestral') DEFAULT 'mensal',
    categoria_id INT NULL,
    subcategoria_id INT NULL,
    status ENUM('ativa', 'quitada', 'cancelada') DEFAULT 'ativa',
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    FOREIGN KEY (categoria_id) REFERENCES categorias(id) ON DELETE SET NULL,
    FOREIGN KEY (subcategoria_id) REFERENCES subcategorias(id) ON DELETE SET NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_status (status),
    INDEX idx_categoria (categoria_id)
);

CREATE TABLE parcelas_divida (
    id INT AUTO_INCREMENT PRIMARY KEY,
    divida_id INT NOT NULL,
    numero_parcela INT NOT NULL,
    data_vencimento DATE NOT NULL,
    valor_previsto DECIMAL(15,2) NOT NULL,
    valor_pago DECIMAL(15,2) DEFAULT 0.00,
    data_pagamento DATE NULL,
    caixa_pagamento_id INT NULL,
    status ENUM('pendente', 'paga', 'atrasada', 'cancelada') DEFAULT 'pendente',
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (divida_id) REFERENCES dividas(id) ON DELETE CASCADE,
    FOREIGN KEY (caixa_pagamento_id) REFERENCES caixas(id) ON DELETE SET NULL,
    UNIQUE KEY unique_parcela (divida_id, numero_parcela),
    INDEX idx_divida (divida_id),
    INDEX idx_vencimento (data_vencimento),
    INDEX idx_status (status)
);

-- =====================================================
-- TABELAS DE INADIMPLÊNCIA E COBRANÇA
-- =====================================================

CREATE TABLE inadimplentes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Dados pessoais
    nome VARCHAR(100) NOT NULL,
    cpf VARCHAR(14) UNIQUE NOT NULL,
    telefone1 VARCHAR(20) NOT NULL,
    telefone2 VARCHAR(20) NULL,
    
    -- Endereço
    cep VARCHAR(10) NULL,
    rua VARCHAR(200) NULL,
    numero VARCHAR(20) NULL,
    bairro VARCHAR(100) NULL,
    cidade VARCHAR(100) NULL,
    estado VARCHAR(2) NULL,
    
    -- Dados da dívida
    representante_id INT NULL,
    numero_contrato VARCHAR(50) NULL,
    saldo_original DECIMAL(15,2) NOT NULL,
    saldo_atual DECIMAL(15,2) NOT NULL,
    taxa_juros DECIMAL(8,4) DEFAULT 0.00,
    taxa_multa DECIMAL(8,4) DEFAULT 0.00,
    
    -- Status e situação
    negativado_spc BOOLEAN DEFAULT FALSE,
    data_negativacao DATE NULL,
    cobranca_juridica BOOLEAN DEFAULT FALSE,
    status_cobranca ENUM('ativo', 'negociando', 'acordo_fechado', 'quitado', 'inativo') DEFAULT 'ativo',
    
    -- Controle
    data_ultimo_contato DATE NULL,
    tentativas_contato INT DEFAULT 0,
    observacoes TEXT NULL,
    
    -- Auditoria
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    operador VARCHAR(100) NULL,
    
    FOREIGN KEY (representante_id) REFERENCES representantes(id) ON DELETE SET NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    
    INDEX idx_cpf (cpf),
    INDEX idx_nome (nome),
    INDEX idx_representante (representante_id),
    INDEX idx_status (status_cobranca),
    INDEX idx_negativado (negativado_spc),
    INDEX idx_juridica (cobranca_juridica)
);

CREATE TABLE acordos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    inadimplente_id INT NOT NULL,
    tipo_acordo ENUM('vista', 'parcelado') NOT NULL,
    valor_original DECIMAL(15,2) NOT NULL,
    valor_acordo DECIMAL(15,2) NOT NULL,
    desconto_aplicado DECIMAL(15,2) DEFAULT 0.00,
    desconto_percentual DECIMAL(5,2) DEFAULT 0.00,
    
    -- Parcelamento
    numero_parcelas INT DEFAULT 1,
    valor_entrada DECIMAL(15,2) DEFAULT 0.00,
    valor_parcela DECIMAL(15,2) DEFAULT 0.00,
    data_primeira_parcela DATE NULL,
    
    -- Controle
    status ENUM('proposto', 'aceito', 'rejeitado', 'quitado', 'cancelado') DEFAULT 'proposto',
    data_acordo DATE NOT NULL,
    data_vencimento DATE NULL,
    
    -- PIX e pagamento
    chave_pix VARCHAR(100) NULL,
    instrucoes_pagamento TEXT NULL,
    
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    
    FOREIGN KEY (inadimplente_id) REFERENCES inadimplentes(id) ON DELETE CASCADE,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    
    INDEX idx_inadimplente (inadimplente_id),
    INDEX idx_status (status),
    INDEX idx_data_acordo (data_acordo)
);

CREATE TABLE parcelas_acordo (
    id INT AUTO_INCREMENT PRIMARY KEY,
    acordo_id INT NOT NULL,
    numero_parcela INT NOT NULL,
    data_vencimento DATE NOT NULL,
    valor DECIMAL(15,2) NOT NULL,
    valor_pago DECIMAL(15,2) DEFAULT 0.00,
    data_pagamento DATE NULL,
    comprovante_path VARCHAR(500) NULL,
    status ENUM('pendente', 'paga', 'atrasada', 'cancelada') DEFAULT 'pendente',
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (acordo_id) REFERENCES acordos(id) ON DELETE CASCADE,
    UNIQUE KEY unique_parcela_acordo (acordo_id, numero_parcela),
    
    INDEX idx_acordo (acordo_id),
    INDEX idx_vencimento (data_vencimento),
    INDEX idx_status (status)
);

-- =====================================================
-- TABELAS DE CONFIGURAÇÃO E CAMPANHAS
-- =====================================================

CREATE TABLE configuracoes_desconto (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    tipo ENUM('faixa_valor', 'tempo_atraso', 'situacao_cliente', 'campanha_especial') NOT NULL,
    
    -- Faixa de valor
    valor_minimo DECIMAL(15,2) NULL,
    valor_maximo DECIMAL(15,2) NULL,
    
    -- Tempo de atraso
    dias_minimo INT NULL,
    dias_maximo INT NULL,
    
    -- Configurações de desconto
    desconto_percentual DECIMAL(5,2) NOT NULL,
    desconto_maximo DECIMAL(15,2) NULL,
    eliminar_juros BOOLEAN DEFAULT FALSE,
    eliminar_multa BOOLEAN DEFAULT FALSE,
    
    -- Controle
    ativo BOOLEAN DEFAULT TRUE,
    data_inicio DATE NULL,
    data_fim DATE NULL,
    
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_tipo (tipo),
    INDEX idx_ativo (ativo),
    INDEX idx_vigencia (data_inicio, data_fim)
);

CREATE TABLE campanhas_cobranca (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    descricao TEXT NULL,
    data_inicio DATE NOT NULL,
    data_fim DATE NOT NULL,
    
    -- Configurações da campanha
    desconto_especial DECIMAL(5,2) DEFAULT 0.00,
    meta_acordos INT DEFAULT 0,
    meta_valor DECIMAL(15,2) DEFAULT 0.00,
    
    -- Filtros de público-alvo
    valor_minimo DECIMAL(15,2) NULL,
    valor_maximo DECIMAL(15,2) NULL,
    dias_atraso_minimo INT NULL,
    representante_id INT NULL,
    apenas_nao_negativados BOOLEAN DEFAULT FALSE,
    
    status ENUM('planejada', 'ativa', 'pausada', 'finalizada') DEFAULT 'planejada',
    
    observacoes TEXT NULL,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    criado_por INT NULL,
    
    FOREIGN KEY (representante_id) REFERENCES representantes(id) ON DELETE SET NULL,
    FOREIGN KEY (criado_por) REFERENCES usuarios(id) ON DELETE SET NULL,
    
    INDEX idx_status (status),
    INDEX idx_vigencia (data_inicio, data_fim)
);

-- =====================================================
-- TABELAS DE INTEGRAÇÃO E COMUNICAÇÃO
-- =====================================================

CREATE TABLE configuracoes_whatsapp (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    token_api VARCHAR(500) NOT NULL,
    numero_whatsapp VARCHAR(20) NOT NULL,
    webhook_url VARCHAR(500) NULL,
    ativo BOOLEAN DEFAULT TRUE,
    limite_diario INT DEFAULT 100,
    horario_inicio TIME DEFAULT '08:00:00',
    horario_fim TIME DEFAULT '18:00:00',
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE templates_mensagem (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    tipo ENUM('cobranca_inicial', 'cobranca_seguimento', 'acordo_proposta', 'acordo_aceito', 'lembrete_vencimento') NOT NULL,
    template TEXT NOT NULL,
    variaveis JSON NULL,
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE historico_mensagens (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    inadimplente_id INT NULL,
    numero_destino VARCHAR(20) NOT NULL,
    tipo_mensagem VARCHAR(50) NOT NULL,
    template_id INT NULL,
    conteudo TEXT NOT NULL,
    status ENUM('enviada', 'entregue', 'lida', 'respondida', 'erro') DEFAULT 'enviada',
    resposta TEXT NULL,
    data_envio TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    data_entrega TIMESTAMP NULL,
    data_leitura TIMESTAMP NULL,
    data_resposta TIMESTAMP NULL,
    
    FOREIGN KEY (inadimplente_id) REFERENCES inadimplentes(id) ON DELETE SET NULL,
    FOREIGN KEY (template_id) REFERENCES templates_mensagem(id) ON DELETE SET NULL,
    
    INDEX idx_inadimplente (inadimplente_id),
    INDEX idx_numero (numero_destino),
    INDEX idx_data_envio (data_envio),
    INDEX idx_status (status)
);

-- =====================================================
-- TABELAS DE BACKUP E CONFIGURAÇÕES GERAIS
-- =====================================================

CREATE TABLE configuracoes_sistema (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chave VARCHAR(100) UNIQUE NOT NULL,
    valor TEXT NULL,
    tipo ENUM('string', 'number', 'boolean', 'json') DEFAULT 'string',
    descricao TEXT NULL,
    categoria VARCHAR(50) DEFAULT 'geral',
    editavel BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_chave (chave),
    INDEX idx_categoria (categoria)
);

CREATE TABLE backups (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(200) NOT NULL,
    tipo ENUM('automatico', 'manual', 'incremental') NOT NULL,
    tamanho_bytes BIGINT NULL,
    caminho_arquivo VARCHAR(500) NULL,
    url_cloud VARCHAR(500) NULL,
    hash_verificacao VARCHAR(64) NULL,
    status ENUM('iniciado', 'concluido', 'erro', 'restaurado') DEFAULT 'iniciado',
    data_inicio TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    data_conclusao TIMESTAMP NULL,
    observacoes TEXT NULL,
    
    INDEX idx_tipo (tipo),
    INDEX idx_status (status),
    INDEX idx_data (data_inicio)
);

-- =====================================================
-- INSERÇÃO DE DADOS INICIAIS
-- =====================================================

-- Usuário administrador padrão
INSERT INTO usuarios (nome, email, senha_hash, tipo_usuario) VALUES 
('Administrador', 'admin@villajoias.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'administrador');

-- Configurações iniciais do sistema
INSERT INTO configuracoes_sistema (chave, valor, tipo, descricao, categoria) VALUES
('empresa_nome', 'Villa Joias', 'string', 'Nome da empresa', 'geral'),
('empresa_cnpj', '', 'string', 'CNPJ da empresa', 'geral'),
('chave_pix_principal', '', 'string', 'Chave PIX principal para recebimentos', 'financeiro'),
('backup_automatico', 'true', 'boolean', 'Ativar backup automático diário', 'sistema'),
('limite_tentativas_login', '5', 'number', 'Número máximo de tentativas de login', 'seguranca'),
('sessao_timeout', '3600', 'number', 'Timeout da sessão em segundos', 'seguranca');

-- Categorias padrão
INSERT INTO categorias (nome, descricao, cor) VALUES
('Despesas Operacionais', 'Gastos necessários para operação da empresa', '#3498DB'),
('Recursos Humanos', 'Gastos com funcionários e encargos', '#E74C3C'),
('Marketing e Publicidade', 'Investimentos em marketing e propaganda', '#9B59B6'),
('Fornecedores', 'Pagamentos a fornecedores de produtos', '#F39C12'),
('Impostos e Taxas', 'Tributos e taxas governamentais', '#34495E'),
('Investimentos', 'Aplicações e investimentos da empresa', '#27AE60');

-- Subcategorias padrão
INSERT INTO subcategorias (categoria_id, nome, descricao) VALUES
(1, 'Aluguel', 'Aluguel do estabelecimento'),
(1, 'Energia Elétrica', 'Conta de luz'),
(1, 'Água', 'Conta de água'),
(1, 'Internet/Telefone', 'Telecomunicações'),
(1, 'Material de Escritório', 'Materiais para escritório'),
(2, 'Salários', 'Salários dos funcionários'),
(2, 'INSS', 'Contribuição previdenciária'),
(2, 'FGTS', 'Fundo de Garantia'),
(2, 'Vale Transporte', 'Auxílio transporte'),
(2, 'Vale Refeição', 'Auxílio alimentação'),
(3, 'Publicidade Online', 'Anúncios digitais'),
(3, 'Material Gráfico', 'Impressos e materiais'),
(4, 'Joias e Acessórios', 'Produtos para revenda'),
(4, 'Embalagens', 'Materiais de embalagem'),
(5, 'ICMS', 'Imposto sobre circulação de mercadorias'),
(5, 'ISS', 'Imposto sobre serviços'),
(6, 'CDB', 'Certificado de Depósito Bancário'),
(6, 'Poupança', 'Conta poupança');

-- Templates de mensagem padrão
INSERT INTO templates_mensagem (nome, tipo, template, variaveis) VALUES
('Cobrança Inicial', 'cobranca_inicial', 
'Oi {nome}! Aqui é a Vi, da Villa Joias. Tudo bem? Estou entrando em contato sobre sua pendência de R$ {valor}. Temos uma proposta especial para você quitar com desconto. Posso te ajudar?',
'["nome", "valor"]'),

('Proposta de Acordo', 'acordo_proposta',
'Oi {nome}! Tenho uma proposta especial para você: quitação da sua pendência de R$ {valor_original} por apenas R$ {valor_acordo} à vista via PIX. Desconto de {desconto}%! Aceita?',
'["nome", "valor_original", "valor_acordo", "desconto"]'),

('Lembrete de Vencimento', 'lembrete_vencimento',
'Oi {nome}! Lembrete: sua parcela de R$ {valor} vence {data_vencimento}. Para pagar, use nossa chave PIX: {chave_pix}. Após o pagamento, envie o comprovante aqui. Obrigada!',
'["nome", "valor", "data_vencimento", "chave_pix"]');

SET FOREIGN_KEY_CHECKS = 1;

-- =====================================================
-- VIEWS PARA RELATÓRIOS
-- =====================================================

CREATE VIEW vw_saldos_caixas AS
SELECT 
    c.id,
    c.nome,
    c.tipo,
    c.saldo_atual,
    c.limite_cheque_especial,
    (c.saldo_atual + c.limite_cheque_especial) as saldo_disponivel,
    c.ativo
FROM caixas c
WHERE c.ativo = TRUE;

CREATE VIEW vw_transacoes_mes_atual AS
SELECT 
    t.*,
    c.nome as categoria_nome,
    sc.nome as subcategoria_nome,
    cx.nome as caixa_nome,
    r.nome as representante_nome
FROM transacoes t
LEFT JOIN categorias c ON t.categoria_id = c.id
LEFT JOIN subcategorias sc ON t.subcategoria_id = sc.id
LEFT JOIN caixas cx ON t.caixa_origem_id = cx.id
LEFT JOIN representantes r ON t.representante_id = r.id
WHERE YEAR(t.data_transacao) = YEAR(CURDATE()) 
AND MONTH(t.data_transacao) = MONTH(CURDATE())
AND t.status = 'confirmado';

CREATE VIEW vw_inadimplencia_resumo AS
SELECT 
    COUNT(*) as total_inadimplentes,
    SUM(saldo_atual) as valor_total,
    SUM(CASE WHEN negativado_spc = TRUE THEN 1 ELSE 0 END) as total_negativados,
    SUM(CASE WHEN cobranca_juridica = TRUE THEN 1 ELSE 0 END) as total_juridica,
    AVG(saldo_atual) as ticket_medio
FROM inadimplentes 
WHERE status_cobranca = 'ativo';

-- =====================================================
-- PROCEDURES PARA OPERAÇÕES COMUNS
-- =====================================================

DELIMITER //

CREATE PROCEDURE sp_atualizar_saldo_caixa(
    IN p_caixa_id INT,
    IN p_valor DECIMAL(15,2),
    IN p_operacao ENUM('credito', 'debito')
)
BEGIN
    DECLARE EXIT HANDLER FOR SQLEXCEPTION
    BEGIN
        ROLLBACK;
        RESIGNAL;
    END;
    
    START TRANSACTION;
    
    IF p_operacao = 'credito' THEN
        UPDATE caixas SET saldo_atual = saldo_atual + p_valor WHERE id = p_caixa_id;
    ELSE
        UPDATE caixas SET saldo_atual = saldo_atual - p_valor WHERE id = p_caixa_id;
    END IF;
    
    COMMIT;
END //

CREATE PROCEDURE sp_calcular_juros_inadimplente(
    IN p_inadimplente_id INT
)
BEGIN
    DECLARE v_saldo_original DECIMAL(15,2);
    DECLARE v_taxa_juros DECIMAL(8,4);
    DECLARE v_taxa_multa DECIMAL(8,4);
    DECLARE v_dias_atraso INT;
    DECLARE v_juros DECIMAL(15,2);
    DECLARE v_multa DECIMAL(15,2);
    DECLARE v_novo_saldo DECIMAL(15,2);
    
    SELECT saldo_original, taxa_juros, taxa_multa,
           DATEDIFF(CURDATE(), criado_em)
    INTO v_saldo_original, v_taxa_juros, v_taxa_multa, v_dias_atraso
    FROM inadimplentes 
    WHERE id = p_inadimplente_id;
    
    SET v_juros = v_saldo_original * (v_taxa_juros / 100) * (v_dias_atraso / 30);
    SET v_multa = v_saldo_original * (v_taxa_multa / 100);
    SET v_novo_saldo = v_saldo_original + v_juros + v_multa;
    
    UPDATE inadimplentes 
    SET saldo_atual = v_novo_saldo 
    WHERE id = p_inadimplente_id;
END //

DELIMITER ;

-- =====================================================
-- TRIGGERS PARA AUDITORIA E CONTROLE
-- =====================================================

DELIMITER //

CREATE TRIGGER tr_transacoes_after_insert
AFTER INSERT ON transacoes
FOR EACH ROW
BEGIN
    -- Atualizar saldo do caixa origem (saída)
    IF NEW.caixa_origem_id IS NOT NULL AND NEW.tipo IN ('saida', 'transferencia') THEN
        CALL sp_atualizar_saldo_caixa(NEW.caixa_origem_id, NEW.valor, 'debito');
    END IF;
    
    -- Atualizar saldo do caixa destino (entrada)
    IF NEW.caixa_destino_id IS NOT NULL AND NEW.tipo IN ('entrada', 'transferencia', 'rendimento') THEN
        CALL sp_atualizar_saldo_caixa(NEW.caixa_destino_id, NEW.valor, 'credito');
    END IF;
    
    -- Log de auditoria
    INSERT INTO logs_auditoria (usuario_id, acao, tabela_afetada, registro_id, dados_novos)
    VALUES (NEW.criado_por, 'INSERT', 'transacoes', NEW.id, JSON_OBJECT('valor', NEW.valor, 'tipo', NEW.tipo));
END //

CREATE TRIGGER tr_usuarios_after_update
AFTER UPDATE ON usuarios
FOR EACH ROW
BEGIN
    INSERT INTO logs_auditoria (usuario_id, acao, tabela_afetada, registro_id, dados_anteriores, dados_novos)
    VALUES (NEW.id, 'UPDATE', 'usuarios', NEW.id, 
            JSON_OBJECT('ativo', OLD.ativo, 'tipo_usuario', OLD.tipo_usuario),
            JSON_OBJECT('ativo', NEW.ativo, 'tipo_usuario', NEW.tipo_usuario));
END //

DELIMITER ;

-- =====================================================
-- ÍNDICES ADICIONAIS PARA PERFORMANCE
-- =====================================================

CREATE INDEX idx_transacoes_mes_ano ON transacoes (YEAR(data_transacao), MONTH(data_transacao));
CREATE INDEX idx_inadimplentes_saldo ON inadimplentes (saldo_atual);
CREATE INDEX idx_acordos_valor ON acordos (valor_acordo);
CREATE INDEX idx_mensagens_data_tipo ON historico_mensagens (data_envio, tipo_mensagem);

-- =====================================================
-- COMENTÁRIOS FINAIS
-- =====================================================

-- Esta estrutura de banco de dados foi projetada para:
-- 1. Suportar todas as funcionalidades especificadas
-- 2. Garantir integridade referencial
-- 3. Otimizar performance através de índices estratégicos
-- 4. Facilitar auditoria e compliance
-- 5. Permitir escalabilidade futura
-- 6. Manter flexibilidade para evoluções

-- Próximos passos:
-- 1. Implementar as classes PHP para acesso aos dados
-- 2. Criar APIs REST para todas as operações
-- 3. Desenvolver interface web responsiva
-- 4. Integrar com WhatsApp Business API
-- 5. Implementar sistema de backup automatizado

