# Script de Otimização do Banco de Dados Villa Joias

import mysql.connector
import os
from datetime import datetime

class DatabaseOptimizer:
    def __init__(self, host='localhost', user='root', password='', database='villa_joias'):
        self.host = host
        self.user = user
        self.password = password
        self.database = database
        self.connection = None
    
    def connect(self):
        """Conectar ao banco de dados"""
        try:
            self.connection = mysql.connector.connect(
                host=self.host,
                user=self.user,
                password=self.password,
                database=self.database
            )
            print(f"✅ Conectado ao banco de dados {self.database}")
            return True
        except Exception as e:
            print(f"❌ Erro ao conectar: {e}")
            return False
    
    def execute_query(self, query, params=None):
        """Executar query no banco"""
        if not self.connection:
            return False
        
        try:
            cursor = self.connection.cursor()
            cursor.execute(query, params or ())
            self.connection.commit()
            cursor.close()
            return True
        except Exception as e:
            print(f"❌ Erro ao executar query: {e}")
            return False
    
    def create_indexes(self):
        """Criar índices para otimização"""
        print("\n🔧 Criando índices de otimização...")
        
        indexes = [
            # Índices para transações
            "CREATE INDEX IF NOT EXISTS idx_transacoes_data ON transacoes(data_transacao)",
            "CREATE INDEX IF NOT EXISTS idx_transacoes_caixa ON transacoes(caixa_id)",
            "CREATE INDEX IF NOT EXISTS idx_transacoes_tipo ON transacoes(tipo)",
            "CREATE INDEX IF NOT EXISTS idx_transacoes_valor ON transacoes(valor)",
            "CREATE INDEX IF NOT EXISTS idx_transacoes_caixa_data ON transacoes(caixa_id, data_transacao)",
            "CREATE INDEX IF NOT EXISTS idx_transacoes_rep_data ON transacoes(representante_id, data_transacao)",
            
            # Índices para inadimplentes
            "CREATE INDEX IF NOT EXISTS idx_inadimplentes_status ON inadimplentes(status)",
            "CREATE INDEX IF NOT EXISTS idx_inadimplentes_representante ON inadimplentes(representante_id)",
            "CREATE INDEX IF NOT EXISTS idx_inadimplentes_spc ON inadimplentes(spc_negativado)",
            "CREATE INDEX IF NOT EXISTS idx_inadimplentes_rep_status ON inadimplentes(representante_id, status)",
            
            # Índices para acordos
            "CREATE INDEX IF NOT EXISTS idx_acordos_status ON acordos(status)",
            "CREATE INDEX IF NOT EXISTS idx_acordos_data ON acordos(data_acordo)",
            "CREATE INDEX IF NOT EXISTS idx_acordos_tipo ON acordos(tipo_pagamento)",
            
            # Índices para cartões
            "CREATE INDEX IF NOT EXISTS idx_despesas_cartao_data ON despesas_cartao(data_compra)",
            "CREATE INDEX IF NOT EXISTS idx_despesas_cartao_cartao ON despesas_cartao(cartao_id)",
            
            # Índices para usuários
            "CREATE INDEX IF NOT EXISTS idx_usuarios_email ON usuarios(email)",
            "CREATE INDEX IF NOT EXISTS idx_usuarios_status ON usuarios(status)",
        ]
        
        for index in indexes:
            if self.execute_query(index):
                print(f"✅ Índice criado: {index.split('idx_')[1].split(' ')[0]}")
            else:
                print(f"❌ Erro ao criar índice")
    
    def create_views(self):
        """Criar views otimizadas"""
        print("\n📊 Criando views otimizadas...")
        
        views = [
            # View para dashboard
            """
            CREATE OR REPLACE VIEW vw_dashboard_metricas AS
            SELECT 
                (SELECT COALESCE(SUM(saldo_atual), 0) FROM caixas WHERE ativo = 1) as saldo_total,
                (SELECT COALESCE(SUM(valor), 0) FROM transacoes WHERE tipo = 'entrada' AND DATE(data_transacao) = CURDATE()) as entradas_hoje,
                (SELECT COALESCE(SUM(valor), 0) FROM transacoes WHERE tipo = 'saida' AND DATE(data_transacao) = CURDATE()) as saidas_hoje,
                (SELECT COUNT(*) FROM inadimplentes WHERE status = 'ativo') as inadimplentes_ativos,
                (SELECT COALESCE(SUM(saldo_atual), 0) FROM inadimplentes WHERE status = 'ativo') as valor_inadimplencia,
                (SELECT COUNT(*) FROM acordos WHERE status = 'ativo' AND MONTH(data_acordo) = MONTH(CURDATE())) as acordos_mes
            """,
            
            # View para performance de representantes
            """
            CREATE OR REPLACE VIEW vw_performance_representantes AS
            SELECT 
                r.id,
                r.nome,
                r.codigo,
                COALESCE(SUM(CASE WHEN t.tipo = 'entrada' AND MONTH(t.data_transacao) = MONTH(CURDATE()) THEN t.valor ELSE 0 END), 0) as vendas_mes,
                COALESCE(SUM(CASE WHEN t.tipo = 'entrada' AND YEAR(t.data_transacao) = YEAR(CURDATE()) THEN t.valor ELSE 0 END), 0) as vendas_ano,
                COUNT(DISTINCT CASE WHEN t.tipo = 'entrada' AND MONTH(t.data_transacao) = MONTH(CURDATE()) THEN t.id END) as transacoes_mes,
                (SELECT COUNT(*) FROM inadimplentes WHERE representante_id = r.id AND status = 'ativo') as inadimplentes_count
            FROM representantes r
            LEFT JOIN transacoes t ON r.id = t.representante_id
            WHERE r.ativo = 1
            GROUP BY r.id, r.nome, r.codigo
            """,
            
            # View para fluxo de caixa
            """
            CREATE OR REPLACE VIEW vw_fluxo_caixa_mensal AS
            SELECT 
                DATE_FORMAT(data_transacao, '%Y-%m') as mes_ano,
                COALESCE(SUM(CASE WHEN tipo = 'entrada' THEN valor ELSE 0 END), 0) as entradas,
                COALESCE(SUM(CASE WHEN tipo = 'saida' THEN valor ELSE 0 END), 0) as saidas,
                COALESCE(SUM(CASE WHEN tipo = 'entrada' THEN valor ELSE -valor END), 0) as saldo_liquido
            FROM transacoes
            WHERE data_transacao >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
            GROUP BY DATE_FORMAT(data_transacao, '%Y-%m')
            ORDER BY mes_ano
            """
        ]
        
        for view in views:
            if self.execute_query(view):
                view_name = view.split('vw_')[1].split(' ')[0]
                print(f"✅ View criada: {view_name}")
            else:
                print(f"❌ Erro ao criar view")
    
    def optimize_tables(self):
        """Otimizar tabelas existentes"""
        print("\n⚡ Otimizando tabelas...")
        
        tables = [
            'transacoes', 'caixas', 'representantes', 'categorias',
            'inadimplentes', 'acordos', 'cartoes', 'despesas_cartao',
            'usuarios', 'logs_atividade'
        ]
        
        for table in tables:
            if self.execute_query(f"OPTIMIZE TABLE {table}"):
                print(f"✅ Tabela otimizada: {table}")
            else:
                print(f"❌ Erro ao otimizar tabela: {table}")
    
    def analyze_tables(self):
        """Analisar tabelas para estatísticas"""
        print("\n📈 Analisando tabelas...")
        
        tables = [
            'transacoes', 'caixas', 'representantes', 'categorias',
            'inadimplentes', 'acordos', 'cartoes', 'despesas_cartao',
            'usuarios'
        ]
        
        for table in tables:
            if self.execute_query(f"ANALYZE TABLE {table}"):
                print(f"✅ Tabela analisada: {table}")
            else:
                print(f"❌ Erro ao analisar tabela: {table}")
    
    def create_procedures(self):
        """Criar procedures otimizadas"""
        print("\n🔧 Criando procedures...")
        
        procedures = [
            # Procedure para relatório de cobrança
            """
            DROP PROCEDURE IF EXISTS sp_relatorio_cobranca;
            CREATE PROCEDURE sp_relatorio_cobranca(
                IN p_data_inicio DATE,
                IN p_data_fim DATE
            )
            BEGIN
                SELECT 
                    COUNT(*) as total_inadimplentes,
                    SUM(saldo_atual) as valor_total,
                    COUNT(CASE WHEN spc_negativado = 1 THEN 1 END) as negativados,
                    COUNT(CASE WHEN cobranca_juridica = 1 THEN 1 END) as juridicos,
                    AVG(saldo_atual) as ticket_medio,
                    (SELECT COUNT(*) FROM acordos WHERE data_acordo BETWEEN p_data_inicio AND p_data_fim) as acordos_periodo,
                    (SELECT SUM(valor_acordo) FROM acordos WHERE data_acordo BETWEEN p_data_inicio AND p_data_fim) as valor_recuperado
                FROM inadimplentes 
                WHERE data_registro BETWEEN p_data_inicio AND p_data_fim;
            END
            """,
            
            # Procedure para análise de performance
            """
            DROP PROCEDURE IF EXISTS sp_analise_performance;
            CREATE PROCEDURE sp_analise_performance(
                IN p_representante_id INT,
                IN p_meses INT
            )
            BEGIN
                DECLARE data_inicio DATE DEFAULT DATE_SUB(CURDATE(), INTERVAL p_meses MONTH);
                
                SELECT 
                    DATE_FORMAT(t.data_transacao, '%Y-%m') as mes,
                    COUNT(*) as total_transacoes,
                    SUM(t.valor) as total_vendas,
                    AVG(t.valor) as ticket_medio,
                    COUNT(DISTINCT t.cliente_nome) as clientes_unicos
                FROM transacoes t
                WHERE t.representante_id = p_representante_id 
                    AND t.tipo = 'entrada'
                    AND t.data_transacao >= data_inicio
                GROUP BY DATE_FORMAT(t.data_transacao, '%Y-%m')
                ORDER BY mes;
            END
            """
        ]
        
        for procedure in procedures:
            if self.execute_query(procedure):
                proc_name = procedure.split('sp_')[1].split('(')[0]
                print(f"✅ Procedure criada: {proc_name}")
            else:
                print(f"❌ Erro ao criar procedure")
    
    def run_optimization(self):
        """Executar otimização completa"""
        print("🚀 Iniciando otimização do banco de dados Villa Joias...")
        print(f"📅 Data/Hora: {datetime.now().strftime('%d/%m/%Y %H:%M:%S')}")
        
        if not self.connect():
            return False
        
        try:
            self.create_indexes()
            self.create_views()
            self.create_procedures()
            self.analyze_tables()
            self.optimize_tables()
            
            print("\n✅ Otimização concluída com sucesso!")
            print("📊 Banco de dados otimizado para máxima performance")
            
        except Exception as e:
            print(f"\n❌ Erro durante otimização: {e}")
            return False
        
        finally:
            if self.connection:
                self.connection.close()
                print("🔌 Conexão fechada")
        
        return True

def main():
    """Função principal"""
    print("=" * 60)
    print("🏆 SISTEMA VILLA JOIAS - OTIMIZAÇÃO DE BANCO DE DADOS")
    print("=" * 60)
    
    # Configurações do banco (ajustar conforme necessário)
    optimizer = DatabaseOptimizer(
        host='localhost',
        user='root',
        password='',  # Definir senha se necessário
        database='villa_joias'
    )
    
    # Executar otimização
    success = optimizer.run_optimization()
    
    if success:
        print("\n🎉 Sistema otimizado e pronto para produção!")
    else:
        print("\n⚠️ Otimização falhou. Verifique as configurações.")

if __name__ == "__main__":
    main()

