#!/bin/bash

# Script de Configuração Automática GoDaddy + VPS
# Versão: 1.0
# Uso: ./setup_godaddy_vps.sh seudominio.com.br seu@email.com

DOMAIN=$1
EMAIL=$2

if [ -z "$DOMAIN" ] || [ -z "$EMAIL" ]; then
    echo "❌ Erro: Você deve especificar domínio e email"
    echo "Uso: ./setup_godaddy_vps.sh seudominio.com.br seu@email.com"
    echo ""
    echo "Exemplo: ./setup_godaddy_vps.sh villajoias.com.br admin@villajoias.com.br"
    exit 1
fi

if [ "$EUID" -ne 0 ]; then
    echo "❌ Este script deve ser executado como root (sudo)"
    exit 1
fi

# Cores
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo -e "${BLUE}🚀 CONFIGURAÇÃO AUTOMÁTICA GODADDY + VPS${NC}"
echo "=========================================="
echo ""
echo "Domínio: $DOMAIN"
echo "Email: $EMAIL"
echo ""

# Obter IP do servidor
SERVER_IP=$(curl -s ifconfig.me)
echo -e "${BLUE}🖥️  IP do Servidor: $SERVER_IP${NC}"
echo ""

# Verificar se DNS já está configurado
echo -e "${BLUE}🔍 Verificando DNS atual...${NC}"
CURRENT_IP=$(dig +short $DOMAIN)

if [ -n "$CURRENT_IP" ]; then
    if [ "$CURRENT_IP" = "$SERVER_IP" ]; then
        echo -e "${GREEN}✅ DNS já está configurado corretamente!${NC}"
        DNS_READY=true
    else
        echo -e "${YELLOW}⚠️  DNS aponta para: $CURRENT_IP (deveria ser: $SERVER_IP)${NC}"
        DNS_READY=false
    fi
else
    echo -e "${RED}❌ DNS não configurado${NC}"
    DNS_READY=false
fi

echo ""

# Se DNS não está pronto, mostrar instruções
if [ "$DNS_READY" = false ]; then
    echo -e "${YELLOW}📋 CONFIGURAÇÃO NECESSÁRIA NO GODADDY${NC}"
    echo "====================================="
    echo ""
    echo "1. Acesse: https://www.godaddy.com/"
    echo "2. Faça login na sua conta"
    echo "3. Vá em 'Meus Produtos' > 'Domínios'"
    echo "4. Clique em 'DNS' ao lado de $DOMAIN"
    echo "5. Configure os registros:"
    echo ""
    echo -e "${BLUE}   Registro A (domínio principal):${NC}"
    echo "   Tipo: A"
    echo "   Nome: @"
    echo "   Valor: $SERVER_IP"
    echo "   TTL: 1 hora"
    echo ""
    echo -e "${BLUE}   Registro A (www):${NC}"
    echo "   Tipo: A"
    echo "   Nome: www"
    echo "   Valor: $SERVER_IP"
    echo "   TTL: 1 hora"
    echo ""
    echo "6. Salve as alterações"
    echo ""
    
    # Aguardar confirmação do usuário
    read -p "Pressione ENTER após configurar o DNS no GoDaddy..."
    echo ""
    
    # Aguardar propagação
    echo -e "${BLUE}⏳ Aguardando propagação DNS...${NC}"
    echo "Isso pode demorar de 15 minutos a 6 horas"
    echo ""
    
    # Loop de verificação
    ATTEMPTS=0
    MAX_ATTEMPTS=20
    
    while [ $ATTEMPTS -lt $MAX_ATTEMPTS ]; do
        CURRENT_IP=$(dig +short $DOMAIN)
        
        if [ "$CURRENT_IP" = "$SERVER_IP" ]; then
            echo -e "${GREEN}✅ DNS propagado com sucesso!${NC}"
            DNS_READY=true
            break
        else
            ATTEMPTS=$((ATTEMPTS + 1))
            echo "Tentativa $ATTEMPTS/$MAX_ATTEMPTS - DNS ainda não propagado..."
            echo "IP atual: $CURRENT_IP (esperado: $SERVER_IP)"
            sleep 30
        fi
    done
    
    if [ "$DNS_READY" = false ]; then
        echo -e "${YELLOW}⚠️  DNS ainda não propagou completamente${NC}"
        echo "Continuando mesmo assim... (pode funcionar localmente)"
    fi
fi

echo ""

# Configurar Virtual Host
echo -e "${BLUE}🌐 Configurando Virtual Host...${NC}"

# Desabilitar site padrão
a2dissite 000-default 2>/dev/null

# Criar configuração do Virtual Host
cat > /etc/apache2/sites-available/villa_joias.conf << EOF
<VirtualHost *:80>
    ServerName $DOMAIN
    ServerAlias www.$DOMAIN
    DocumentRoot /var/www/html/villa_joias
    
    <Directory /var/www/html/villa_joias>
        AllowOverride All
        Require all granted
        Options -Indexes
    </Directory>
    
    # Logs
    ErrorLog \${APACHE_LOG_DIR}/villa_joias_error.log
    CustomLog \${APACHE_LOG_DIR}/villa_joias_access.log combined
    
    # Segurança
    <Files "config.php">
        Require all denied
    </Files>
    
    <Files "database.sql">
        Require all denied
    </Files>
    
    <Files "*.log">
        Require all denied
    </Files>
</VirtualHost>
EOF

# Habilitar site
a2ensite villa_joias

# Criar .htaccess otimizado
cat > /var/www/html/villa_joias/.htaccess << 'EOF'
# Compressão GZIP
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
    AddOutputFilterByType DEFLATE application/json
</IfModule>

# Cache de arquivos estáticos
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
    ExpiresByType image/ico "access plus 1 month"
    ExpiresByType image/icon "access plus 1 month"
    ExpiresByType text/plain "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType font/woff "access plus 1 month"
    ExpiresByType font/woff2 "access plus 1 month"
</IfModule>

# Headers de segurança
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
    Header always set Permissions-Policy "geolocation=(), microphone=(), camera=()"
</IfModule>

# Segurança de arquivos
<Files "config.php">
    Require all denied
</Files>

<Files "database.sql">
    Require all denied
</Files>

<FilesMatch "\.(log|bak|backup|old)$">
    Require all denied
</FilesMatch>

# URL amigáveis
RewriteEngine On

# Redirecionar www para não-www
RewriteCond %{HTTP_HOST} ^www\.(.*)$ [NC]
RewriteRule ^(.*)$ http://%1/$1 [R=301,L]

# Rotas da aplicação
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^api/(.*)$ api/index.php [QSA,L]

RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]
EOF

# Testar configuração
echo "🔍 Testando configuração do Apache..."
if apache2ctl configtest; then
    echo -e "${GREEN}✅ Configuração do Apache OK${NC}"
    
    # Reiniciar Apache
    systemctl restart apache2
    
    if systemctl is-active --quiet apache2; then
        echo -e "${GREEN}✅ Apache reiniciado com sucesso${NC}"
    else
        echo -e "${RED}❌ Erro ao reiniciar Apache${NC}"
        systemctl status apache2
        exit 1
    fi
else
    echo -e "${RED}❌ Erro na configuração do Apache${NC}"
    exit 1
fi

echo ""

# Testar acesso HTTP
echo -e "${BLUE}🌐 Testando acesso HTTP...${NC}"
sleep 5

HTTP_CODE=$(curl -s -o /dev/null -w "%{http_code}" "http://$DOMAIN" 2>/dev/null)
if [ "$HTTP_CODE" = "200" ] || [ "$HTTP_CODE" = "301" ] || [ "$HTTP_CODE" = "302" ]; then
    echo -e "${GREEN}✅ Site acessível via HTTP (código: $HTTP_CODE)${NC}"
    HTTP_WORKING=true
else
    echo -e "${YELLOW}⚠️  Site não acessível via HTTP (código: $HTTP_CODE)${NC}"
    echo "Isso pode ser normal se DNS ainda não propagou"
    HTTP_WORKING=false
fi

echo ""

# Configurar SSL apenas se HTTP estiver funcionando
if [ "$HTTP_WORKING" = true ] || [ "$DNS_READY" = true ]; then
    echo -e "${BLUE}🔒 Configurando SSL...${NC}"
    
    # Verificar se Certbot está instalado
    if ! command -v certbot &> /dev/null; then
        echo "📦 Instalando Certbot..."
        apt install snapd -y
        snap install core
        snap refresh core
        snap install --classic certbot
        ln -sf /snap/bin/certbot /usr/bin/certbot
    fi
    
    # Obter certificado SSL
    echo "🔐 Obtendo certificado SSL..."
    certbot --apache \
        --non-interactive \
        --agree-tos \
        --email "$EMAIL" \
        --domains "$DOMAIN,www.$DOMAIN" \
        --redirect
    
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✅ Certificado SSL configurado com sucesso!${NC}"
        
        # Atualizar .htaccess para forçar HTTPS
        sed -i 's/RewriteRule \^(.*)\$ http:\/\/%1\/\$1 \[R=301,L\]/RewriteRule ^(.*)$ https:\/\/%1\/$1 [R=301,L]/' /var/www/html/villa_joias/.htaccess
        
        # Adicionar redirecionamento HTTPS no .htaccess
        sed -i '/RewriteEngine On/a\\n# Forçar HTTPS\nRewriteCond %{HTTPS} off\nRewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]' /var/www/html/villa_joias/.htaccess
        
        # Reiniciar Apache
        systemctl restart apache2
        
        SSL_CONFIGURED=true
    else
        echo -e "${YELLOW}⚠️  Não foi possível configurar SSL automaticamente${NC}"
        echo "Você pode tentar manualmente depois com:"
        echo "certbot --apache -d $DOMAIN -d www.$DOMAIN"
        SSL_CONFIGURED=false
    fi
else
    echo -e "${YELLOW}⚠️  Pulando configuração SSL (aguarde DNS propagar)${NC}"
    SSL_CONFIGURED=false
fi

echo ""

# Criar script de verificação personalizado
cat > /root/check_$DOMAIN.sh << EOF
#!/bin/bash
./check_dns.sh $DOMAIN
EOF
chmod +x /root/check_$DOMAIN.sh

# Resumo final
echo -e "${BLUE}🎉 CONFIGURAÇÃO CONCLUÍDA!${NC}"
echo "=========================="
echo ""
echo -e "${GREEN}✅ Virtual Host configurado para $DOMAIN${NC}"
echo -e "${GREEN}✅ .htaccess otimizado criado${NC}"
echo -e "${GREEN}✅ Compressão GZIP ativada${NC}"
echo -e "${GREEN}✅ Headers de segurança configurados${NC}"

if [ "$SSL_CONFIGURED" = true ]; then
    echo -e "${GREEN}✅ SSL configurado e ativo${NC}"
    echo -e "${GREEN}✅ Redirecionamento HTTPS ativo${NC}"
fi

echo ""
echo -e "${BLUE}🔗 LINKS PARA TESTAR:${NC}"
echo "- HTTP: http://$DOMAIN"
echo "- HTTPS: https://$DOMAIN"
echo "- WWW: http://www.$DOMAIN"
echo "- Sistema: https://$DOMAIN/install.php"
echo ""

echo -e "${BLUE}🛠️  COMANDOS ÚTEIS:${NC}"
echo "- Verificar DNS: ./check_dns.sh $DOMAIN"
echo "- Verificar rápido: /root/check_$DOMAIN.sh"
echo "- Monitorar sistema: ./monitor.sh"
echo "- Logs Apache: tail -f /var/log/apache2/villa_joias_error.log"
echo ""

if [ "$SSL_CONFIGURED" = false ]; then
    echo -e "${YELLOW}📋 PRÓXIMOS PASSOS:${NC}"
    echo "1. Aguarde DNS propagar completamente (até 6 horas)"
    echo "2. Execute: ./configure_ssl.sh $DOMAIN $EMAIL"
    echo "3. Teste: https://$DOMAIN"
    echo ""
fi

echo -e "${BLUE}📞 SUPORTE:${NC}"
echo "- GoDaddy: 0800 606 4656"
echo "- Hostgator: 0800 600 2000"
echo "- Villa Joias: suporte@villajoias.com.br"
echo ""

echo -e "${GREEN}🚀 Seu domínio $DOMAIN está configurado no VPS!${NC}"

