"""
Aplicação principal do Sistema Villa Joias
"""
import os
import sys
from flask import Flask, send_from_directory
from flask_cors import CORS
from flask_jwt_extended import JWTManager
from src.config import config
from src.models.core import db

def create_app(config_name=None):
    """Factory function para criar a aplicação Flask"""
    
    if config_name is None:
        config_name = os.environ.get('FLASK_ENV', 'development')
    
    app = Flask(__name__, static_folder=os.path.join(os.path.dirname(__file__), 'static'))
    
    # Configurações
    app.config.from_object(config[config_name])
    
    # CORS
    CORS(app, origins=app.config['CORS_ORIGINS'])
    
    # JWT
    jwt = JWTManager(app)
    
    # Banco de dados
    db.init_app(app)
    
    # Registrar blueprints
    from src.routes.user import user_bp
    from src.routes.caixas import caixas_bp
    from src.routes.transacoes import transacoes_bp
    from src.routes.representantes import representantes_bp
    from src.routes.categorias import categorias_bp
    from src.routes.cartoes import cartoes_bp
    from src.routes.dividas import dividas_bp
    from src.routes.relatorios import relatorios_bp
    from src.routes.cobranca import cobranca_bp
    from src.routes.ai import ai_bp
    from src.routes.usuarios import usuarios_bp
    from src.routes.backup import backup_bp
    
    app.register_blueprint(user_bp, url_prefix='/api')
    app.register_blueprint(caixas_bp, url_prefix='/api')
    app.register_blueprint(transacoes_bp, url_prefix='/api')
    app.register_blueprint(representantes_bp, url_prefix='/api')
    app.register_blueprint(categorias_bp, url_prefix='/api')
    app.register_blueprint(cartoes_bp, url_prefix='/api')
    app.register_blueprint(dividas_bp, url_prefix='/api')
    app.register_blueprint(relatorios_bp, url_prefix='/api')
    app.register_blueprint(cobranca_bp, url_prefix='/api')
    app.register_blueprint(ai_bp, url_prefix='/api')
    app.register_blueprint(usuarios_bp, url_prefix='/api')
    app.register_blueprint(backup_bp, url_prefix='/api')
    
    # Criar tabelas se não existirem
    with app.app_context():
        try:
            db.create_all()
            print("✅ Banco de dados inicializado com sucesso")
        except Exception as e:
            print(f"❌ Erro ao inicializar banco de dados: {e}")
    
    # Rota para servir arquivos estáticos
    @app.route('/', defaults={'path': ''})
    @app.route('/<path:path>')
    def serve(path):
        static_folder_path = app.static_folder
        if static_folder_path is None:
            return "Static folder not configured", 404

        if path != "" and os.path.exists(os.path.join(static_folder_path, path)):
            return send_from_directory(static_folder_path, path)
        else:
            index_path = os.path.join(static_folder_path, 'index.html')
            if os.path.exists(index_path):
                return send_from_directory(static_folder_path, 'index.html')
            else:
                return "index.html not found", 404
    
    # Handlers de erro
    @app.errorhandler(404)
    def not_found(error):
        return {'success': False, 'message': 'Endpoint não encontrado'}, 404
    
    @app.errorhandler(500)
    def internal_error(error):
        return {'success': False, 'message': 'Erro interno do servidor'}, 500
    
    @jwt.expired_token_loader
    def expired_token_callback(jwt_header, jwt_payload):
        return {'success': False, 'message': 'Token expirado'}, 401
    
    @jwt.invalid_token_loader
    def invalid_token_callback(error):
        return {'success': False, 'message': 'Token inválido'}, 401
    
    @jwt.unauthorized_loader
    def missing_token_callback(error):
        return {'success': False, 'message': 'Token de acesso necessário'}, 401
    
    return app

if __name__ == '__main__':
    # Adicionar o diretório pai ao path
    sys.path.insert(0, os.path.dirname(os.path.dirname(__file__)))
    
    app = create_app()
    
    print("🚀 Iniciando Sistema Villa Joias...")
    print("📊 Sistema de Controle de Fluxo de Caixa")
    print("🔗 API disponível em: http://localhost:5000/api")
    print("💎 Villa Joias - Gestão Financeira Inteligente")
    
    app.run(host='0.0.0.0', port=5000, debug=True)

