"""
Modelos de dados para inadimplência e cobrança
"""
from src.models.core import db
from datetime import datetime, date
from decimal import Decimal
from sqlalchemy import func

class Inadimplente(db.Model):
    __tablename__ = 'inadimplentes'
    
    id = db.Column(db.Integer, primary_key=True)
    
    # Dados do sistema
    data_registro = db.Column(db.Date, default=date.today, nullable=False)
    operador_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'), nullable=False)
    
    # Situação da dívida
    spc_negativado = db.Column(db.Boolean, default=False)
    data_negativacao = db.Column(db.Date)
    cobranca_juridica = db.Column(db.Boolean, default=False)
    representante_id = db.Column(db.Integer, db.ForeignKey('representantes.id'))
    
    # Dados do devedor
    nome_devedor = db.Column(db.String(200), nullable=False)
    cpf = db.Column(db.String(14), nullable=False)
    numero_contrato = db.Column(db.String(50))
    
    # Valores
    saldo_original = db.Column(db.Numeric(15, 2), nullable=False)
    saldo_atual = db.Column(db.Numeric(15, 2), nullable=False)
    juros_multa_percentual = db.Column(db.Numeric(5, 2), default=0.00)
    data_ultima_atualizacao = db.Column(db.Date, default=date.today)
    
    # Contatos
    telefone_01 = db.Column(db.String(20), nullable=False)
    telefone_02 = db.Column(db.String(20))
    
    # Endereço
    cep = db.Column(db.String(10))
    rua = db.Column(db.String(200))
    numero_casa = db.Column(db.String(20))
    bairro = db.Column(db.String(100))
    cidade = db.Column(db.String(100))
    estado = db.Column(db.String(2))
    
    # Status e controle
    status = db.Column(db.Enum('ativo', 'negociando', 'acordo_fechado', 'quitado', 'cancelado'), default='ativo')
    prioridade = db.Column(db.Enum('baixa', 'media', 'alta', 'urgente'), default='media')
    observacoes = db.Column(db.Text)
    
    # Timestamps
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relacionamentos
    operador = db.relationship('Usuario', backref='inadimplentes_cadastrados')
    representante = db.relationship('Representante', backref='inadimplentes')
    
    def atualizar_saldo(self):
        """Atualiza o saldo atual com juros e multa"""
        if self.juros_multa_percentual > 0:
            dias_atraso = (date.today() - self.data_ultima_atualizacao).days
            if dias_atraso > 0:
                juros_diario = self.juros_multa_percentual / 30  # Juros mensal convertido para diário
                multiplicador = 1 + (juros_diario / 100) * dias_atraso
                self.saldo_atual = self.saldo_atual * Decimal(str(multiplicador))
                self.data_ultima_atualizacao = date.today()
    
    def dias_em_atraso(self):
        """Calcula dias em atraso desde o registro"""
        return (date.today() - self.data_registro).days
    
    def to_dict(self):
        return {
            'id': self.id,
            'data_registro': self.data_registro.isoformat(),
            'operador_id': self.operador_id,
            'operador_nome': self.operador.nome if self.operador else None,
            'spc_negativado': self.spc_negativado,
            'data_negativacao': self.data_negativacao.isoformat() if self.data_negativacao else None,
            'cobranca_juridica': self.cobranca_juridica,
            'representante_id': self.representante_id,
            'representante_nome': self.representante.nome if self.representante else None,
            'nome_devedor': self.nome_devedor,
            'cpf': self.cpf,
            'numero_contrato': self.numero_contrato,
            'saldo_original': float(self.saldo_original),
            'saldo_atual': float(self.saldo_atual),
            'juros_multa_percentual': float(self.juros_multa_percentual),
            'telefone_01': self.telefone_01,
            'telefone_02': self.telefone_02,
            'endereco_completo': f"{self.rua}, {self.numero_casa} - {self.bairro}, {self.cidade}/{self.estado} - {self.cep}",
            'cep': self.cep,
            'rua': self.rua,
            'numero_casa': self.numero_casa,
            'bairro': self.bairro,
            'cidade': self.cidade,
            'estado': self.estado,
            'status': self.status,
            'prioridade': self.prioridade,
            'dias_em_atraso': self.dias_em_atraso(),
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class CampanhaCobranca(db.Model):
    __tablename__ = 'campanhas_cobranca'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    descricao = db.Column(db.Text)
    data_inicio = db.Column(db.Date, nullable=False)
    data_fim = db.Column(db.Date, nullable=False)
    
    # Configurações de desconto
    desconto_especial = db.Column(db.Numeric(5, 2), default=0.00)  # % adicional
    eliminar_juros = db.Column(db.Boolean, default=False)
    
    # Filtros de público-alvo
    valor_minimo = db.Column(db.Numeric(15, 2))
    valor_maximo = db.Column(db.Numeric(15, 2))
    dias_atraso_minimo = db.Column(db.Integer)
    apenas_nao_negativados = db.Column(db.Boolean, default=False)
    representantes_especificos = db.Column(db.Text)  # JSON com IDs
    
    # Metas
    meta_acordos = db.Column(db.Integer)
    meta_valor_recuperado = db.Column(db.Numeric(15, 2))
    
    # Status
    ativa = db.Column(db.Boolean, default=True)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'descricao': self.descricao,
            'data_inicio': self.data_inicio.isoformat(),
            'data_fim': self.data_fim.isoformat(),
            'desconto_especial': float(self.desconto_especial),
            'eliminar_juros': self.eliminar_juros,
            'valor_minimo': float(self.valor_minimo) if self.valor_minimo else None,
            'valor_maximo': float(self.valor_maximo) if self.valor_maximo else None,
            'dias_atraso_minimo': self.dias_atraso_minimo,
            'apenas_nao_negativados': self.apenas_nao_negativados,
            'meta_acordos': self.meta_acordos,
            'meta_valor_recuperado': float(self.meta_valor_recuperado) if self.meta_valor_recuperado else None,
            'ativa': self.ativa,
            'criado_em': self.criado_em.isoformat()
        }

class AcordoCobranca(db.Model):
    __tablename__ = 'acordos_cobranca'
    
    id = db.Column(db.Integer, primary_key=True)
    inadimplente_id = db.Column(db.Integer, db.ForeignKey('inadimplentes.id'), nullable=False)
    campanha_id = db.Column(db.Integer, db.ForeignKey('campanhas_cobranca.id'))
    
    # Dados do acordo
    data_acordo = db.Column(db.Date, default=date.today, nullable=False)
    valor_original = db.Column(db.Numeric(15, 2), nullable=False)
    valor_acordo = db.Column(db.Numeric(15, 2), nullable=False)
    desconto_percentual = db.Column(db.Numeric(5, 2), nullable=False)
    desconto_valor = db.Column(db.Numeric(15, 2), nullable=False)
    
    # Tipo de acordo
    tipo_acordo = db.Column(db.Enum('a_vista', 'parcelado'), nullable=False)
    numero_parcelas = db.Column(db.Integer, default=1)
    valor_entrada = db.Column(db.Numeric(15, 2), default=0.00)
    
    # Pagamento
    chave_pix = db.Column(db.String(200))
    dados_bancarios = db.Column(db.Text)
    
    # Status e controle
    status = db.Column(db.Enum('ativo', 'cumprido', 'quebrado', 'cancelado'), default='ativo')
    data_vencimento = db.Column(db.Date)
    observacoes = db.Column(db.Text)
    
    # Timestamps
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    # Relacionamentos
    inadimplente = db.relationship('Inadimplente', backref='acordos')
    campanha = db.relationship('CampanhaCobranca', backref='acordos')
    
    def percentual_desconto_real(self):
        """Calcula o percentual de desconto real"""
        if float(self.valor_original) > 0:
            return (float(self.desconto_valor) / float(self.valor_original)) * 100
        return 0
    
    def economia_gerada(self):
        """Valor economizado pelo cliente"""
        return float(self.valor_original) - float(self.valor_acordo)
    
    def to_dict(self):
        return {
            'id': self.id,
            'inadimplente_id': self.inadimplente_id,
            'inadimplente_nome': self.inadimplente.nome_devedor if self.inadimplente else None,
            'campanha_id': self.campanha_id,
            'campanha_nome': self.campanha.nome if self.campanha else None,
            'data_acordo': self.data_acordo.isoformat(),
            'valor_original': float(self.valor_original),
            'valor_acordo': float(self.valor_acordo),
            'desconto_percentual': float(self.desconto_percentual),
            'desconto_valor': float(self.desconto_valor),
            'economia_gerada': self.economia_gerada(),
            'tipo_acordo': self.tipo_acordo,
            'numero_parcelas': self.numero_parcelas,
            'valor_entrada': float(self.valor_entrada),
            'chave_pix': self.chave_pix,
            'status': self.status,
            'data_vencimento': self.data_vencimento.isoformat() if self.data_vencimento else None,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class ParcelaAcordo(db.Model):
    __tablename__ = 'parcelas_acordo'
    
    id = db.Column(db.Integer, primary_key=True)
    acordo_id = db.Column(db.Integer, db.ForeignKey('acordos_cobranca.id'), nullable=False)
    numero_parcela = db.Column(db.Integer, nullable=False)
    data_vencimento = db.Column(db.Date, nullable=False)
    valor_parcela = db.Column(db.Numeric(15, 2), nullable=False)
    
    # Pagamento
    data_pagamento = db.Column(db.Date)
    valor_pago = db.Column(db.Numeric(15, 2), default=0.00)
    comprovante_path = db.Column(db.String(500))
    validado = db.Column(db.Boolean, default=False)
    validado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    data_validacao = db.Column(db.DateTime)
    
    # Status
    status = db.Column(db.Enum('pendente', 'pago', 'atrasado', 'validado'), default='pendente')
    observacoes = db.Column(db.Text)
    
    # Timestamps
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relacionamentos
    acordo = db.relationship('AcordoCobranca', backref='parcelas')
    validador = db.relationship('Usuario', backref='parcelas_validadas')
    
    def dias_atraso(self):
        """Calcula dias de atraso"""
        if self.status == 'pendente' and self.data_vencimento < date.today():
            return (date.today() - self.data_vencimento).days
        return 0
    
    def esta_atrasada(self):
        """Verifica se a parcela está atrasada"""
        return self.status == 'pendente' and self.data_vencimento < date.today()
    
    def to_dict(self):
        return {
            'id': self.id,
            'acordo_id': self.acordo_id,
            'numero_parcela': self.numero_parcela,
            'data_vencimento': self.data_vencimento.isoformat(),
            'valor_parcela': float(self.valor_parcela),
            'data_pagamento': self.data_pagamento.isoformat() if self.data_pagamento else None,
            'valor_pago': float(self.valor_pago),
            'comprovante_path': self.comprovante_path,
            'validado': self.validado,
            'validado_por': self.validado_por,
            'data_validacao': self.data_validacao.isoformat() if self.data_validacao else None,
            'status': self.status,
            'dias_atraso': self.dias_atraso(),
            'esta_atrasada': self.esta_atrasada(),
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class HistoricoCobranca(db.Model):
    __tablename__ = 'historico_cobranca'
    
    id = db.Column(db.Integer, primary_key=True)
    inadimplente_id = db.Column(db.Integer, db.ForeignKey('inadimplentes.id'), nullable=False)
    
    # Dados da tentativa
    data_tentativa = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    tipo_contato = db.Column(db.Enum('whatsapp', 'telefone', 'email', 'sms'), nullable=False)
    telefone_usado = db.Column(db.String(20))
    
    # Conteúdo
    mensagem_enviada = db.Column(db.Text)
    resposta_cliente = db.Column(db.Text)
    
    # Resultado
    status_tentativa = db.Column(db.Enum('enviado', 'entregue', 'lido', 'respondido', 'erro'), default='enviado')
    resultado = db.Column(db.Enum('sem_resposta', 'interessado', 'nao_interessado', 'acordo_fechado', 'numero_invalido'), default='sem_resposta')
    
    # Controle
    automatico = db.Column(db.Boolean, default=True)
    operador_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    observacoes = db.Column(db.Text)
    
    # Relacionamentos
    inadimplente = db.relationship('Inadimplente', backref='historico_cobranca')
    operador = db.relationship('Usuario', backref='tentativas_cobranca')
    
    def to_dict(self):
        return {
            'id': self.id,
            'inadimplente_id': self.inadimplente_id,
            'data_tentativa': self.data_tentativa.isoformat(),
            'tipo_contato': self.tipo_contato,
            'telefone_usado': self.telefone_usado,
            'mensagem_enviada': self.mensagem_enviada,
            'resposta_cliente': self.resposta_cliente,
            'status_tentativa': self.status_tentativa,
            'resultado': self.resultado,
            'automatico': self.automatico,
            'operador_id': self.operador_id,
            'observacoes': self.observacoes
        }

class ConfiguracaoDesconto(db.Model):
    __tablename__ = 'configuracao_descontos'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    
    # Faixas de valor (6 faixas)
    faixa1_min = db.Column(db.Numeric(15, 2), default=0.00)
    faixa1_max = db.Column(db.Numeric(15, 2), default=200.00)
    faixa1_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    faixa2_min = db.Column(db.Numeric(15, 2), default=200.01)
    faixa2_max = db.Column(db.Numeric(15, 2), default=500.00)
    faixa2_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    faixa3_min = db.Column(db.Numeric(15, 2), default=500.01)
    faixa3_max = db.Column(db.Numeric(15, 2), default=1000.00)
    faixa3_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    faixa4_min = db.Column(db.Numeric(15, 2), default=1000.01)
    faixa4_max = db.Column(db.Numeric(15, 2), default=2500.00)
    faixa4_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    faixa5_min = db.Column(db.Numeric(15, 2), default=2500.01)
    faixa5_max = db.Column(db.Numeric(15, 2), default=5000.00)
    faixa5_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    faixa6_min = db.Column(db.Numeric(15, 2), default=5000.01)
    faixa6_max = db.Column(db.Numeric(15, 2), default=999999.99)
    faixa6_desconto = db.Column(db.Numeric(5, 2), default=0.00)
    
    # Configurações gerais
    eliminar_juros_disponivel = db.Column(db.Boolean, default=True)
    desconto_maximo_automatico = db.Column(db.Numeric(5, 2), default=50.00)
    priorizar_pagamento_vista = db.Column(db.Boolean, default=True)
    tentativas_vista_antes_parcelamento = db.Column(db.Integer, default=3)
    
    # PIX
    chave_pix = db.Column(db.String(200))
    dados_bancarios = db.Column(db.Text)
    
    # Status
    ativa = db.Column(db.Boolean, default=True)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    def obter_desconto_por_valor(self, valor):
        """Retorna o desconto aplicável para um valor específico"""
        valor = float(valor)
        
        if self.faixa1_min <= valor <= self.faixa1_max:
            return float(self.faixa1_desconto)
        elif self.faixa2_min <= valor <= self.faixa2_max:
            return float(self.faixa2_desconto)
        elif self.faixa3_min <= valor <= self.faixa3_max:
            return float(self.faixa3_desconto)
        elif self.faixa4_min <= valor <= self.faixa4_max:
            return float(self.faixa4_desconto)
        elif self.faixa5_min <= valor <= self.faixa5_max:
            return float(self.faixa5_desconto)
        elif valor >= self.faixa6_min:
            return float(self.faixa6_desconto)
        
        return 0.00
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'faixas': [
                {
                    'numero': 1,
                    'min': float(self.faixa1_min),
                    'max': float(self.faixa1_max),
                    'desconto': float(self.faixa1_desconto)
                },
                {
                    'numero': 2,
                    'min': float(self.faixa2_min),
                    'max': float(self.faixa2_max),
                    'desconto': float(self.faixa2_desconto)
                },
                {
                    'numero': 3,
                    'min': float(self.faixa3_min),
                    'max': float(self.faixa3_max),
                    'desconto': float(self.faixa3_desconto)
                },
                {
                    'numero': 4,
                    'min': float(self.faixa4_min),
                    'max': float(self.faixa4_max),
                    'desconto': float(self.faixa4_desconto)
                },
                {
                    'numero': 5,
                    'min': float(self.faixa5_min),
                    'max': float(self.faixa5_max),
                    'desconto': float(self.faixa5_desconto)
                },
                {
                    'numero': 6,
                    'min': float(self.faixa6_min),
                    'max': float(self.faixa6_max),
                    'desconto': float(self.faixa6_desconto)
                }
            ],
            'eliminar_juros_disponivel': self.eliminar_juros_disponivel,
            'desconto_maximo_automatico': float(self.desconto_maximo_automatico),
            'priorizar_pagamento_vista': self.priorizar_pagamento_vista,
            'tentativas_vista_antes_parcelamento': self.tentativas_vista_antes_parcelamento,
            'chave_pix': self.chave_pix,
            'dados_bancarios': self.dados_bancarios,
            'ativa': self.ativa,
            'criado_em': self.criado_em.isoformat()
        }

