"""
Modelos de dados para sistema de usuários e permissões
"""
from src.models.core import db
from datetime import datetime
from werkzeug.security import generate_password_hash, check_password_hash
import json

class Usuario(db.Model):
    __tablename__ = 'usuarios'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    email = db.Column(db.String(120), unique=True, nullable=False)
    senha_hash = db.Column(db.String(255), nullable=False)
    tipo_usuario = db.Column(db.Enum('administrador', 'gerente', 'operador', 'consulta', 'personalizado'), 
                            default='operador', nullable=False)
    ativo = db.Column(db.Boolean, default=True, nullable=False)
    ultimo_login = db.Column(db.DateTime)
    data_criacao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    observacoes = db.Column(db.Text)
    
    # Relacionamentos
    permissoes = db.relationship('PermissaoUsuario', backref='usuario', lazy=True, cascade='all, delete-orphan')
    logs_atividade = db.relationship('LogAtividade', backref='usuario', lazy=True)
    sessoes = db.relationship('SessaoUsuario', backref='usuario', lazy=True, cascade='all, delete-orphan')
    
    def set_senha(self, senha):
        """Define a senha do usuário com hash"""
        self.senha_hash = generate_password_hash(senha)
    
    def verificar_senha(self, senha):
        """Verifica se a senha está correta"""
        return check_password_hash(self.senha_hash, senha)
    
    def tem_permissao(self, modulo, acao):
        """Verifica se o usuário tem permissão para uma ação específica"""
        if self.tipo_usuario == 'administrador':
            return True
        
        permissao = PermissaoUsuario.query.filter_by(
            usuario_id=self.id,
            modulo=modulo
        ).first()
        
        if not permissao:
            return False
        
        return getattr(permissao, acao, False)
    
    def get_caixas_permitidos(self):
        """Retorna lista de IDs dos caixas que o usuário pode acessar"""
        if self.tipo_usuario == 'administrador':
            return None  # Acesso a todos
        
        restricoes = RestricaoUsuario.query.filter_by(
            usuario_id=self.id,
            tipo_restricao='caixa'
        ).all()
        
        if not restricoes:
            return None  # Acesso a todos
        
        return [r.valor_restricao for r in restricoes]
    
    def get_representantes_permitidos(self):
        """Retorna lista de IDs dos representantes que o usuário pode acessar"""
        if self.tipo_usuario == 'administrador':
            return None  # Acesso a todos
        
        restricoes = RestricaoUsuario.query.filter_by(
            usuario_id=self.id,
            tipo_restricao='representante'
        ).all()
        
        if not restricoes:
            return None  # Acesso a todos
        
        return [r.valor_restricao for r in restricoes]
    
    def to_dict(self):
        """Converte o usuário para dicionário"""
        return {
            'id': self.id,
            'nome': self.nome,
            'email': self.email,
            'tipo_usuario': self.tipo_usuario,
            'ativo': self.ativo,
            'ultimo_login': self.ultimo_login.isoformat() if self.ultimo_login else None,
            'data_criacao': self.data_criacao.isoformat(),
            'observacoes': self.observacoes
        }

class PermissaoUsuario(db.Model):
    __tablename__ = 'permissoes_usuario'
    
    id = db.Column(db.Integer, primary_key=True)
    usuario_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'), nullable=False)
    modulo = db.Column(db.Enum('caixas', 'transacoes', 'representantes', 'categorias', 'cartoes', 
                              'dividas', 'relatorios', 'cobranca', 'usuarios', 'configuracoes'), 
                      nullable=False)
    
    # Permissões específicas por módulo
    visualizar = db.Column(db.Boolean, default=False, nullable=False)
    criar = db.Column(db.Boolean, default=False, nullable=False)
    editar = db.Column(db.Boolean, default=False, nullable=False)
    excluir = db.Column(db.Boolean, default=False, nullable=False)
    exportar = db.Column(db.Boolean, default=False, nullable=False)
    
    # Permissões especiais
    aprovar_desconto = db.Column(db.Boolean, default=False, nullable=False)  # Para cobrança
    ver_valores_altos = db.Column(db.Boolean, default=False, nullable=False)  # Transações acima de X
    gerenciar_backup = db.Column(db.Boolean, default=False, nullable=False)  # Backup/restore
    
    data_criacao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    
    def to_dict(self):
        """Converte a permissão para dicionário"""
        return {
            'id': self.id,
            'modulo': self.modulo,
            'visualizar': self.visualizar,
            'criar': self.criar,
            'editar': self.editar,
            'excluir': self.excluir,
            'exportar': self.exportar,
            'aprovar_desconto': self.aprovar_desconto,
            'ver_valores_altos': self.ver_valores_altos,
            'gerenciar_backup': self.gerenciar_backup
        }

class RestricaoUsuario(db.Model):
    __tablename__ = 'restricoes_usuario'
    
    id = db.Column(db.Integer, primary_key=True)
    usuario_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'), nullable=False)
    tipo_restricao = db.Column(db.Enum('caixa', 'representante', 'periodo', 'valor'), nullable=False)
    valor_restricao = db.Column(db.String(100), nullable=False)  # ID do caixa, representante, etc.
    descricao = db.Column(db.String(255))
    data_criacao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    
    # Relacionamento
    usuario = db.relationship('Usuario', backref='restricoes')
    
    def to_dict(self):
        """Converte a restrição para dicionário"""
        return {
            'id': self.id,
            'tipo_restricao': self.tipo_restricao,
            'valor_restricao': self.valor_restricao,
            'descricao': self.descricao
        }

class SessaoUsuario(db.Model):
    __tablename__ = 'sessoes_usuario'
    
    id = db.Column(db.Integer, primary_key=True)
    usuario_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'), nullable=False)
    token = db.Column(db.String(255), unique=True, nullable=False)
    ip_address = db.Column(db.String(45))
    user_agent = db.Column(db.Text)
    data_criacao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    data_expiracao = db.Column(db.DateTime, nullable=False)
    ativa = db.Column(db.Boolean, default=True, nullable=False)
    
    def is_valida(self):
        """Verifica se a sessão ainda é válida"""
        return self.ativa and datetime.utcnow() < self.data_expiracao
    
    def to_dict(self):
        """Converte a sessão para dicionário"""
        return {
            'id': self.id,
            'ip_address': self.ip_address,
            'data_criacao': self.data_criacao.isoformat(),
            'data_expiracao': self.data_expiracao.isoformat(),
            'ativa': self.ativa
        }

class LogAtividade(db.Model):
    __tablename__ = 'logs_atividade'
    
    id = db.Column(db.Integer, primary_key=True)
    usuario_id = db.Column(db.Integer, db.ForeignKey('usuarios.id'), nullable=False)
    acao = db.Column(db.String(100), nullable=False)  # login, logout, criar_transacao, etc.
    modulo = db.Column(db.String(50), nullable=False)  # caixas, transacoes, etc.
    detalhes = db.Column(db.JSON)  # Dados específicos da ação
    ip_address = db.Column(db.String(45))
    user_agent = db.Column(db.Text)
    data_acao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    
    def to_dict(self):
        """Converte o log para dicionário"""
        return {
            'id': self.id,
            'usuario_id': self.usuario_id,
            'acao': self.acao,
            'modulo': self.modulo,
            'detalhes': self.detalhes,
            'ip_address': self.ip_address,
            'data_acao': self.data_acao.isoformat()
        }

class ConfiguracaoSistema(db.Model):
    __tablename__ = 'configuracoes_sistema'
    
    id = db.Column(db.Integer, primary_key=True)
    chave = db.Column(db.String(100), unique=True, nullable=False)
    valor = db.Column(db.Text, nullable=False)
    tipo = db.Column(db.Enum('string', 'integer', 'float', 'boolean', 'json'), default='string')
    descricao = db.Column(db.String(255))
    categoria = db.Column(db.String(50), default='geral')
    editavel = db.Column(db.Boolean, default=True, nullable=False)
    data_criacao = db.Column(db.DateTime, default=datetime.utcnow, nullable=False)
    data_atualizacao = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    def get_valor(self):
        """Retorna o valor convertido para o tipo correto"""
        if self.tipo == 'integer':
            return int(self.valor)
        elif self.tipo == 'float':
            return float(self.valor)
        elif self.tipo == 'boolean':
            return self.valor.lower() in ('true', '1', 'yes', 'on')
        elif self.tipo == 'json':
            return json.loads(self.valor)
        else:
            return self.valor
    
    def set_valor(self, valor):
        """Define o valor convertendo para string"""
        if self.tipo == 'json':
            self.valor = json.dumps(valor)
        else:
            self.valor = str(valor)
    
    def to_dict(self):
        """Converte a configuração para dicionário"""
        return {
            'id': self.id,
            'chave': self.chave,
            'valor': self.get_valor(),
            'tipo': self.tipo,
            'descricao': self.descricao,
            'categoria': self.categoria,
            'editavel': self.editavel
        }

# Função para criar usuário administrador padrão
def criar_admin_padrao():
    """Cria o usuário administrador padrão se não existir"""
    admin = Usuario.query.filter_by(email='admin@villajoias.com').first()
    
    if not admin:
        admin = Usuario(
            nome='Administrador Villa Joias',
            email='admin@villajoias.com',
            tipo_usuario='administrador',
            ativo=True
        )
        admin.set_senha('123456')  # Senha padrão - deve ser alterada
        
        db.session.add(admin)
        db.session.commit()
        
        print("✅ Usuário administrador criado: admin@villajoias.com / 123456")
    
    return admin

# Função para criar configurações padrão do sistema
def criar_configuracoes_padrao():
    """Cria configurações padrão do sistema"""
    configuracoes_padrao = [
        {
            'chave': 'sistema_nome',
            'valor': 'Villa Joias - Sistema de Controle Financeiro',
            'tipo': 'string',
            'descricao': 'Nome do sistema',
            'categoria': 'geral'
        },
        {
            'chave': 'sistema_versao',
            'valor': '1.0.0',
            'tipo': 'string',
            'descricao': 'Versão do sistema',
            'categoria': 'geral',
            'editavel': False
        },
        {
            'chave': 'backup_automatico',
            'valor': 'true',
            'tipo': 'boolean',
            'descricao': 'Ativar backup automático diário',
            'categoria': 'backup'
        },
        {
            'chave': 'limite_valor_alto',
            'valor': '10000',
            'tipo': 'float',
            'descricao': 'Valor limite para transações consideradas altas',
            'categoria': 'seguranca'
        },
        {
            'chave': 'sessao_duracao_horas',
            'valor': '8',
            'tipo': 'integer',
            'descricao': 'Duração da sessão em horas',
            'categoria': 'seguranca'
        },
        {
            'chave': 'whatsapp_manus_numero',
            'valor': '',
            'tipo': 'string',
            'descricao': 'Número do WhatsApp para Manus',
            'categoria': 'integracao'
        },
        {
            'chave': 'whatsapp_vi_token',
            'valor': '',
            'tipo': 'string',
            'descricao': 'Token do WhatsApp Business para Vi',
            'categoria': 'integracao'
        }
    ]
    
    for config_data in configuracoes_padrao:
        config_existente = ConfiguracaoSistema.query.filter_by(chave=config_data['chave']).first()
        
        if not config_existente:
            config = ConfiguracaoSistema(**config_data)
            db.session.add(config)
    
    db.session.commit()
    print("✅ Configurações padrão do sistema criadas")

