"""
Rotas para integração com IA (Manus/Vi) e WhatsApp
"""
from flask import Blueprint, request, jsonify
from datetime import datetime
from src.services.ai_service import manus_service, vi_service
from src.models.cobranca import Inadimplente, ConfiguracaoDesconto
from src.models.core import db

ai_bp = Blueprint('ai', __name__)

# ==================== MANUS AI (CIO) ====================

@ai_bp.route('/manus/analise-fluxo', methods=['GET'])
def manus_analise_fluxo():
    """Análise inteligente do fluxo de caixa pelo Manus"""
    try:
        periodo = request.args.get('periodo', 30, type=int)
        resultado = manus_service.analisar_fluxo_caixa(periodo)
        return jsonify(resultado)
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/manus/dicas', methods=['GET'])
def manus_dicas():
    """Obtém dicas estratégicas do Manus"""
    try:
        dicas = manus_service.gerar_dicas_estrategicas()
        return jsonify({
            "success": True,
            "dicas": dicas,
            "timestamp": datetime.now().isoformat()
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/manus/pergunta', methods=['POST'])
def manus_pergunta():
    """Processa pergunta do usuário para o Manus"""
    try:
        data = request.get_json()
        pergunta = data.get('pergunta', '')
        
        if not pergunta:
            return jsonify({"error": "Pergunta é obrigatória"}), 400
        
        resposta = manus_service.processar_pergunta_usuario(pergunta)
        
        return jsonify({
            "success": True,
            "pergunta": pergunta,
            "resposta": resposta,
            "timestamp": datetime.now().isoformat()
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/manus/configurar-whatsapp', methods=['POST'])
def manus_configurar_whatsapp():
    """Configura WhatsApp do usuário para comunicação direta"""
    try:
        data = request.get_json()
        numero = data.get('numero_whatsapp', '')
        
        if not numero:
            return jsonify({"error": "Número do WhatsApp é obrigatório"}), 400
        
        manus_service.configurar_whatsapp_usuario(numero)
        
        # Enviar mensagem de boas-vindas
        mensagem_boas_vindas = """🏆 Oi! Sou o Manus, seu CIO da Villa Joias!

Estou aqui para te ajudar com:
📊 Análises financeiras
💡 Dicas estratégicas  
⚠️ Alertas importantes
📈 Relatórios personalizados

Pode me perguntar qualquer coisa sobre seu negócio! 🚀"""
        
        manus_service.enviar_notificacao_whatsapp(mensagem_boas_vindas, "boas_vindas")
        
        return jsonify({
            "success": True,
            "message": "WhatsApp configurado com sucesso",
            "numero": numero
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/manus/notificar', methods=['POST'])
def manus_notificar():
    """Envia notificação via WhatsApp"""
    try:
        data = request.get_json()
        mensagem = data.get('mensagem', '')
        tipo = data.get('tipo', 'info')
        
        if not mensagem:
            return jsonify({"error": "Mensagem é obrigatória"}), 400
        
        sucesso = manus_service.enviar_notificacao_whatsapp(mensagem, tipo)
        
        return jsonify({
            "success": sucesso,
            "message": "Notificação enviada" if sucesso else "Erro ao enviar notificação"
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ==================== VI AI (COBRANÇA) ====================

@ai_bp.route('/vi/configurar-whatsapp', methods=['POST'])
def vi_configurar_whatsapp():
    """Configura WhatsApp Business para Vi"""
    try:
        data = request.get_json()
        token = data.get('token', '')
        numero = data.get('numero', '')
        
        if not token or not numero:
            return jsonify({"error": "Token e número são obrigatórios"}), 400
        
        vi_service.configurar_whatsapp_business(token, numero)
        
        return jsonify({
            "success": True,
            "message": "WhatsApp Business configurado para Vi",
            "numero": numero
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/vi/proposta-acordo/<int:inadimplente_id>', methods=['GET'])
def vi_proposta_acordo(inadimplente_id):
    """Gera proposta de acordo para inadimplente"""
    try:
        inadimplente = Inadimplente.query.get(inadimplente_id)
        if not inadimplente:
            return jsonify({"error": "Inadimplente não encontrado"}), 404
        
        proposta = vi_service.calcular_proposta_acordo(inadimplente)
        
        return jsonify({
            "success": True,
            "proposta": proposta
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/vi/gerar-mensagem', methods=['POST'])
def vi_gerar_mensagem():
    """Gera mensagem de cobrança personalizada"""
    try:
        data = request.get_json()
        inadimplente_id = data.get('inadimplente_id')
        tipo_mensagem = data.get('tipo', 'inicial')
        
        inadimplente = Inadimplente.query.get(inadimplente_id)
        if not inadimplente:
            return jsonify({"error": "Inadimplente não encontrado"}), 404
        
        mensagem = vi_service.gerar_mensagem_cobranca(inadimplente, tipo_mensagem)
        
        return jsonify({
            "success": True,
            "mensagem": mensagem,
            "tipo": tipo_mensagem
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/vi/processar-resposta', methods=['POST'])
def vi_processar_resposta():
    """Processa resposta do cliente"""
    try:
        data = request.get_json()
        inadimplente_id = data.get('inadimplente_id')
        resposta = data.get('resposta', '')
        
        if not inadimplente_id or not resposta:
            return jsonify({"error": "ID do inadimplente e resposta são obrigatórios"}), 400
        
        resultado = vi_service.processar_resposta_cliente(inadimplente_id, resposta)
        
        return jsonify({
            "success": True,
            "resultado": resultado
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/vi/executar-campanha/<int:campanha_id>', methods=['POST'])
def vi_executar_campanha(campanha_id):
    """Executa campanha de cobrança automatizada"""
    try:
        resultado = vi_service.executar_campanha_automatica(campanha_id)
        
        return jsonify({
            "success": True,
            "resultado": resultado
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/vi/enviar-cobranca', methods=['POST'])
def vi_enviar_cobranca():
    """Envia mensagem de cobrança individual"""
    try:
        data = request.get_json()
        inadimplente_id = data.get('inadimplente_id')
        tipo_mensagem = data.get('tipo', 'inicial')
        
        inadimplente = Inadimplente.query.get(inadimplente_id)
        if not inadimplente:
            return jsonify({"error": "Inadimplente não encontrado"}), 404
        
        # Gerar mensagem
        mensagem = vi_service.gerar_mensagem_cobranca(inadimplente, tipo_mensagem)
        
        # Registrar tentativa
        sucesso = vi_service.registrar_tentativa_cobranca(inadimplente_id, mensagem)
        
        # Simular envio (aqui seria a integração real com WhatsApp)
        print(f"📱 Vi enviando para {inadimplente.telefone_01}: {mensagem[:50]}...")
        
        return jsonify({
            "success": sucesso,
            "mensagem": mensagem,
            "telefone": inadimplente.telefone_01
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ==================== CONFIGURAÇÕES GERAIS ====================

@ai_bp.route('/configuracoes/desconto', methods=['GET'])
def obter_configuracao_desconto():
    """Obtém configuração ativa de descontos"""
    try:
        config = ConfiguracaoDesconto.query.filter_by(ativa=True).first()
        
        if not config:
            return jsonify({"error": "Configuração não encontrada"}), 404
        
        return jsonify({
            "success": True,
            "configuracao": {
                "id": config.id,
                "nome": config.nome,
                "faixas": [
                    {
                        "faixa": 1,
                        "valor_min": config.faixa_1_min,
                        "valor_max": config.faixa_1_max,
                        "desconto": config.faixa_1_desconto
                    },
                    {
                        "faixa": 2,
                        "valor_min": config.faixa_2_min,
                        "valor_max": config.faixa_2_max,
                        "desconto": config.faixa_2_desconto
                    },
                    {
                        "faixa": 3,
                        "valor_min": config.faixa_3_min,
                        "valor_max": config.faixa_3_max,
                        "desconto": config.faixa_3_desconto
                    },
                    {
                        "faixa": 4,
                        "valor_min": config.faixa_4_min,
                        "valor_max": config.faixa_4_max,
                        "desconto": config.faixa_4_desconto
                    },
                    {
                        "faixa": 5,
                        "valor_min": config.faixa_5_min,
                        "valor_max": config.faixa_5_max,
                        "desconto": config.faixa_5_desconto
                    },
                    {
                        "faixa": 6,
                        "valor_min": config.faixa_6_min,
                        "valor_max": config.faixa_6_max,
                        "desconto": config.faixa_6_desconto
                    }
                ],
                "desconto_maximo_automatico": config.desconto_maximo_automatico,
                "tentativas_vista_antes_parcelamento": config.tentativas_vista_antes_parcelamento,
                "chave_pix": config.chave_pix
            }
        })
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/configuracoes/desconto', methods=['POST'])
def salvar_configuracao_desconto():
    """Salva configuração de descontos"""
    try:
        data = request.get_json()
        
        # Desativar configuração atual
        ConfiguracaoDesconto.query.update({"ativa": False})
        
        # Criar nova configuração
        config = ConfiguracaoDesconto(
            nome=data.get('nome', 'Configuração Padrão'),
            faixa_1_min=data.get('faixa_1_min', 0),
            faixa_1_max=data.get('faixa_1_max', 500),
            faixa_1_desconto=data.get('faixa_1_desconto', 30),
            faixa_2_min=data.get('faixa_2_min', 501),
            faixa_2_max=data.get('faixa_2_max', 1000),
            faixa_2_desconto=data.get('faixa_2_desconto', 25),
            faixa_3_min=data.get('faixa_3_min', 1001),
            faixa_3_max=data.get('faixa_3_max', 2500),
            faixa_3_desconto=data.get('faixa_3_desconto', 20),
            faixa_4_min=data.get('faixa_4_min', 2501),
            faixa_4_max=data.get('faixa_4_max', 5000),
            faixa_4_desconto=data.get('faixa_4_desconto', 15),
            faixa_5_min=data.get('faixa_5_min', 5001),
            faixa_5_max=data.get('faixa_5_max', 10000),
            faixa_5_desconto=data.get('faixa_5_desconto', 12),
            faixa_6_min=data.get('faixa_6_min', 10001),
            faixa_6_max=data.get('faixa_6_max', 999999),
            faixa_6_desconto=data.get('faixa_6_desconto', 10),
            desconto_maximo_automatico=data.get('desconto_maximo_automatico', 40),
            tentativas_vista_antes_parcelamento=data.get('tentativas_vista_antes_parcelamento', 3),
            chave_pix=data.get('chave_pix', ''),
            ativa=True
        )
        
        db.session.add(config)
        db.session.commit()
        
        # Atualizar configuração do Vi
        vi_service.configuracao_ativa = config
        
        return jsonify({
            "success": True,
            "message": "Configuração salva com sucesso",
            "id": config.id
        })
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ==================== WEBHOOK WHATSAPP ====================

@ai_bp.route('/webhook/whatsapp', methods=['POST'])
def webhook_whatsapp():
    """Webhook para receber mensagens do WhatsApp"""
    try:
        data = request.get_json()
        
        # Processar mensagem recebida
        # Aqui seria implementada a lógica para processar respostas dos clientes
        
        print(f"📱 Webhook WhatsApp: {data}")
        
        return jsonify({"success": True})
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@ai_bp.route('/webhook/whatsapp', methods=['GET'])
def webhook_whatsapp_verify():
    """Verificação do webhook do WhatsApp"""
    try:
        verify_token = request.args.get('hub.verify_token')
        challenge = request.args.get('hub.challenge')
        
        # Verificar token (configurar token de verificação)
        if verify_token == "villa_joias_webhook_2025":
            return challenge
        else:
            return "Token inválido", 403
    except Exception as e:
        return jsonify({"error": str(e)}), 500

