"""
Rotas para gerenciamento de backup e segurança
Villa Joias - Sistema de Controle de Fluxo de Caixa
"""

from flask import Blueprint, request, jsonify, send_file
from flask_jwt_extended import jwt_required, get_jwt_identity
from datetime import datetime, timedelta
import os
import json
from pathlib import Path

from ..services.backup_service import BackupManager, SecurityManager, DEFAULT_CONFIG

backup_bp = Blueprint('backup', __name__, url_prefix='/api/backup')

# Inicializar serviços
backup_manager = BackupManager(DEFAULT_CONFIG)
security_manager = SecurityManager(DEFAULT_CONFIG)

@backup_bp.route('/create', methods=['POST'])
@jwt_required()
def create_backup():
    """Cria backup manual"""
    try:
        user_id = get_jwt_identity()
        data = request.get_json()
        backup_type = data.get('type', 'manual')
        include_database = data.get('include_database', True)
        include_files = data.get('include_files', True)
        
        results = {}
        
        if include_database:
            db_backup = backup_manager.create_database_backup(backup_type)
            results['database'] = db_backup is not None
            results['database_file'] = str(db_backup) if db_backup else None
        
        if include_files:
            files_backup = backup_manager.create_files_backup(backup_type)
            results['files'] = files_backup is not None
            results['files_file'] = str(files_backup) if files_backup else None
        
        # Log da ação
        security_manager.log_user_action(
            user_id, 
            'backup_created', 
            {'type': backup_type, 'results': results}
        )
        
        return jsonify({
            'success': True,
            'message': 'Backup criado com sucesso',
            'results': results,
            'timestamp': datetime.now().isoformat()
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao criar backup: {str(e)}'
        }), 500

@backup_bp.route('/list', methods=['GET'])
@jwt_required()
def list_backups():
    """Lista todos os backups disponíveis"""
    try:
        backup_info = backup_manager.get_backup_info()
        
        # Adicionar informações de integridade
        for backup_type in backup_info:
            for backup in backup_info[backup_type]:
                backup['integrity_ok'] = backup_manager.verify_backup_integrity(
                    backup['path']
                )
        
        return jsonify({
            'success': True,
            'backups': backup_info,
            'total_backups': sum(len(backups) for backups in backup_info.values())
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao listar backups: {str(e)}'
        }), 500

@backup_bp.route('/restore', methods=['POST'])
@jwt_required()
def restore_backup():
    """Restaura backup específico"""
    try:
        user_id = get_jwt_identity()
        data = request.get_json()
        backup_file = data.get('backup_file')
        restore_type = data.get('restore_type', 'database')
        
        if not backup_file or not os.path.exists(backup_file):
            return jsonify({
                'success': False,
                'message': 'Arquivo de backup não encontrado'
            }), 400
        
        # Verificar integridade antes de restaurar
        if not backup_manager.verify_backup_integrity(backup_file):
            return jsonify({
                'success': False,
                'message': 'Backup corrompido ou inválido'
            }), 400
        
        # Executar restauração
        success = backup_manager.restore_backup(backup_file, restore_type)
        
        # Log da ação
        security_manager.log_user_action(
            user_id,
            'backup_restored',
            {
                'backup_file': backup_file,
                'restore_type': restore_type,
                'success': success
            }
        )
        
        if success:
            return jsonify({
                'success': True,
                'message': 'Backup restaurado com sucesso',
                'timestamp': datetime.now().isoformat()
            })
        else:
            return jsonify({
                'success': False,
                'message': 'Erro ao restaurar backup'
            }), 500
            
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao restaurar backup: {str(e)}'
        }), 500

@backup_bp.route('/download/<backup_id>', methods=['GET'])
@jwt_required()
def download_backup(backup_id):
    """Download de backup específico"""
    try:
        user_id = get_jwt_identity()
        
        # Encontrar arquivo de backup
        backup_info = backup_manager.get_backup_info()
        backup_file = None
        
        for backup_type in backup_info:
            for backup in backup_info[backup_type]:
                if backup['name'] == backup_id:
                    backup_file = backup['path']
                    break
        
        if not backup_file or not os.path.exists(backup_file):
            return jsonify({
                'success': False,
                'message': 'Backup não encontrado'
            }), 404
        
        # Log do download
        security_manager.log_user_action(
            user_id,
            'backup_downloaded',
            {'backup_file': backup_id}
        )
        
        return send_file(
            backup_file,
            as_attachment=True,
            download_name=backup_id
        )
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao baixar backup: {str(e)}'
        }), 500

@backup_bp.route('/delete', methods=['DELETE'])
@jwt_required()
def delete_backup():
    """Remove backup específico"""
    try:
        user_id = get_jwt_identity()
        data = request.get_json()
        backup_file = data.get('backup_file')
        
        if not backup_file or not os.path.exists(backup_file):
            return jsonify({
                'success': False,
                'message': 'Backup não encontrado'
            }), 404
        
        # Remover arquivo
        os.remove(backup_file)
        
        # Log da ação
        security_manager.log_user_action(
            user_id,
            'backup_deleted',
            {'backup_file': backup_file}
        )
        
        return jsonify({
            'success': True,
            'message': 'Backup removido com sucesso'
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao remover backup: {str(e)}'
        }), 500

@backup_bp.route('/cleanup', methods=['POST'])
@jwt_required()
def cleanup_old_backups():
    """Remove backups antigos"""
    try:
        user_id = get_jwt_identity()
        
        backup_manager.cleanup_old_backups()
        
        # Log da ação
        security_manager.log_user_action(
            user_id,
            'backup_cleanup',
            {'timestamp': datetime.now().isoformat()}
        )
        
        return jsonify({
            'success': True,
            'message': 'Limpeza de backups concluída'
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro na limpeza: {str(e)}'
        }), 500

@backup_bp.route('/config', methods=['GET', 'POST'])
@jwt_required()
def backup_config():
    """Gerencia configurações de backup"""
    try:
        user_id = get_jwt_identity()
        
        if request.method == 'GET':
            # Retornar configurações atuais (sem senhas)
            config = DEFAULT_CONFIG.copy()
            config.pop('db_password', None)
            
            return jsonify({
                'success': True,
                'config': config
            })
        
        elif request.method == 'POST':
            # Atualizar configurações
            data = request.get_json()
            
            # Validar e atualizar configurações
            valid_keys = [
                'backup_directory', 'max_backups', 'db_host', 
                'db_name', 'db_user'
            ]
            
            updated_config = {}
            for key in valid_keys:
                if key in data:
                    updated_config[key] = data[key]
            
            # Log da ação
            security_manager.log_user_action(
                user_id,
                'backup_config_updated',
                {'updated_keys': list(updated_config.keys())}
            )
            
            return jsonify({
                'success': True,
                'message': 'Configurações atualizadas',
                'updated': updated_config
            })
            
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro nas configurações: {str(e)}'
        }), 500

@backup_bp.route('/status', methods=['GET'])
@jwt_required()
def backup_status():
    """Status do sistema de backup"""
    try:
        backup_info = backup_manager.get_backup_info()
        
        # Calcular estatísticas
        total_backups = sum(len(backups) for backups in backup_info.values())
        total_size = 0
        last_backup = None
        
        for backup_type in backup_info:
            for backup in backup_info[backup_type]:
                total_size += backup['size']
                if not last_backup or backup['created'] > last_backup:
                    last_backup = backup['created']
        
        # Verificar espaço em disco
        backup_dir = Path(DEFAULT_CONFIG['backup_directory'])
        if backup_dir.exists():
            stat = os.statvfs(backup_dir)
            free_space = stat.f_bavail * stat.f_frsize
            total_space = stat.f_blocks * stat.f_frsize
            used_space = total_space - free_space
        else:
            free_space = total_space = used_space = 0
        
        return jsonify({
            'success': True,
            'status': {
                'total_backups': total_backups,
                'total_size_bytes': total_size,
                'total_size_mb': round(total_size / 1024 / 1024, 2),
                'last_backup': last_backup,
                'disk_usage': {
                    'total_space_gb': round(total_space / 1024 / 1024 / 1024, 2),
                    'used_space_gb': round(used_space / 1024 / 1024 / 1024, 2),
                    'free_space_gb': round(free_space / 1024 / 1024 / 1024, 2),
                    'usage_percent': round((used_space / total_space) * 100, 1) if total_space > 0 else 0
                },
                'backup_directory': str(backup_dir),
                'encryption_enabled': True,
                'scheduler_running': True  # Implementar verificação real
            }
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao obter status: {str(e)}'
        }), 500

@backup_bp.route('/logs', methods=['GET'])
@jwt_required()
def get_backup_logs():
    """Retorna logs de backup"""
    try:
        log_file = Path(DEFAULT_CONFIG['backup_directory']) / 'backup.log'
        
        if not log_file.exists():
            return jsonify({
                'success': True,
                'logs': []
            })
        
        # Ler últimas 100 linhas
        with open(log_file, 'r') as f:
            lines = f.readlines()
            recent_lines = lines[-100:] if len(lines) > 100 else lines
        
        return jsonify({
            'success': True,
            'logs': [line.strip() for line in recent_lines],
            'total_lines': len(lines)
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao ler logs: {str(e)}'
        }), 500

@backup_bp.route('/security/logs', methods=['GET'])
@jwt_required()
def get_security_logs():
    """Retorna logs de segurança"""
    try:
        log_file = Path(DEFAULT_CONFIG['log_directory']) / 'security.log'
        
        if not log_file.exists():
            return jsonify({
                'success': True,
                'logs': []
            })
        
        # Parâmetros de filtro
        limit = request.args.get('limit', 50, type=int)
        user_id = request.args.get('user_id')
        action = request.args.get('action')
        
        logs = []
        with open(log_file, 'r') as f:
            for line in f:
                try:
                    # Extrair JSON do log
                    json_start = line.find('{')
                    if json_start > 0:
                        log_data = json.loads(line[json_start:])
                        
                        # Aplicar filtros
                        if user_id and log_data.get('user_id') != user_id:
                            continue
                        if action and log_data.get('action') != action:
                            continue
                        
                        logs.append(log_data)
                except:
                    continue
        
        # Ordenar por timestamp (mais recentes primeiro)
        logs.sort(key=lambda x: x.get('timestamp', ''), reverse=True)
        
        return jsonify({
            'success': True,
            'logs': logs[:limit],
            'total_logs': len(logs)
        })
        
    except Exception as e:
        return jsonify({
            'success': False,
            'message': f'Erro ao ler logs de segurança: {str(e)}'
        }), 500

