"""
Rotas para gerenciamento de categorias e subcategorias
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Categoria, Subcategoria, Transacao
from datetime import datetime, date
from sqlalchemy import func, extract

categorias_bp = Blueprint('categorias', __name__)

@categorias_bp.route('/categorias', methods=['GET'])
@jwt_required()
def listar_categorias():
    """Lista todas as categorias com suas subcategorias"""
    try:
        ativo = request.args.get('ativo', 'true').lower() == 'true'
        incluir_subcategorias = request.args.get('incluir_subcategorias', 'true').lower() == 'true'
        
        query = Categoria.query.filter_by(ativo=ativo).order_by(Categoria.nome)
        categorias = query.all()
        
        resultado = []
        for categoria in categorias:
            categoria_dict = categoria.to_dict()
            
            if incluir_subcategorias:
                subcategorias = Subcategoria.query.filter_by(
                    categoria_id=categoria.id, 
                    ativo=ativo
                ).order_by(Subcategoria.nome).all()
                categoria_dict['subcategorias'] = [sub.to_dict() for sub in subcategorias]
            
            resultado.append(categoria_dict)
        
        return jsonify({
            'success': True,
            'data': resultado
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/categorias/<int:categoria_id>', methods=['GET'])
@jwt_required()
def obter_categoria(categoria_id):
    """Obtém detalhes de uma categoria específica"""
    try:
        categoria = Categoria.query.get_or_404(categoria_id)
        categoria_dict = categoria.to_dict()
        
        # Incluir subcategorias
        subcategorias = Subcategoria.query.filter_by(categoria_id=categoria_id).order_by(Subcategoria.nome).all()
        categoria_dict['subcategorias'] = [sub.to_dict() for sub in subcategorias]
        
        return jsonify({
            'success': True,
            'data': categoria_dict
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/categorias', methods=['POST'])
@jwt_required()
def criar_categoria():
    """Cria uma nova categoria"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        # Verificar se já existe categoria com o mesmo nome
        if Categoria.query.filter_by(nome=data['nome'], ativo=True).first():
            return jsonify({'success': False, 'message': 'Já existe uma categoria com este nome'}), 400
        
        categoria = Categoria(
            nome=data['nome'],
            descricao=data.get('descricao'),
            cor=data.get('cor', '#2C3E50'),
            icone=data.get('icone'),
            orcamento_mensal=data.get('orcamento_mensal', 0),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(categoria)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Categoria criada com sucesso',
            'data': categoria.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/categorias/<int:categoria_id>', methods=['PUT'])
@jwt_required()
def atualizar_categoria(categoria_id):
    """Atualiza uma categoria existente"""
    try:
        categoria = Categoria.query.get_or_404(categoria_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        # Verificar se já existe outra categoria com o mesmo nome
        if 'nome' in data and data['nome'] != categoria.nome:
            if Categoria.query.filter_by(nome=data['nome'], ativo=True).filter(Categoria.id != categoria_id).first():
                return jsonify({'success': False, 'message': 'Já existe uma categoria com este nome'}), 400
        
        # Atualizar campos
        if 'nome' in data:
            categoria.nome = data['nome']
        if 'descricao' in data:
            categoria.descricao = data['descricao']
        if 'cor' in data:
            categoria.cor = data['cor']
        if 'icone' in data:
            categoria.icone = data['icone']
        if 'orcamento_mensal' in data:
            categoria.orcamento_mensal = data['orcamento_mensal']
        if 'ativo' in data:
            categoria.ativo = data['ativo']
        
        categoria.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Categoria atualizada com sucesso',
            'data': categoria.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/subcategorias', methods=['GET'])
@jwt_required()
def listar_subcategorias():
    """Lista subcategorias, opcionalmente filtradas por categoria"""
    try:
        categoria_id = request.args.get('categoria_id', type=int)
        ativo = request.args.get('ativo', 'true').lower() == 'true'
        
        query = Subcategoria.query.filter_by(ativo=ativo)
        
        if categoria_id:
            query = query.filter_by(categoria_id=categoria_id)
        
        subcategorias = query.order_by(Subcategoria.nome).all()
        
        return jsonify({
            'success': True,
            'data': [sub.to_dict() for sub in subcategorias]
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/subcategorias', methods=['POST'])
@jwt_required()
def criar_subcategoria():
    """Cria uma nova subcategoria"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if not data.get('categoria_id'):
            return jsonify({'success': False, 'message': 'Categoria é obrigatória'}), 400
        
        # Verificar se categoria existe
        categoria = Categoria.query.get(data['categoria_id'])
        if not categoria or not categoria.ativo:
            return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar se já existe subcategoria com o mesmo nome na categoria
        if Subcategoria.query.filter_by(
            categoria_id=data['categoria_id'], 
            nome=data['nome'], 
            ativo=True
        ).first():
            return jsonify({'success': False, 'message': 'Já existe uma subcategoria com este nome nesta categoria'}), 400
        
        subcategoria = Subcategoria(
            categoria_id=data['categoria_id'],
            nome=data['nome'],
            descricao=data.get('descricao'),
            orcamento_mensal=data.get('orcamento_mensal', 0),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(subcategoria)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Subcategoria criada com sucesso',
            'data': subcategoria.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/subcategorias/<int:subcategoria_id>', methods=['PUT'])
@jwt_required()
def atualizar_subcategoria(subcategoria_id):
    """Atualiza uma subcategoria existente"""
    try:
        subcategoria = Subcategoria.query.get_or_404(subcategoria_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if 'categoria_id' in data:
            categoria = Categoria.query.get(data['categoria_id'])
            if not categoria or not categoria.ativo:
                return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar se já existe outra subcategoria com o mesmo nome na categoria
        if 'nome' in data and data['nome'] != subcategoria.nome:
            categoria_id = data.get('categoria_id', subcategoria.categoria_id)
            if Subcategoria.query.filter_by(
                categoria_id=categoria_id, 
                nome=data['nome'], 
                ativo=True
            ).filter(Subcategoria.id != subcategoria_id).first():
                return jsonify({'success': False, 'message': 'Já existe uma subcategoria com este nome nesta categoria'}), 400
        
        # Atualizar campos
        if 'categoria_id' in data:
            subcategoria.categoria_id = data['categoria_id']
        if 'nome' in data:
            subcategoria.nome = data['nome']
        if 'descricao' in data:
            subcategoria.descricao = data['descricao']
        if 'orcamento_mensal' in data:
            subcategoria.orcamento_mensal = data['orcamento_mensal']
        if 'ativo' in data:
            subcategoria.ativo = data['ativo']
        
        subcategoria.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Subcategoria atualizada com sucesso',
            'data': subcategoria.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/categorias/<int:categoria_id>/gastos', methods=['GET'])
@jwt_required()
def gastos_categoria(categoria_id):
    """Obtém gastos de uma categoria por período"""
    try:
        categoria = Categoria.query.get_or_404(categoria_id)
        
        mes = request.args.get('mes', date.today().month, type=int)
        ano = request.args.get('ano', date.today().year, type=int)
        
        # Gastos da categoria no período
        gastos_mes = Transacao.query.filter(
            Transacao.categoria_id == categoria_id,
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            extract('month', Transacao.data_transacao) == mes,
            extract('year', Transacao.data_transacao) == ano
        )
        
        total_mes = gastos_mes.with_entities(func.sum(Transacao.valor)).scalar() or 0
        quantidade_mes = gastos_mes.count()
        
        # Gastos por subcategoria
        gastos_subcategorias = db.session.query(
            Subcategoria.id,
            Subcategoria.nome,
            func.sum(Transacao.valor).label('total'),
            func.count(Transacao.id).label('quantidade')
        ).join(
            Transacao, Transacao.subcategoria_id == Subcategoria.id
        ).filter(
            Subcategoria.categoria_id == categoria_id,
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            extract('month', Transacao.data_transacao) == mes,
            extract('year', Transacao.data_transacao) == ano
        ).group_by(Subcategoria.id, Subcategoria.nome).all()
        
        # Calcular percentual do orçamento
        orcamento_mensal = float(categoria.orcamento_mensal or 0)
        percentual_orcamento = (float(total_mes) / orcamento_mensal * 100) if orcamento_mensal > 0 else 0
        
        resultado = {
            'categoria': categoria.to_dict(),
            'periodo': {
                'mes': mes,
                'ano': ano
            },
            'resumo': {
                'total': float(total_mes),
                'quantidade': quantidade_mes,
                'ticket_medio': float(total_mes / quantidade_mes) if quantidade_mes > 0 else 0,
                'orcamento': orcamento_mensal,
                'percentual_orcamento': round(percentual_orcamento, 2),
                'saldo_orcamento': orcamento_mensal - float(total_mes)
            },
            'subcategorias': [
                {
                    'id': sub_id,
                    'nome': sub_nome,
                    'total': float(total),
                    'quantidade': quantidade,
                    'percentual_categoria': round(float(total) / float(total_mes) * 100, 2) if total_mes > 0 else 0
                }
                for sub_id, sub_nome, total, quantidade in gastos_subcategorias
            ]
        }
        
        return jsonify({
            'success': True,
            'data': resultado
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@categorias_bp.route('/categorias/ranking-gastos', methods=['GET'])
@jwt_required()
def ranking_gastos_categorias():
    """Obtém ranking de categorias por gastos"""
    try:
        mes = request.args.get('mes', date.today().month, type=int)
        ano = request.args.get('ano', date.today().year, type=int)
        
        # Subquery para calcular totais por categoria
        subquery = db.session.query(
            Transacao.categoria_id,
            func.sum(Transacao.valor).label('total'),
            func.count(Transacao.id).label('quantidade')
        ).filter(
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            extract('month', Transacao.data_transacao) == mes,
            extract('year', Transacao.data_transacao) == ano,
            Transacao.categoria_id.isnot(None)
        ).group_by(Transacao.categoria_id).subquery()
        
        # Query principal com join
        ranking = db.session.query(
            Categoria,
            subquery.c.total,
            subquery.c.quantidade
        ).join(
            subquery, Categoria.id == subquery.c.categoria_id
        ).order_by(subquery.c.total.desc()).all()
        
        total_geral = sum(float(total) for _, total, _ in ranking)
        
        resultado = []
        for i, (categoria, total, quantidade) in enumerate(ranking, 1):
            orcamento_mensal = float(categoria.orcamento_mensal or 0)
            percentual_orcamento = (float(total) / orcamento_mensal * 100) if orcamento_mensal > 0 else 0
            percentual_total = (float(total) / total_geral * 100) if total_geral > 0 else 0
            
            resultado.append({
                'posicao': i,
                'categoria': categoria.to_dict(),
                'total': float(total),
                'quantidade': quantidade,
                'ticket_medio': float(total / quantidade) if quantidade > 0 else 0,
                'percentual_orcamento': round(percentual_orcamento, 2),
                'percentual_total': round(percentual_total, 2)
            })
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {'mes': mes, 'ano': ano},
                'total_geral': total_geral,
                'ranking': resultado
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

