"""
Rotas para gerenciamento de representantes e clientes
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Representante, Cliente, Transacao
from datetime import datetime, date
from sqlalchemy import func, extract

representantes_bp = Blueprint('representantes', __name__)

@representantes_bp.route('/representantes', methods=['GET'])
@jwt_required()
def listar_representantes():
    """Lista todos os representantes"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 50, type=int)
        ativo = request.args.get('ativo', 'true').lower() == 'true'
        busca = request.args.get('busca', '').strip()
        
        query = Representante.query.filter_by(ativo=ativo)
        
        if busca:
            query = query.filter(
                (Representante.nome.contains(busca)) |
                (Representante.codigo.contains(busca))
            )
        
        representantes = query.order_by(Representante.nome).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'success': True,
            'data': [rep.to_dict() for rep in representantes.items],
            'pagination': {
                'page': page,
                'pages': representantes.pages,
                'per_page': per_page,
                'total': representantes.total
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/representantes/<int:representante_id>', methods=['GET'])
@jwt_required()
def obter_representante(representante_id):
    """Obtém detalhes de um representante específico"""
    try:
        representante = Representante.query.get_or_404(representante_id)
        return jsonify({
            'success': True,
            'data': representante.to_dict()
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/representantes', methods=['POST'])
@jwt_required()
def criar_representante():
    """Cria um novo representante"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        # Gerar código automático se não fornecido
        codigo = data.get('codigo')
        if not codigo:
            # Gerar código baseado no nome
            nome_limpo = ''.join(c for c in data['nome'].upper() if c.isalnum())[:3]
            ultimo_id = db.session.query(func.max(Representante.id)).scalar() or 0
            codigo = f"{nome_limpo}{ultimo_id + 1:03d}"
        
        # Verificar se código já existe
        if Representante.query.filter_by(codigo=codigo).first():
            return jsonify({'success': False, 'message': 'Código já existe'}), 400
        
        representante = Representante(
            codigo=codigo,
            nome=data['nome'],
            email=data.get('email'),
            telefone=data.get('telefone'),
            regiao=data.get('regiao'),
            comissao_padrao=data.get('comissao_padrao', 0),
            meta_mensal=data.get('meta_mensal', 0),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(representante)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Representante criado com sucesso',
            'data': representante.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/representantes/<int:representante_id>', methods=['PUT'])
@jwt_required()
def atualizar_representante(representante_id):
    """Atualiza um representante existente"""
    try:
        representante = Representante.query.get_or_404(representante_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if 'codigo' in data and data['codigo'] != representante.codigo:
            if Representante.query.filter_by(codigo=data['codigo']).filter(Representante.id != representante_id).first():
                return jsonify({'success': False, 'message': 'Código já existe'}), 400
        
        # Atualizar campos
        if 'codigo' in data:
            representante.codigo = data['codigo']
        if 'nome' in data:
            representante.nome = data['nome']
        if 'email' in data:
            representante.email = data['email']
        if 'telefone' in data:
            representante.telefone = data['telefone']
        if 'regiao' in data:
            representante.regiao = data['regiao']
        if 'comissao_padrao' in data:
            representante.comissao_padrao = data['comissao_padrao']
        if 'meta_mensal' in data:
            representante.meta_mensal = data['meta_mensal']
        if 'observacoes' in data:
            representante.observacoes = data['observacoes']
        if 'ativo' in data:
            representante.ativo = data['ativo']
        
        representante.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Representante atualizado com sucesso',
            'data': representante.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/representantes/<int:representante_id>/performance', methods=['GET'])
@jwt_required()
def performance_representante(representante_id):
    """Obtém performance de um representante"""
    try:
        representante = Representante.query.get_or_404(representante_id)
        
        mes = request.args.get('mes', date.today().month, type=int)
        ano = request.args.get('ano', date.today().year, type=int)
        
        # Query para transações do representante no período
        transacoes_mes = Transacao.query.filter(
            Transacao.representante_id == representante_id,
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            extract('month', Transacao.data_transacao) == mes,
            extract('year', Transacao.data_transacao) == ano
        )
        
        total_mes = transacoes_mes.with_entities(func.sum(Transacao.valor)).scalar() or 0
        quantidade_mes = transacoes_mes.count()
        
        # Performance do ano
        transacoes_ano = Transacao.query.filter(
            Transacao.representante_id == representante_id,
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            extract('year', Transacao.data_transacao) == ano
        )
        
        total_ano = transacoes_ano.with_entities(func.sum(Transacao.valor)).scalar() or 0
        quantidade_ano = transacoes_ano.count()
        
        # Calcular percentual da meta
        meta_mensal = float(representante.meta_mensal or 0)
        percentual_meta = (float(total_mes) / meta_mensal * 100) if meta_mensal > 0 else 0
        
        performance = {
            'representante': representante.to_dict(),
            'periodo': {
                'mes': mes,
                'ano': ano
            },
            'mes_atual': {
                'total': float(total_mes),
                'quantidade': quantidade_mes,
                'ticket_medio': float(total_mes / quantidade_mes) if quantidade_mes > 0 else 0,
                'meta': meta_mensal,
                'percentual_meta': round(percentual_meta, 2)
            },
            'ano_atual': {
                'total': float(total_ano),
                'quantidade': quantidade_ano,
                'ticket_medio': float(total_ano / quantidade_ano) if quantidade_ano > 0 else 0,
                'media_mensal': float(total_ano / mes) if mes > 0 else 0
            }
        }
        
        return jsonify({
            'success': True,
            'data': performance
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/representantes/ranking', methods=['GET'])
@jwt_required()
def ranking_representantes():
    """Obtém ranking de representantes por performance"""
    try:
        mes = request.args.get('mes', date.today().month, type=int)
        ano = request.args.get('ano', date.today().year, type=int)
        
        # Subquery para calcular totais por representante
        subquery = db.session.query(
            Transacao.representante_id,
            func.sum(Transacao.valor).label('total'),
            func.count(Transacao.id).label('quantidade')
        ).filter(
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            extract('month', Transacao.data_transacao) == mes,
            extract('year', Transacao.data_transacao) == ano,
            Transacao.representante_id.isnot(None)
        ).group_by(Transacao.representante_id).subquery()
        
        # Query principal com join
        ranking = db.session.query(
            Representante,
            subquery.c.total,
            subquery.c.quantidade
        ).join(
            subquery, Representante.id == subquery.c.representante_id
        ).order_by(subquery.c.total.desc()).all()
        
        resultado = []
        for i, (representante, total, quantidade) in enumerate(ranking, 1):
            meta_mensal = float(representante.meta_mensal or 0)
            percentual_meta = (float(total) / meta_mensal * 100) if meta_mensal > 0 else 0
            
            resultado.append({
                'posicao': i,
                'representante': representante.to_dict(),
                'total': float(total),
                'quantidade': quantidade,
                'ticket_medio': float(total / quantidade) if quantidade > 0 else 0,
                'percentual_meta': round(percentual_meta, 2)
            })
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {'mes': mes, 'ano': ano},
                'ranking': resultado
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

# Rotas para clientes
@representantes_bp.route('/clientes', methods=['GET'])
@jwt_required()
def listar_clientes():
    """Lista todos os clientes"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 50, type=int)
        representante_id = request.args.get('representante_id', type=int)
        busca = request.args.get('busca', '').strip()
        
        query = Cliente.query
        
        if representante_id:
            query = query.filter_by(representante_id=representante_id)
        
        if busca:
            query = query.filter(Cliente.nome.contains(busca))
        
        clientes = query.order_by(Cliente.nome).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'success': True,
            'data': [cliente.to_dict() for cliente in clientes.items],
            'pagination': {
                'page': page,
                'pages': clientes.pages,
                'per_page': per_page,
                'total': clientes.total
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/clientes', methods=['POST'])
@jwt_required()
def criar_cliente():
    """Cria um novo cliente"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if not data.get('representante_id'):
            return jsonify({'success': False, 'message': 'Representante é obrigatório'}), 400
        
        # Verificar se representante existe
        representante = Representante.query.get(data['representante_id'])
        if not representante or not representante.ativo:
            return jsonify({'success': False, 'message': 'Representante inválido'}), 400
        
        cliente = Cliente(
            nome=data['nome'],
            representante_id=data['representante_id'],
            telefone=data.get('telefone'),
            email=data.get('email'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(cliente)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Cliente criado com sucesso',
            'data': cliente.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@representantes_bp.route('/clientes/<int:cliente_id>', methods=['PUT'])
@jwt_required()
def atualizar_cliente(cliente_id):
    """Atualiza um cliente existente"""
    try:
        cliente = Cliente.query.get_or_404(cliente_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if 'representante_id' in data:
            representante = Representante.query.get(data['representante_id'])
            if not representante or not representante.ativo:
                return jsonify({'success': False, 'message': 'Representante inválido'}), 400
        
        # Atualizar campos
        if 'nome' in data:
            cliente.nome = data['nome']
        if 'representante_id' in data:
            cliente.representante_id = data['representante_id']
        if 'telefone' in data:
            cliente.telefone = data['telefone']
        if 'email' in data:
            cliente.email = data['email']
        if 'observacoes' in data:
            cliente.observacoes = data['observacoes']
        
        cliente.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Cliente atualizado com sucesso',
            'data': cliente.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

