"""
Sistema de Backup e Segurança
Villa Joias - Sistema de Controle de Fluxo de Caixa
"""

import os
import json
import gzip
import shutil
import hashlib
from datetime import datetime, timedelta
from pathlib import Path
import subprocess
import logging
from cryptography.fernet import Fernet
import schedule
import time
import threading

class BackupManager:
    """Gerenciador de backup automático e manual"""
    
    def __init__(self, config):
        self.config = config
        self.backup_dir = Path(config.get('backup_directory', '/var/backups/villa_joias'))
        self.max_backups = config.get('max_backups', 30)
        self.encryption_key = self._get_or_create_key()
        self.cipher = Fernet(self.encryption_key)
        
        # Criar diretórios necessários
        self.backup_dir.mkdir(parents=True, exist_ok=True)
        (self.backup_dir / 'daily').mkdir(exist_ok=True)
        (self.backup_dir / 'weekly').mkdir(exist_ok=True)
        (self.backup_dir / 'monthly').mkdir(exist_ok=True)
        
        # Configurar logging
        self._setup_logging()
        
    def _get_or_create_key(self):
        """Gera ou recupera chave de criptografia"""
        key_file = self.backup_dir / '.encryption_key'
        if key_file.exists():
            with open(key_file, 'rb') as f:
                return f.read()
        else:
            key = Fernet.generate_key()
            with open(key_file, 'wb') as f:
                f.write(key)
            os.chmod(key_file, 0o600)  # Apenas owner pode ler
            return key
    
    def _setup_logging(self):
        """Configura sistema de logs"""
        log_file = self.backup_dir / 'backup.log'
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(levelname)s - %(message)s',
            handlers=[
                logging.FileHandler(log_file),
                logging.StreamHandler()
            ]
        )
        self.logger = logging.getLogger(__name__)
    
    def create_database_backup(self, backup_type='manual'):
        """Cria backup do banco de dados MySQL"""
        try:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            backup_name = f"villa_joias_db_{backup_type}_{timestamp}.sql"
            backup_path = self.backup_dir / backup_type / backup_name
            
            # Comando mysqldump
            cmd = [
                'mysqldump',
                '--host=' + self.config.get('db_host', 'localhost'),
                '--user=' + self.config.get('db_user', 'root'),
                '--password=' + self.config.get('db_password', ''),
                '--single-transaction',
                '--routines',
                '--triggers',
                self.config.get('db_name', 'villa_joias')
            ]
            
            # Executar backup
            with open(backup_path, 'w') as f:
                result = subprocess.run(cmd, stdout=f, stderr=subprocess.PIPE, text=True)
            
            if result.returncode == 0:
                # Comprimir e criptografar
                compressed_path = self._compress_and_encrypt(backup_path)
                os.remove(backup_path)  # Remove arquivo original
                
                self.logger.info(f"Backup do banco criado: {compressed_path}")
                return compressed_path
            else:
                self.logger.error(f"Erro no backup: {result.stderr}")
                return None
                
        except Exception as e:
            self.logger.error(f"Erro ao criar backup do banco: {str(e)}")
            return None
    
    def create_files_backup(self, backup_type='manual'):
        """Cria backup dos arquivos do sistema"""
        try:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            backup_name = f"villa_joias_files_{backup_type}_{timestamp}.tar.gz"
            backup_path = self.backup_dir / backup_type / backup_name
            
            # Diretórios para backup
            source_dirs = [
                '/var/www/villa_joias',  # Código da aplicação
                '/etc/nginx/sites-available/villa_joias',  # Configuração nginx
                '/etc/systemd/system/villa_joias.service'  # Service file
            ]
            
            # Criar arquivo tar comprimido
            with gzip.open(backup_path, 'wb') as f_out:
                cmd = ['tar', '-cf', '-'] + [d for d in source_dirs if os.path.exists(d)]
                result = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                
                if result.returncode == 0:
                    f_out.write(result.stdout)
                    
            # Criptografar arquivo
            encrypted_path = self._encrypt_file(backup_path)
            os.remove(backup_path)  # Remove arquivo original
            
            self.logger.info(f"Backup de arquivos criado: {encrypted_path}")
            return encrypted_path
            
        except Exception as e:
            self.logger.error(f"Erro ao criar backup de arquivos: {str(e)}")
            return None
    
    def _compress_and_encrypt(self, file_path):
        """Comprime e criptografa arquivo"""
        compressed_path = str(file_path) + '.gz'
        encrypted_path = compressed_path + '.enc'
        
        # Comprimir
        with open(file_path, 'rb') as f_in:
            with gzip.open(compressed_path, 'wb') as f_out:
                shutil.copyfileobj(f_in, f_out)
        
        # Criptografar
        with open(compressed_path, 'rb') as f_in:
            with open(encrypted_path, 'wb') as f_out:
                f_out.write(self.cipher.encrypt(f_in.read()))
        
        os.remove(compressed_path)  # Remove arquivo comprimido
        return encrypted_path
    
    def _encrypt_file(self, file_path):
        """Criptografa arquivo"""
        encrypted_path = str(file_path) + '.enc'
        
        with open(file_path, 'rb') as f_in:
            with open(encrypted_path, 'wb') as f_out:
                f_out.write(self.cipher.encrypt(f_in.read()))
        
        return encrypted_path
    
    def restore_backup(self, backup_file, restore_type='database'):
        """Restaura backup específico"""
        try:
            # Descriptografar
            decrypted_path = self._decrypt_file(backup_file)
            
            if restore_type == 'database':
                return self._restore_database(decrypted_path)
            elif restore_type == 'files':
                return self._restore_files(decrypted_path)
            
        except Exception as e:
            self.logger.error(f"Erro ao restaurar backup: {str(e)}")
            return False
    
    def _decrypt_file(self, encrypted_file):
        """Descriptografa arquivo"""
        decrypted_path = str(encrypted_file).replace('.enc', '')
        
        with open(encrypted_file, 'rb') as f_in:
            with open(decrypted_path, 'wb') as f_out:
                f_out.write(self.cipher.decrypt(f_in.read()))
        
        return decrypted_path
    
    def _restore_database(self, sql_file):
        """Restaura banco de dados"""
        try:
            # Descomprimir se necessário
            if sql_file.endswith('.gz'):
                with gzip.open(sql_file, 'rt') as f_in:
                    sql_content = f_in.read()
            else:
                with open(sql_file, 'r') as f_in:
                    sql_content = f_in.read()
            
            # Executar restore
            cmd = [
                'mysql',
                '--host=' + self.config.get('db_host', 'localhost'),
                '--user=' + self.config.get('db_user', 'root'),
                '--password=' + self.config.get('db_password', ''),
                self.config.get('db_name', 'villa_joias')
            ]
            
            result = subprocess.run(cmd, input=sql_content, text=True, 
                                  capture_output=True)
            
            if result.returncode == 0:
                self.logger.info("Banco de dados restaurado com sucesso")
                return True
            else:
                self.logger.error(f"Erro ao restaurar banco: {result.stderr}")
                return False
                
        except Exception as e:
            self.logger.error(f"Erro na restauração: {str(e)}")
            return False
    
    def cleanup_old_backups(self):
        """Remove backups antigos"""
        for backup_type in ['daily', 'weekly', 'monthly']:
            backup_dir = self.backup_dir / backup_type
            if not backup_dir.exists():
                continue
                
            # Listar arquivos por data
            files = []
            for file_path in backup_dir.glob('*.enc'):
                stat = file_path.stat()
                files.append((stat.st_mtime, file_path))
            
            # Ordenar por data (mais antigos primeiro)
            files.sort()
            
            # Remover excesso
            max_files = self.max_backups
            if backup_type == 'weekly':
                max_files = 12  # 3 meses
            elif backup_type == 'monthly':
                max_files = 24  # 2 anos
            
            while len(files) > max_files:
                _, old_file = files.pop(0)
                old_file.unlink()
                self.logger.info(f"Backup antigo removido: {old_file}")
    
    def get_backup_info(self):
        """Retorna informações sobre backups"""
        info = {
            'daily': [],
            'weekly': [],
            'monthly': [],
            'manual': []
        }
        
        for backup_type in info.keys():
            backup_dir = self.backup_dir / backup_type
            if not backup_dir.exists():
                continue
                
            for file_path in backup_dir.glob('*.enc'):
                stat = file_path.stat()
                info[backup_type].append({
                    'name': file_path.name,
                    'size': stat.st_size,
                    'created': datetime.fromtimestamp(stat.st_mtime).isoformat(),
                    'path': str(file_path)
                })
        
        return info
    
    def verify_backup_integrity(self, backup_file):
        """Verifica integridade do backup"""
        try:
            # Tentar descriptografar
            decrypted = self._decrypt_file(backup_file)
            
            # Verificar se é arquivo válido
            if backup_file.endswith('_db_'):
                # Verificar SQL
                with open(decrypted, 'r') as f:
                    content = f.read(1000)
                    if 'CREATE TABLE' in content or 'INSERT INTO' in content:
                        os.remove(decrypted)
                        return True
            
            os.remove(decrypted)
            return False
            
        except Exception:
            return False

class SecurityManager:
    """Gerenciador de segurança e auditoria"""
    
    def __init__(self, config):
        self.config = config
        self.log_dir = Path(config.get('log_directory', '/var/log/villa_joias'))
        self.log_dir.mkdir(parents=True, exist_ok=True)
        
        # Configurar logging de segurança
        self._setup_security_logging()
    
    def _setup_security_logging(self):
        """Configura logs de segurança"""
        security_log = self.log_dir / 'security.log'
        
        self.security_logger = logging.getLogger('security')
        handler = logging.FileHandler(security_log)
        formatter = logging.Formatter(
            '%(asctime)s - %(levelname)s - %(message)s'
        )
        handler.setFormatter(formatter)
        self.security_logger.addHandler(handler)
        self.security_logger.setLevel(logging.INFO)
    
    def log_user_action(self, user_id, action, details=None):
        """Registra ação do usuário"""
        log_entry = {
            'timestamp': datetime.now().isoformat(),
            'user_id': user_id,
            'action': action,
            'details': details or {},
            'ip_address': self._get_client_ip()
        }
        
        self.security_logger.info(json.dumps(log_entry))
    
    def log_login_attempt(self, email, success, ip_address):
        """Registra tentativa de login"""
        log_entry = {
            'timestamp': datetime.now().isoformat(),
            'type': 'login_attempt',
            'email': email,
            'success': success,
            'ip_address': ip_address
        }
        
        self.security_logger.info(json.dumps(log_entry))
    
    def check_suspicious_activity(self, user_id):
        """Verifica atividade suspeita"""
        # Implementar lógica de detecção
        # Por exemplo: muitos logins falhados, ações em horários incomuns, etc.
        pass
    
    def _get_client_ip(self):
        """Obtém IP do cliente (implementar conforme framework)"""
        return "127.0.0.1"  # Placeholder

class BackupScheduler:
    """Agendador de backups automáticos"""
    
    def __init__(self, backup_manager):
        self.backup_manager = backup_manager
        self.running = False
        
    def start_scheduler(self):
        """Inicia agendamento de backups"""
        # Backup diário às 2:00
        schedule.every().day.at("02:00").do(
            self._run_daily_backup
        )
        
        # Backup semanal aos domingos às 3:00
        schedule.every().sunday.at("03:00").do(
            self._run_weekly_backup
        )
        
        # Backup mensal no dia 1 às 4:00
        schedule.every().month.do(
            self._run_monthly_backup
        )
        
        # Limpeza de backups antigos diariamente às 5:00
        schedule.every().day.at("05:00").do(
            self.backup_manager.cleanup_old_backups
        )
        
        self.running = True
        
        # Thread para executar agendamentos
        def run_scheduler():
            while self.running:
                schedule.run_pending()
                time.sleep(60)  # Verificar a cada minuto
        
        scheduler_thread = threading.Thread(target=run_scheduler, daemon=True)
        scheduler_thread.start()
    
    def stop_scheduler(self):
        """Para agendamento"""
        self.running = False
        schedule.clear()
    
    def _run_daily_backup(self):
        """Executa backup diário"""
        self.backup_manager.create_database_backup('daily')
        self.backup_manager.create_files_backup('daily')
    
    def _run_weekly_backup(self):
        """Executa backup semanal"""
        self.backup_manager.create_database_backup('weekly')
        self.backup_manager.create_files_backup('weekly')
    
    def _run_monthly_backup(self):
        """Executa backup mensal"""
        self.backup_manager.create_database_backup('monthly')
        self.backup_manager.create_files_backup('monthly')

# Configuração padrão
DEFAULT_CONFIG = {
    'backup_directory': '/var/backups/villa_joias',
    'log_directory': '/var/log/villa_joias',
    'max_backups': 30,
    'db_host': 'localhost',
    'db_name': 'villa_joias',
    'db_user': 'villa_joias_user',
    'db_password': 'secure_password'
}

# Inicialização
def initialize_backup_system(config=None):
    """Inicializa sistema de backup"""
    if config is None:
        config = DEFAULT_CONFIG
    
    backup_manager = BackupManager(config)
    security_manager = SecurityManager(config)
    scheduler = BackupScheduler(backup_manager)
    
    return backup_manager, security_manager, scheduler

