import pytest
import sys
import os
from datetime import datetime, timedelta
import json

# Adicionar o diretório src ao path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'src'))

from main import create_app
from config import Config

class TestConfig(Config):
    TESTING = True
    SQLALCHEMY_DATABASE_URI = 'sqlite:///:memory:'
    JWT_SECRET_KEY = 'test-secret-key'
    WTF_CSRF_ENABLED = False

@pytest.fixture
def app():
    """Criar aplicação Flask para testes"""
    app = create_app(TestConfig)
    
    with app.app_context():
        # Criar tabelas de teste
        from models.core import db
        db.create_all()
        
        # Dados de teste básicos
        from models.usuarios import Usuario
        from werkzeug.security import generate_password_hash
        
        # Criar usuário administrador de teste
        admin = Usuario(
            nome='Admin Teste',
            email='admin@test.com',
            senha=generate_password_hash('123456'),
            tipo='administrador',
            status='ativo'
        )
        db.session.add(admin)
        db.session.commit()
        
        yield app
        
        db.session.remove()
        db.drop_all()

@pytest.fixture
def client(app):
    """Cliente de teste"""
    return app.test_client()

@pytest.fixture
def auth_headers(client):
    """Headers de autenticação para testes"""
    # Login para obter token
    response = client.post('/api/auth/login', json={
        'email': 'admin@test.com',
        'senha': '123456'
    })
    
    data = json.loads(response.data)
    token = data.get('token')
    
    return {'Authorization': f'Bearer {token}'}

class TestAuth:
    """Testes de autenticação"""
    
    def test_login_sucesso(self, client):
        """Teste de login com sucesso"""
        response = client.post('/api/auth/login', json={
            'email': 'admin@test.com',
            'senha': '123456'
        })
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert 'token' in data
        assert data['usuario']['email'] == 'admin@test.com'
    
    def test_login_senha_incorreta(self, client):
        """Teste de login com senha incorreta"""
        response = client.post('/api/auth/login', json={
            'email': 'admin@test.com',
            'senha': 'senha_errada'
        })
        
        assert response.status_code == 401
        data = json.loads(response.data)
        assert 'error' in data
    
    def test_login_usuario_inexistente(self, client):
        """Teste de login com usuário inexistente"""
        response = client.post('/api/auth/login', json={
            'email': 'inexistente@test.com',
            'senha': '123456'
        })
        
        assert response.status_code == 401

class TestCaixas:
    """Testes do módulo de caixas"""
    
    def test_criar_caixa(self, client, auth_headers):
        """Teste de criação de caixa"""
        response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Teste',
                'tipo': 'capital_giro',
                'descricao': 'Caixa para testes',
                'saldo_inicial': 1000.00,
                'instituicao': 'Banco Teste'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'Caixa Teste'
        assert data['saldo_atual'] == 1000.00
    
    def test_listar_caixas(self, client, auth_headers):
        """Teste de listagem de caixas"""
        # Criar uma caixa primeiro
        client.post('/api/caixas', 
            json={
                'nome': 'Caixa Lista',
                'tipo': 'reserva',
                'saldo_inicial': 500.00
            },
            headers=auth_headers
        )
        
        response = client.get('/api/caixas', headers=auth_headers)
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert len(data) >= 1
        assert any(caixa['nome'] == 'Caixa Lista' for caixa in data)
    
    def test_obter_caixa_por_id(self, client, auth_headers):
        """Teste de obtenção de caixa por ID"""
        # Criar caixa
        response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa ID',
                'tipo': 'conta_corrente',
                'saldo_inicial': 750.00
            },
            headers=auth_headers
        )
        
        caixa_data = json.loads(response.data)
        caixa_id = caixa_data['id']
        
        # Buscar por ID
        response = client.get(f'/api/caixas/{caixa_id}', headers=auth_headers)
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert data['nome'] == 'Caixa ID'
        assert data['id'] == caixa_id

class TestTransacoes:
    """Testes do módulo de transações"""
    
    def test_criar_entrada(self, client, auth_headers):
        """Teste de criação de entrada"""
        # Criar caixa primeiro
        caixa_response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Entrada',
                'tipo': 'capital_giro',
                'saldo_inicial': 0.00
            },
            headers=auth_headers
        )
        caixa_data = json.loads(caixa_response.data)
        caixa_id = caixa_data['id']
        
        # Criar representante
        rep_response = client.post('/api/representantes', 
            json={
                'nome': 'Rep Teste',
                'codigo': 'REP001',
                'telefone': '11999999999'
            },
            headers=auth_headers
        )
        rep_data = json.loads(rep_response.data)
        rep_id = rep_data['id']
        
        # Criar entrada
        response = client.post('/api/transacoes', 
            json={
                'tipo': 'entrada',
                'valor': 500.00,
                'caixa_id': caixa_id,
                'representante_id': rep_id,
                'descricao': 'Pagamento teste',
                'metodo_pagamento': 'pix'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['valor'] == 500.00
        assert data['tipo'] == 'entrada'
    
    def test_criar_saida(self, client, auth_headers):
        """Teste de criação de saída"""
        # Criar caixa com saldo
        caixa_response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Saida',
                'tipo': 'capital_giro',
                'saldo_inicial': 1000.00
            },
            headers=auth_headers
        )
        caixa_data = json.loads(caixa_response.data)
        caixa_id = caixa_data['id']
        
        # Criar categoria
        cat_response = client.post('/api/categorias', 
            json={
                'nome': 'Despesas Teste',
                'descricao': 'Categoria para testes'
            },
            headers=auth_headers
        )
        cat_data = json.loads(cat_response.data)
        cat_id = cat_data['id']
        
        # Criar saída
        response = client.post('/api/transacoes', 
            json={
                'tipo': 'saida',
                'valor': 200.00,
                'caixa_id': caixa_id,
                'categoria_id': cat_id,
                'descricao': 'Despesa teste'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['valor'] == 200.00
        assert data['tipo'] == 'saida'

class TestRepresentantes:
    """Testes do módulo de representantes"""
    
    def test_criar_representante(self, client, auth_headers):
        """Teste de criação de representante"""
        response = client.post('/api/representantes', 
            json={
                'nome': 'João Silva',
                'codigo': 'JS001',
                'telefone': '11987654321',
                'email': 'joao@test.com'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'João Silva'
        assert data['codigo'] == 'JS001'
    
    def test_listar_representantes(self, client, auth_headers):
        """Teste de listagem de representantes"""
        # Criar representante
        client.post('/api/representantes', 
            json={
                'nome': 'Maria Santos',
                'codigo': 'MS001',
                'telefone': '11999888777'
            },
            headers=auth_headers
        )
        
        response = client.get('/api/representantes', headers=auth_headers)
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert len(data) >= 1
        assert any(rep['nome'] == 'Maria Santos' for rep in data)

class TestCategorias:
    """Testes do módulo de categorias"""
    
    def test_criar_categoria(self, client, auth_headers):
        """Teste de criação de categoria"""
        response = client.post('/api/categorias', 
            json={
                'nome': 'Despesas Operacionais',
                'descricao': 'Gastos operacionais da empresa',
                'cor': '#FF5733'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'Despesas Operacionais'
        assert data['cor'] == '#FF5733'
    
    def test_criar_subcategoria(self, client, auth_headers):
        """Teste de criação de subcategoria"""
        # Criar categoria pai
        cat_response = client.post('/api/categorias', 
            json={
                'nome': 'Funcionários',
                'descricao': 'Gastos com funcionários'
            },
            headers=auth_headers
        )
        cat_data = json.loads(cat_response.data)
        cat_id = cat_data['id']
        
        # Criar subcategoria
        response = client.post('/api/categorias', 
            json={
                'nome': 'Salários',
                'descricao': 'Pagamento de salários',
                'categoria_pai_id': cat_id
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'Salários'
        assert data['categoria_pai_id'] == cat_id

class TestCartoes:
    """Testes do módulo de cartões"""
    
    def test_criar_cartao(self, client, auth_headers):
        """Teste de criação de cartão"""
        response = client.post('/api/cartoes', 
            json={
                'nome': 'Cartão Teste',
                'limite': 5000.00,
                'dia_fechamento': 15,
                'dia_vencimento': 25,
                'banco': 'Banco Teste'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'Cartão Teste'
        assert data['limite'] == 5000.00
    
    def test_criar_despesa_cartao(self, client, auth_headers):
        """Teste de criação de despesa no cartão"""
        # Criar cartão
        cartao_response = client.post('/api/cartoes', 
            json={
                'nome': 'Cartão Despesa',
                'limite': 3000.00,
                'dia_fechamento': 10,
                'dia_vencimento': 20
            },
            headers=auth_headers
        )
        cartao_data = json.loads(cartao_response.data)
        cartao_id = cartao_data['id']
        
        # Criar categoria
        cat_response = client.post('/api/categorias', 
            json={
                'nome': 'Marketing',
                'descricao': 'Gastos com marketing'
            },
            headers=auth_headers
        )
        cat_data = json.loads(cat_response.data)
        cat_id = cat_data['id']
        
        # Criar despesa
        response = client.post('/api/cartoes/despesas', 
            json={
                'cartao_id': cartao_id,
                'valor': 150.00,
                'descricao': 'Publicidade online',
                'categoria_id': cat_id,
                'parcelas': 1
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['valor'] == 150.00
        assert data['cartao_id'] == cartao_id

class TestCobranca:
    """Testes do módulo de cobrança"""
    
    def test_criar_inadimplente(self, client, auth_headers):
        """Teste de criação de inadimplente"""
        # Criar representante
        rep_response = client.post('/api/representantes', 
            json={
                'nome': 'Rep Cobrança',
                'codigo': 'RC001'
            },
            headers=auth_headers
        )
        rep_data = json.loads(rep_response.data)
        rep_id = rep_data['id']
        
        response = client.post('/api/cobranca/inadimplentes', 
            json={
                'nome': 'Cliente Inadimplente',
                'cpf': '12345678901',
                'telefone_01': '11999999999',
                'representante_id': rep_id,
                'saldo_original': 1000.00,
                'saldo_atual': 1200.00,
                'numero_contrato': 'CONT001',
                'cidade': 'São Paulo',
                'estado': 'SP'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['nome'] == 'Cliente Inadimplente'
        assert data['cpf'] == '12345678901'
    
    def test_criar_acordo(self, client, auth_headers):
        """Teste de criação de acordo"""
        # Criar inadimplente primeiro
        rep_response = client.post('/api/representantes', 
            json={'nome': 'Rep Acordo', 'codigo': 'RA001'},
            headers=auth_headers
        )
        rep_data = json.loads(rep_response.data)
        
        inad_response = client.post('/api/cobranca/inadimplentes', 
            json={
                'nome': 'Cliente Acordo',
                'cpf': '98765432100',
                'representante_id': rep_data['id'],
                'saldo_original': 800.00,
                'saldo_atual': 900.00,
                'numero_contrato': 'CONT002'
            },
            headers=auth_headers
        )
        inad_data = json.loads(inad_response.data)
        
        # Criar acordo
        response = client.post('/api/cobranca/acordos', 
            json={
                'inadimplente_id': inad_data['id'],
                'valor_original': 900.00,
                'valor_acordo': 600.00,
                'desconto_percentual': 33.33,
                'tipo_pagamento': 'a_vista',
                'parcelas': 1
            },
            headers=auth_headers
        )
        
        assert response.status_code == 201
        data = json.loads(response.data)
        assert data['valor_acordo'] == 600.00
        assert data['tipo_pagamento'] == 'a_vista'

if __name__ == '__main__':
    pytest.main([__file__, '-v'])

