import React, { useState, useEffect } from 'react';
import { MessageSquare, Send, Bot, User, Settings, Phone, Zap } from 'lucide-react';

const ChatManus = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState([
    {
      id: 1,
      type: 'manus',
      content: '🏆 Oi! Sou o Manus, seu CIO da Villa Joias! Como posso ajudar hoje?',
      timestamp: new Date()
    }
  ]);
  const [inputMessage, setInputMessage] = useState('');
  const [isTyping, setIsTyping] = useState(false);
  const [whatsappConfig, setWhatsappConfig] = useState({
    numero: '',
    configurado: false
  });

  // Perguntas rápidas sugeridas
  const perguntasRapidas = [
    "Como está o faturamento este mês?",
    "Qual representante está melhor?",
    "Quanto tenho de saldo nos caixas?",
    "Como está a inadimplência?",
    "Preciso transferir dinheiro?"
  ];

  const enviarMensagem = async () => {
    if (!inputMessage.trim()) return;

    const novaMensagem = {
      id: Date.now(),
      type: 'user',
      content: inputMessage,
      timestamp: new Date()
    };

    setMessages(prev => [...prev, novaMensagem]);
    setInputMessage('');
    setIsTyping(true);

    try {
      // Simular resposta do Manus
      setTimeout(() => {
        const resposta = gerarRespostaManus(inputMessage);
        setMessages(prev => [...prev, {
          id: Date.now() + 1,
          type: 'manus',
          content: resposta,
          timestamp: new Date()
        }]);
        setIsTyping(false);
      }, 1500);

    } catch (error) {
      console.error('Erro ao enviar mensagem:', error);
      setIsTyping(false);
    }
  };

  const gerarRespostaManus = (pergunta) => {
    const perguntaLower = pergunta.toLowerCase();

    if (perguntaLower.includes('faturamento') || perguntaLower.includes('vendas')) {
      return `📊 **Faturamento Junho 2025**

💰 **Total:** R$ 145.230,80
📈 **Crescimento:** +12,5% vs maio
🥇 **Melhor dia:** 15/06 - R$ 8.750,00

🎯 **Meta mensal:** R$ 150.000 (97% atingido)
⚡ **Projeção:** Vamos bater a meta até dia 28!`;
    }

    if (perguntaLower.includes('representante')) {
      return `👥 **Performance Representantes**

🥇 **João Silva:** R$ 15.430,50 (40% acima da média)
🥈 **Maria Santos:** R$ 12.850,30 (25% acima)
🥉 **Carlos Lima:** R$ 9.750,20 (na média)

💡 **Insight:** João está usando visitas presenciais que aumentaram conversão em 35%. Recomendo replicar!`;
    }

    if (perguntaLower.includes('saldo') || perguntaLower.includes('caixa')) {
      return `💰 **Situação dos Caixas**

🏦 **Principal:** R$ 85.430,50
💎 **Reserva CDB:** R$ 125.000,00 (+12% a.a.)
💳 **Conta Corrente:** R$ 15.750,30
💵 **Físico:** R$ 2.850,00

⚡ **Recomendação:** Transferir R$ 20.000 do principal para CDB para otimizar rendimentos.`;
    }

    if (perguntaLower.includes('inadimpl') || perguntaLower.includes('atraso')) {
      return `🚨 **Status Inadimplência**

📊 **Total:** R$ 485.750,80 (156 clientes)
⏰ **Críticos:** 23 clientes há +60 dias
💰 **Recuperado:** R$ 125.430,50 este mês
📈 **Taxa:** 14,7% (meta: 15%)

🎯 **Ação:** Vi já está negociando com os 5 maiores devedores!`;
    }

    if (perguntaLower.includes('transferir') || perguntaLower.includes('dinheiro')) {
      return `💸 **Análise de Transferências**

✅ **Recomendado agora:**
• R$ 20.000: Principal → CDB (otimizar rendimento)
• R$ 5.000: Reserva → Principal (fluxo operacional)

⚠️ **Não recomendado:**
• Sacar da reserva CDB (perderia R$ 200/mês em juros)

🎯 **Resultado:** +R$ 300/mês em rendimentos!`;
    }

    // Resposta padrão
    return `🤔 Entendi sua pergunta! Posso ajudar com:

📊 Análises financeiras
💰 Situação dos caixas  
👥 Performance de representantes
🚨 Status de inadimplência
💡 Dicas estratégicas
📈 Projeções e metas

Que informação específica você precisa?`;
  };

  const configurarWhatsApp = async () => {
    if (!whatsappConfig.numero) {
      alert('Digite seu número do WhatsApp');
      return;
    }

    try {
      // Simular configuração
      setWhatsappConfig(prev => ({ ...prev, configurado: true }));
      
      setMessages(prev => [...prev, {
        id: Date.now(),
        type: 'manus',
        content: `✅ **WhatsApp Configurado!**

📱 Número: ${whatsappConfig.numero}

Agora você receberá:
⚡ Alertas importantes
📊 Relatórios diários
💡 Dicas estratégicas
🚨 Notificações urgentes

Teste: Vou te enviar uma mensagem agora! 📲`,
        timestamp: new Date()
      }]);

    } catch (error) {
      console.error('Erro ao configurar WhatsApp:', error);
    }
  };

  return (
    <>
      {/* Botão flutuante */}
      <div className="fixed bottom-6 right-6 z-50">
        <button
          onClick={() => setIsOpen(!isOpen)}
          className="bg-gradient-to-r from-yellow-500 to-yellow-600 text-white p-4 rounded-full shadow-lg hover:shadow-xl transition-all duration-300 transform hover:scale-105"
        >
          <MessageSquare className="h-6 w-6" />
        </button>
      </div>

      {/* Chat Window */}
      {isOpen && (
        <div className="fixed bottom-24 right-6 w-96 h-[600px] bg-white rounded-lg shadow-2xl border border-gray-200 z-50 flex flex-col">
          {/* Header */}
          <div className="bg-gradient-to-r from-yellow-500 to-yellow-600 text-white p-4 rounded-t-lg">
            <div className="flex items-center justify-between">
              <div className="flex items-center">
                <div className="w-10 h-10 bg-white rounded-full flex items-center justify-center mr-3">
                  <Bot className="h-6 w-6 text-yellow-600" />
                </div>
                <div>
                  <h3 className="font-semibold">Manus - CIO</h3>
                  <p className="text-sm text-yellow-100">Villa Joias • Online</p>
                </div>
              </div>
              <button
                onClick={() => setIsOpen(false)}
                className="text-white hover:text-yellow-200"
              >
                ✕
              </button>
            </div>
          </div>

          {/* Configuração WhatsApp */}
          {!whatsappConfig.configurado && (
            <div className="p-4 bg-yellow-50 border-b border-yellow-200">
              <div className="flex items-center mb-2">
                <Phone className="h-4 w-4 text-yellow-600 mr-2" />
                <span className="text-sm font-medium text-yellow-800">Configurar WhatsApp</span>
              </div>
              <div className="flex gap-2">
                <input
                  type="text"
                  placeholder="(11) 99999-9999"
                  value={whatsappConfig.numero}
                  onChange={(e) => setWhatsappConfig(prev => ({ ...prev, numero: e.target.value }))}
                  className="flex-1 px-3 py-2 text-sm border border-yellow-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
                />
                <button
                  onClick={configurarWhatsApp}
                  className="px-3 py-2 bg-yellow-600 text-white text-sm rounded-md hover:bg-yellow-700"
                >
                  <Zap className="h-4 w-4" />
                </button>
              </div>
            </div>
          )}

          {/* Messages */}
          <div className="flex-1 overflow-y-auto p-4 space-y-4">
            {messages.map((message) => (
              <div
                key={message.id}
                className={`flex ${message.type === 'user' ? 'justify-end' : 'justify-start'}`}
              >
                <div
                  className={`max-w-[80%] p-3 rounded-lg ${
                    message.type === 'user'
                      ? 'bg-yellow-500 text-white'
                      : 'bg-gray-100 text-gray-900'
                  }`}
                >
                  <div className="whitespace-pre-wrap text-sm">{message.content}</div>
                  <div className={`text-xs mt-1 ${
                    message.type === 'user' ? 'text-yellow-100' : 'text-gray-500'
                  }`}>
                    {message.timestamp.toLocaleTimeString('pt-BR', { 
                      hour: '2-digit', 
                      minute: '2-digit' 
                    })}
                  </div>
                </div>
              </div>
            ))}

            {isTyping && (
              <div className="flex justify-start">
                <div className="bg-gray-100 p-3 rounded-lg">
                  <div className="flex space-x-1">
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }}></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }}></div>
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Perguntas Rápidas */}
          <div className="p-3 border-t border-gray-200 bg-gray-50">
            <div className="text-xs text-gray-600 mb-2">Perguntas rápidas:</div>
            <div className="flex flex-wrap gap-1">
              {perguntasRapidas.slice(0, 3).map((pergunta, index) => (
                <button
                  key={index}
                  onClick={() => setInputMessage(pergunta)}
                  className="text-xs bg-white border border-gray-300 rounded-full px-2 py-1 hover:bg-yellow-50 hover:border-yellow-300 transition-colors"
                >
                  {pergunta}
                </button>
              ))}
            </div>
          </div>

          {/* Input */}
          <div className="p-4 border-t border-gray-200">
            <div className="flex gap-2">
              <input
                type="text"
                value={inputMessage}
                onChange={(e) => setInputMessage(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && enviarMensagem()}
                placeholder="Digite sua pergunta..."
                className="flex-1 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
              />
              <button
                onClick={enviarMensagem}
                disabled={!inputMessage.trim() || isTyping}
                className="px-4 py-2 bg-yellow-500 text-white rounded-md hover:bg-yellow-600 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                <Send className="h-4 w-4" />
              </button>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default ChatManus;

