import React, { useState, useEffect } from 'react';
import { 
  BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
  LineChart, Line, PieChart, Pie, Cell, Area, AreaChart
} from 'recharts';
import { 
  TrendingUp, TrendingDown, DollarSign, Users, AlertTriangle, 
  CheckCircle, Clock, Target, Phone, MessageSquare, FileText,
  Download, Upload, Settings, Eye, Edit, Trash2, Plus
} from 'lucide-react';

// Cores do tema Villa Joias
const colors = {
  primary: '#D4AF37',    // Dourado
  secondary: '#1A1A1A',  // Preto
  accent: '#F5F5F5',     // Cinza claro
  success: '#10B981',    // Verde
  warning: '#F59E0B',    // Amarelo
  danger: '#EF4444',     // Vermelho
  info: '#3B82F6'        // Azul
};

const CobrancaModule = () => {
  const [activeTab, setActiveTab] = useState('dashboard');
  const [inadimplentes, setInadimplentes] = useState([]);
  const [acordos, setAcordos] = useState([]);
  const [configuracaoDescontos, setConfiguracaoDescontos] = useState(null);
  const [loading, setLoading] = useState(false);

  // Dados simulados para demonstração
  const dashboardData = {
    metricas_principais: {
      total_inadimplentes_ativos: 156,
      valor_total_inadimplencia: 485750.80,
      acordos_mes: 23,
      valor_recuperado_mes: 125430.50,
      taxa_conversao_mes: 14.7,
      parcelas_vencendo_hoje: 8
    },
    top_inadimplentes: [
      { nome_devedor: 'João Silva Santos', saldo_atual: 15750.00, dias_em_atraso: 45 },
      { nome_devedor: 'Maria Oliveira Costa', saldo_atual: 12300.50, dias_em_atraso: 32 },
      { nome_devedor: 'Carlos Eduardo Lima', saldo_atual: 9850.75, dias_em_atraso: 67 },
      { nome_devedor: 'Ana Paula Ferreira', saldo_atual: 8420.30, dias_em_atraso: 28 },
      { nome_devedor: 'Roberto Alves Souza', saldo_atual: 7650.90, dias_em_atraso: 51 }
    ]
  };

  const performanceData = [
    { mes: 'Jan', acordos: 18, valor_recuperado: 95000, taxa_conversao: 12.5 },
    { mes: 'Fev', valor_recuperado: 110000, acordos: 21, taxa_conversao: 13.8 },
    { mes: 'Mar', valor_recuperado: 125000, acordos: 23, taxa_conversao: 14.7 },
    { mes: 'Abr', valor_recuperado: 98000, acordos: 19, taxa_conversao: 11.9 },
    { mes: 'Mai', valor_recuperado: 135000, acordos: 26, taxa_conversao: 16.2 },
    { mes: 'Jun', valor_recuperado: 125430, acordos: 23, taxa_conversao: 14.7 }
  ];

  const tiposAcordo = [
    { name: 'À Vista', value: 68, color: colors.success },
    { name: 'Parcelado', value: 32, color: colors.warning }
  ];

  const MetricCard = ({ title, value, subtitle, icon: Icon, trend, color = colors.primary }) => (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-start justify-between">
        <div className="flex-1">
          <p className="text-sm font-medium text-gray-600 text-left">{title}</p>
          <p className="text-2xl font-bold text-gray-900 mt-1 text-left">{value}</p>
          {subtitle && (
            <p className="text-sm text-gray-500 mt-1 text-left">{subtitle}</p>
          )}
          {trend && (
            <div className="mt-4 flex items-center justify-start">
              {trend > 0 ? (
                <TrendingUp className="h-4 w-4 text-green-500 mr-1" />
              ) : (
                <TrendingDown className="h-4 w-4 text-red-500 mr-1" />
              )}
              <span className={`text-sm font-medium ${trend > 0 ? 'text-green-600' : 'text-red-600'}`}>
                {Math.abs(trend)}%
              </span>
              <span className="text-sm text-gray-500 ml-1">vs mês anterior</span>
            </div>
          )}
        </div>
        <div className={`p-3 rounded-lg ml-4`} style={{ backgroundColor: `${color}20` }}>
          <Icon className="h-6 w-6" style={{ color }} />
        </div>
      </div>
    </div>
  );

  const Dashboard = () => (
    <div className="space-y-6">
      {/* Métricas Principais */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        <MetricCard
          title="Inadimplentes Ativos"
          value={dashboardData.metricas_principais.total_inadimplentes_ativos}
          icon={Users}
          color={colors.danger}
        />
        <MetricCard
          title="Valor Total Inadimplência"
          value={`R$ ${dashboardData.metricas_principais.valor_total_inadimplencia.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}`}
          icon={DollarSign}
          color={colors.warning}
        />
        <MetricCard
          title="Acordos Este Mês"
          value={dashboardData.metricas_principais.acordos_mes}
          subtitle={`Taxa: ${dashboardData.metricas_principais.taxa_conversao_mes}%`}
          icon={CheckCircle}
          color={colors.success}
          trend={2.3}
        />
        <MetricCard
          title="Valor Recuperado"
          value={`R$ ${dashboardData.metricas_principais.valor_recuperado_mes.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}`}
          icon={TrendingUp}
          color={colors.primary}
          trend={8.5}
        />
        <MetricCard
          title="Parcelas Hoje"
          value={dashboardData.metricas_principais.parcelas_vencendo_hoje}
          subtitle="Vencendo hoje"
          icon={Clock}
          color={colors.info}
        />
        <MetricCard
          title="Taxa Conversão"
          value={`${dashboardData.metricas_principais.taxa_conversao_mes}%`}
          subtitle="Meta: 15%"
          icon={Target}
          color={colors.primary}
        />
      </div>

      {/* Gráficos */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Performance Mensal */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 text-left">Performance de Recuperação</h3>
          <ResponsiveContainer width="100%" height={300}>
            <AreaChart data={performanceData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="mes" />
              <YAxis />
              <Tooltip 
                formatter={(value, name) => [
                  name === 'valor_recuperado' 
                    ? `R$ ${value.toLocaleString('pt-BR')}` 
                    : value,
                  name === 'valor_recuperado' ? 'Valor Recuperado' : 'Acordos'
                ]}
              />
              <Area 
                type="monotone" 
                dataKey="valor_recuperado" 
                stroke={colors.primary} 
                fill={`${colors.primary}30`}
              />
            </AreaChart>
          </ResponsiveContainer>
        </div>

        {/* Tipos de Acordo */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 text-left">Tipos de Acordo</h3>
          <ResponsiveContainer width="100%" height={300}>
            <PieChart>
              <Pie
                data={tiposAcordo}
                cx="50%"
                cy="50%"
                innerRadius={60}
                outerRadius={100}
                paddingAngle={5}
                dataKey="value"
              >
                {tiposAcordo.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={entry.color} />
                ))}
              </Pie>
              <Tooltip formatter={(value) => [`${value}%`, 'Percentual']} />
            </PieChart>
          </ResponsiveContainer>
          <div className="mt-4 space-y-2">
            {tiposAcordo.map((item, index) => (
              <div key={index} className="flex items-center justify-between">
                <div className="flex items-center">
                  <div 
                    className="w-3 h-3 rounded-full mr-2" 
                    style={{ backgroundColor: item.color }}
                  ></div>
                  <span className="text-sm text-gray-600 text-left">{item.name}</span>
                </div>
                <span className="text-sm font-medium text-gray-900">{item.value}%</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Top Inadimplentes */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="px-6 py-4 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900 text-left">Maiores Inadimplências</h3>
        </div>
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Cliente
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Valor
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Dias em Atraso
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Ações
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {dashboardData.top_inadimplentes.map((cliente, index) => (
                <tr key={index} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="text-sm font-medium text-gray-900">{cliente.nome_devedor}</div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="text-sm text-gray-900">
                      R$ {cliente.saldo_atual.toLocaleString('pt-BR', { minimumFractionDigits: 2 })}
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                      cliente.dias_em_atraso > 60 
                        ? 'bg-red-100 text-red-800'
                        : cliente.dias_em_atraso > 30
                        ? 'bg-yellow-100 text-yellow-800'
                        : 'bg-green-100 text-green-800'
                    }`}>
                      {cliente.dias_em_atraso} dias
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <button className="text-blue-600 hover:text-blue-900 mr-3">
                      <MessageSquare className="h-4 w-4" />
                    </button>
                    <button className="text-green-600 hover:text-green-900 mr-3">
                      <Phone className="h-4 w-4" />
                    </button>
                    <button className="text-gray-600 hover:text-gray-900">
                      <Eye className="h-4 w-4" />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );

  const Sidebar = () => (
    <div className="w-64 bg-gray-900 text-white h-screen fixed left-0 top-0 overflow-y-auto">
      {/* Logo */}
      <div className="p-6 border-b border-gray-700">
        <div className="flex items-center">
          <div className="w-10 h-10 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-lg flex items-center justify-center">
            <span className="text-black font-bold text-lg">VJ</span>
          </div>
          <div className="ml-3">
            <h1 className="text-lg font-bold">Villa Joias</h1>
            <p className="text-sm text-gray-400">Cobrança</p>
          </div>
        </div>
      </div>

      {/* Menu */}
      <nav className="mt-6">
        <div className="px-4 space-y-2">
          <button
            onClick={() => setActiveTab('dashboard')}
            className={`w-full flex items-center px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
              activeTab === 'dashboard'
                ? 'bg-yellow-600 text-white'
                : 'text-gray-300 hover:bg-gray-800 hover:text-white'
            }`}
          >
            <BarChart className="mr-3 h-5 w-5" />
            Dashboard
          </button>

          <button
            onClick={() => setActiveTab('inadimplentes')}
            className={`w-full flex items-center px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
              activeTab === 'inadimplentes'
                ? 'bg-yellow-600 text-white'
                : 'text-gray-300 hover:bg-gray-800 hover:text-white'
            }`}
          >
            <Users className="mr-3 h-5 w-5" />
            Inadimplentes
          </button>

          <button
            onClick={() => setActiveTab('acordos')}
            className={`w-full flex items-center px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
              activeTab === 'acordos'
                ? 'bg-yellow-600 text-white'
                : 'text-gray-300 hover:bg-gray-800 hover:text-white'
            }`}
          >
            <CheckCircle className="mr-3 h-5 w-5" />
            Acordos
          </button>

          <button
            onClick={() => setActiveTab('relatorios')}
            className={`w-full flex items-center px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
              activeTab === 'relatorios'
                ? 'bg-yellow-600 text-white'
                : 'text-gray-300 hover:bg-gray-800 hover:text-white'
            }`}
          >
            <FileText className="mr-3 h-5 w-5" />
            Relatórios
          </button>

          <button
            onClick={() => setActiveTab('configuracoes')}
            className={`w-full flex items-center px-4 py-3 text-sm font-medium rounded-lg transition-colors ${
              activeTab === 'configuracoes'
                ? 'bg-yellow-600 text-white'
                : 'text-gray-300 hover:bg-gray-800 hover:text-white'
            }`}
          >
            <Settings className="mr-3 h-5 w-5" />
            Configurações
          </button>
        </div>
      </nav>

      {/* Vi Assistant */}
      <div className="absolute bottom-0 left-0 right-0 p-4 border-t border-gray-700">
        <div className="bg-gradient-to-r from-yellow-600 to-yellow-500 rounded-lg p-3">
          <div className="flex items-center">
            <div className="w-8 h-8 bg-white rounded-full flex items-center justify-center">
              <MessageSquare className="h-4 w-4 text-yellow-600" />
            </div>
            <div className="ml-2">
              <p className="text-sm font-medium text-white">Vi Assistant</p>
              <p className="text-xs text-yellow-100">Online</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );

  const Header = () => (
    <div className="bg-white shadow-sm border-b border-gray-200 px-6 py-4">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">
            {activeTab === 'dashboard' && 'Dashboard de Cobrança'}
            {activeTab === 'inadimplentes' && 'Gestão de Inadimplentes'}
            {activeTab === 'acordos' && 'Controle de Acordos'}
            {activeTab === 'relatorios' && 'Relatórios de Performance'}
            {activeTab === 'configuracoes' && 'Configurações de Desconto'}
          </h1>
          <p className="text-sm text-gray-600 mt-1">
            Sistema de recuperação de crédito Villa Joias
          </p>
        </div>
        <div className="flex items-center space-x-4">
          <button className="bg-yellow-600 text-white px-4 py-2 rounded-lg hover:bg-yellow-700 transition-colors flex items-center">
            <Plus className="h-4 w-4 mr-2" />
            Nova Cobrança
          </button>
          <div className="w-8 h-8 bg-gray-300 rounded-full"></div>
        </div>
      </div>
    </div>
  );

  return (
    <div className="min-h-screen bg-gray-50">
      <Sidebar />
      <div className="ml-64">
        <Header />
        <main className="p-6">
          {activeTab === 'dashboard' && <Dashboard />}
          {activeTab === 'inadimplentes' && (
            <div className="text-center py-12">
              <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Módulo de Inadimplentes</h3>
              <p className="text-gray-600">Em desenvolvimento...</p>
            </div>
          )}
          {activeTab === 'acordos' && (
            <div className="text-center py-12">
              <CheckCircle className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Controle de Acordos</h3>
              <p className="text-gray-600">Em desenvolvimento...</p>
            </div>
          )}
          {activeTab === 'relatorios' && (
            <div className="text-center py-12">
              <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Relatórios de Performance</h3>
              <p className="text-gray-600">Em desenvolvimento...</p>
            </div>
          )}
          {activeTab === 'configuracoes' && (
            <div className="text-center py-12">
              <Settings className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Configurações</h3>
              <p className="text-gray-600">Em desenvolvimento...</p>
            </div>
          )}
        </main>
      </div>
    </div>
  );
};

export default CobrancaModule;

