import React, { useState, useEffect } from 'react';
import { 
  BarChart3, 
  TrendingUp, 
  DollarSign, 
  Users, 
  AlertTriangle,
  Calendar,
  PieChart,
  ArrowUpRight,
  ArrowDownRight,
  Target,
  Zap,
  Clock
} from 'lucide-react';
import { AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart as RechartsPieChart, Cell } from 'recharts';

const Dashboard = () => {
  const [loading, setLoading] = useState(true);
  const [metricas, setMetricas] = useState({});

  // Dados simulados para demonstração
  const dadosSimulados = {
    saldoTotal: 485750.80,
    faturamentoMes: 145230.80,
    crescimentoMes: 12.5,
    despesasMes: 89450.25,
    saldoLiquido: 55780.55,
    inadimplencia: 125430.50,
    acordosMes: 23,
    taxaConversao: 14.7,
    representantesAtivos: 28,
    clientesAtivos: 699,
    proximosVencimentos: 15,
    alertasCriticos: 3
  };

  // Dados para gráfico de fluxo de caixa (últimos 6 meses)
  const dadosFluxoCaixa = [
    { mes: 'Jan', entradas: 120000, saidas: 85000, liquido: 35000 },
    { mes: 'Fev', entradas: 135000, saidas: 92000, liquido: 43000 },
    { mes: 'Mar', entradas: 128000, saidas: 88000, liquido: 40000 },
    { mes: 'Abr', entradas: 142000, saidas: 95000, liquido: 47000 },
    { mes: 'Mai', entradas: 138000, saidas: 89000, liquido: 49000 },
    { mes: 'Jun', entradas: 145000, saidas: 89000, liquido: 56000 }
  ];

  // Dados para gráfico de representantes
  const dadosRepresentantes = [
    { nome: 'João Silva', valor: 28500, cor: '#D4AF37' },
    { nome: 'Maria Santos', valor: 24200, cor: '#B8860B' },
    { nome: 'Pedro Costa', valor: 21800, cor: '#DAA520' },
    { nome: 'Ana Lima', valor: 19400, cor: '#F4A460' },
    { nome: 'Outros', valor: 51330, cor: '#DDD' }
  ];

  useEffect(() => {
    // Simular carregamento
    setTimeout(() => {
      setMetricas(dadosSimulados);
      setLoading(false);
    }, 1500);
  }, []);

  const formatarMoeda = (valor) => {
    return new Intl.NumberFormat('pt-BR', {
      style: 'currency',
      currency: 'BRL'
    }).format(valor);
  };

  const formatarPorcentagem = (valor) => {
    return `${valor > 0 ? '+' : ''}${valor.toFixed(1)}%`;
  };

  const CardMetrica = ({ titulo, valor, icone: Icone, tendencia, corTendencia, descricao, loading = false }) => {
    if (loading) {
      return (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="animate-pulse">
            <div className="h-4 bg-gray-200 rounded w-1/2 mb-4"></div>
            <div className="h-8 bg-gray-200 rounded w-3/4 mb-2"></div>
            <div className="h-3 bg-gray-200 rounded w-1/3"></div>
          </div>
        </div>
      );
    }

    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center">
            <div className="p-2 bg-yellow-100 rounded-lg">
              <Icone className="h-5 w-5 text-yellow-600" />
            </div>
            <h3 className="ml-3 text-sm font-medium text-gray-600">{titulo}</h3>
          </div>
          {tendencia && (
            <div className={`flex items-center text-sm ${corTendencia}`}>
              {tendencia > 0 ? (
                <ArrowUpRight className="h-4 w-4 mr-1" />
              ) : (
                <ArrowDownRight className="h-4 w-4 mr-1" />
              )}
              {formatarPorcentagem(tendencia)}
            </div>
          )}
        </div>
        
        <div className="mb-2">
          <p className="text-2xl font-bold text-gray-900">{valor}</p>
        </div>
        
        {descricao && (
          <p className="text-sm text-gray-500">{descricao}</p>
        )}
      </div>
    );
  };

  const DicasManus = () => {
    const dicas = [
      {
        tipo: 'oportunidade',
        icone: <TrendingUp className="h-5 w-5 text-green-600" />,
        titulo: 'Oportunidade de Crescimento',
        descricao: 'Representante João Silva está 35% acima da média. Replique sua estratégia com outros representantes.',
        acao: 'Analisar estratégia'
      },
      {
        tipo: 'alerta',
        icone: <AlertTriangle className="h-5 w-5 text-yellow-600" />,
        titulo: 'Atenção: Despesas Operacionais',
        descricao: 'Categoria "Energia" aumentou 28% este mês. Revisar contratos e consumo.',
        acao: 'Revisar gastos'
      },
      {
        tipo: 'sugestao',
        icone: <Zap className="h-5 w-5 text-blue-600" />,
        titulo: 'Otimização de Caixa',
        descricao: 'R$ 85.000 parados no caixa operacional podem render R$ 850/mês em CDB.',
        acao: 'Transferir para reserva'
      },
      {
        tipo: 'meta',
        icone: <Target className="h-5 w-5 text-purple-600" />,
        titulo: 'Meta do Mês',
        descricao: 'Faltam R$ 4.769 para bater a meta de R$ 150.000. Projeção: atingir até dia 28.',
        acao: 'Acompanhar progresso'
      }
    ];

    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <div className="flex items-center mb-4">
          <div className="p-2 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-lg">
            <span className="text-black font-bold text-sm">M</span>
          </div>
          <h3 className="ml-3 text-lg font-semibold text-gray-900">Insights do Manus</h3>
          <span className="ml-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">Online</span>
        </div>

        <div className="space-y-4">
          {dicas.map((dica, index) => (
            <div key={index} className="border border-gray-100 rounded-lg p-4 hover:bg-gray-50 transition-colors">
              <div className="flex items-start">
                <div className="flex-shrink-0 mr-3 mt-1">
                  {dica.icone}
                </div>
                <div className="flex-1">
                  <h4 className="text-sm font-medium text-gray-900 mb-1">{dica.titulo}</h4>
                  <p className="text-sm text-gray-600 mb-2">{dica.descricao}</p>
                  <button className="text-xs text-yellow-600 hover:text-yellow-700 font-medium">
                    {dica.acao} →
                  </button>
                </div>
              </div>
            </div>
          ))}
        </div>

        <div className="mt-4 pt-4 border-t border-gray-100">
          <p className="text-xs text-gray-500 text-center">
            💡 Dicas atualizadas em tempo real baseadas nos seus dados
          </p>
        </div>
      </div>
    );
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="mb-6">
          <div className="animate-pulse">
            <div className="h-8 bg-gray-200 rounded w-1/3 mb-2"></div>
            <div className="h-4 bg-gray-200 rounded w-1/2"></div>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {[1, 2, 3, 4].map((i) => (
            <CardMetrica key={i} loading={true} />
          ))}
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="animate-pulse">
                <div className="h-6 bg-gray-200 rounded w-1/4 mb-4"></div>
                <div className="h-64 bg-gray-200 rounded"></div>
              </div>
            </div>
          </div>
          <div>
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <div className="animate-pulse">
                <div className="h-6 bg-gray-200 rounded w-1/3 mb-4"></div>
                <div className="space-y-3">
                  {[1, 2, 3, 4].map((i) => (
                    <div key={i} className="h-16 bg-gray-200 rounded"></div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900 flex items-center">
          <BarChart3 className="h-8 w-8 text-yellow-500 mr-3" />
          Dashboard Villa Joias
        </h1>
        <p className="text-gray-600 mt-1">Visão geral do seu negócio em tempo real</p>
      </div>

      {/* Cards de Métricas Principais */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <CardMetrica
          titulo="SALDO TOTAL DOS CAIXAS"
          valor={formatarMoeda(metricas.saldoTotal)}
          icone={DollarSign}
          tendencia={8.2}
          corTendencia="text-green-600"
          descricao="Todos os caixas consolidados"
        />
        
        <CardMetrica
          titulo="FATURAMENTO DO MÊS"
          valor={formatarMoeda(metricas.faturamentoMes)}
          icone={TrendingUp}
          tendencia={metricas.crescimentoMes}
          corTendencia="text-green-600"
          descricao="vs mês anterior"
        />
        
        <CardMetrica
          titulo="INADIMPLÊNCIA TOTAL"
          valor={formatarMoeda(metricas.inadimplencia)}
          icone={AlertTriangle}
          tendencia={-5.3}
          corTendencia="text-green-600"
          descricao={`${metricas.acordosMes} acordos este mês`}
        />
        
        <CardMetrica
          titulo="REPRESENTANTES ATIVOS"
          valor={metricas.representantesAtivos}
          icone={Users}
          tendencia={3.7}
          corTendencia="text-green-600"
          descricao={`${metricas.clientesAtivos} clientes ativos`}
        />
      </div>

      {/* Cards Secundários */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <CardMetrica
          titulo="SALDO LÍQUIDO MENSAL"
          valor={formatarMoeda(metricas.saldoLiquido)}
          icone={PieChart}
          tendencia={15.8}
          corTendencia="text-green-600"
          descricao="Entradas - Saídas"
        />
        
        <CardMetrica
          titulo="TAXA DE CONVERSÃO"
          valor={`${metricas.taxaConversao}%`}
          icone={Target}
          tendencia={2.1}
          corTendencia="text-green-600"
          descricao="Inadimplentes → Acordos"
        />
        
        <CardMetrica
          titulo="PRÓXIMOS VENCIMENTOS"
          valor={metricas.proximosVencimentos}
          icone={Clock}
          descricao="Parcelas nos próximos 7 dias"
        />
      </div>

      {/* Gráficos e Insights */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Gráfico de Fluxo de Caixa */}
        <div className="lg:col-span-2">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-lg font-semibold text-gray-900">Evolução do Fluxo de Caixa</h3>
              <div className="flex items-center space-x-4 text-sm">
                <div className="flex items-center">
                  <div className="w-3 h-3 bg-green-500 rounded-full mr-2"></div>
                  <span className="text-gray-600">Entradas</span>
                </div>
                <div className="flex items-center">
                  <div className="w-3 h-3 bg-red-500 rounded-full mr-2"></div>
                  <span className="text-gray-600">Saídas</span>
                </div>
                <div className="flex items-center">
                  <div className="w-3 h-3 bg-yellow-500 rounded-full mr-2"></div>
                  <span className="text-gray-600">Líquido</span>
                </div>
              </div>
            </div>
            
            <ResponsiveContainer width="100%" height={300}>
              <AreaChart data={dadosFluxoCaixa}>
                <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                <XAxis dataKey="mes" stroke="#666" />
                <YAxis stroke="#666" tickFormatter={(value) => `${value/1000}k`} />
                <Tooltip 
                  formatter={(value) => [formatarMoeda(value), '']}
                  labelStyle={{ color: '#333' }}
                />
                <Area 
                  type="monotone" 
                  dataKey="entradas" 
                  stackId="1" 
                  stroke="#10B981" 
                  fill="#10B981" 
                  fillOpacity={0.6}
                />
                <Area 
                  type="monotone" 
                  dataKey="saidas" 
                  stackId="2" 
                  stroke="#EF4444" 
                  fill="#EF4444" 
                  fillOpacity={0.6}
                />
                <Area 
                  type="monotone" 
                  dataKey="liquido" 
                  stackId="3" 
                  stroke="#D4AF37" 
                  fill="#D4AF37" 
                  fillOpacity={0.8}
                />
              </AreaChart>
            </ResponsiveContainer>
          </div>
        </div>

        {/* Insights do Manus */}
        <div>
          <DicasManus />
        </div>
      </div>

      {/* Performance de Representantes */}
      <div className="mt-8">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Top Representantes do Mês</h3>
            <button className="text-yellow-600 hover:text-yellow-700 text-sm font-medium">
              Ver todos →
            </button>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Lista de Representantes */}
            <div className="space-y-4">
              {dadosRepresentantes.slice(0, 4).map((rep, index) => (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-8 h-8 bg-yellow-500 rounded-full flex items-center justify-center text-white font-bold text-sm mr-3">
                      {index + 1}
                    </div>
                    <div>
                      <p className="font-medium text-gray-900">{rep.nome}</p>
                      <p className="text-sm text-gray-500">{formatarMoeda(rep.valor)}</p>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="text-sm font-medium text-gray-900">
                      {((rep.valor / dadosRepresentantes.reduce((acc, r) => acc + r.valor, 0)) * 100).toFixed(1)}%
                    </p>
                    <p className="text-xs text-gray-500">do total</p>
                  </div>
                </div>
              ))}
            </div>

            {/* Gráfico de Pizza */}
            <div className="flex items-center justify-center">
              <ResponsiveContainer width="100%" height={200}>
                <RechartsPieChart>
                  <Tooltip formatter={(value) => formatarMoeda(value)} />
                  <RechartsPieChart data={dadosRepresentantes}>
                    {dadosRepresentantes.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={entry.cor} />
                    ))}
                  </RechartsPieChart>
                </RechartsPieChart>
              </ResponsiveContainer>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Dashboard;

