#!/bin/bash

# Script de Verificação DNS - Sistema Villa Joias
# Versão: 1.0
# Uso: ./check_dns.sh seudominio.com.br

DOMAIN=$1

if [ -z "$DOMAIN" ]; then
    echo "❌ Erro: Você deve especificar o domínio"
    echo "Uso: ./check_dns.sh seudominio.com.br"
    exit 1
fi

echo "🔍 VERIFICAÇÃO DNS - $DOMAIN"
echo "============================="
echo ""

# Cores
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Obter IP do servidor atual
SERVER_IP=$(curl -s ifconfig.me)
echo -e "${BLUE}🖥️  IP do Servidor: $SERVER_IP${NC}"
echo ""

# Verificar resolução DNS
echo -e "${BLUE}🌐 VERIFICAÇÃO DNS${NC}"
echo "=================="

# Verificar domínio principal
DOMAIN_IP=$(dig +short $DOMAIN)
if [ -n "$DOMAIN_IP" ]; then
    if [ "$DOMAIN_IP" = "$SERVER_IP" ]; then
        echo -e "${GREEN}✅ $DOMAIN → $DOMAIN_IP (CORRETO)${NC}"
    else
        echo -e "${RED}❌ $DOMAIN → $DOMAIN_IP (INCORRETO - deveria ser $SERVER_IP)${NC}"
    fi
else
    echo -e "${RED}❌ $DOMAIN não resolve${NC}"
fi

# Verificar www
WWW_IP=$(dig +short www.$DOMAIN)
if [ -n "$WWW_IP" ]; then
    if [ "$WWW_IP" = "$SERVER_IP" ]; then
        echo -e "${GREEN}✅ www.$DOMAIN → $WWW_IP (CORRETO)${NC}"
    else
        echo -e "${RED}❌ www.$DOMAIN → $WWW_IP (INCORRETO - deveria ser $SERVER_IP)${NC}"
    fi
else
    echo -e "${RED}❌ www.$DOMAIN não resolve${NC}"
fi

echo ""

# Verificar propagação em diferentes DNS servers
echo -e "${BLUE}🌍 PROPAGAÇÃO MUNDIAL${NC}"
echo "===================="

DNS_SERVERS=("8.8.8.8:Google" "1.1.1.1:Cloudflare" "208.67.222.222:OpenDNS" "8.26.56.26:Comodo")

for dns_info in "${DNS_SERVERS[@]}"; do
    IFS=':' read -r dns_ip dns_name <<< "$dns_info"
    result=$(dig @$dns_ip +short $DOMAIN 2>/dev/null)
    
    if [ -n "$result" ]; then
        if [ "$result" = "$SERVER_IP" ]; then
            echo -e "${GREEN}✅ $dns_name ($dns_ip): $result${NC}"
        else
            echo -e "${YELLOW}⚠️  $dns_name ($dns_ip): $result (diferente)${NC}"
        fi
    else
        echo -e "${RED}❌ $dns_name ($dns_ip): Não resolve${NC}"
    fi
done

echo ""

# Verificar conectividade HTTP/HTTPS
echo -e "${BLUE}🌐 CONECTIVIDADE WEB${NC}"
echo "==================="

# Teste HTTP
HTTP_CODE=$(curl -s -o /dev/null -w "%{http_code}" "http://$DOMAIN" 2>/dev/null)
if [ "$HTTP_CODE" = "200" ] || [ "$HTTP_CODE" = "301" ] || [ "$HTTP_CODE" = "302" ]; then
    echo -e "${GREEN}✅ HTTP ($HTTP_CODE): http://$DOMAIN${NC}"
else
    echo -e "${RED}❌ HTTP ($HTTP_CODE): http://$DOMAIN${NC}"
fi

# Teste HTTPS
HTTPS_CODE=$(curl -s -o /dev/null -w "%{http_code}" "https://$DOMAIN" 2>/dev/null)
if [ "$HTTPS_CODE" = "200" ]; then
    echo -e "${GREEN}✅ HTTPS ($HTTPS_CODE): https://$DOMAIN${NC}"
else
    echo -e "${RED}❌ HTTPS ($HTTPS_CODE): https://$DOMAIN${NC}"
fi

# Teste WWW HTTP
WWW_HTTP_CODE=$(curl -s -o /dev/null -w "%{http_code}" "http://www.$DOMAIN" 2>/dev/null)
if [ "$WWW_HTTP_CODE" = "200" ] || [ "$WWW_HTTP_CODE" = "301" ] || [ "$WWW_HTTP_CODE" = "302" ]; then
    echo -e "${GREEN}✅ WWW HTTP ($WWW_HTTP_CODE): http://www.$DOMAIN${NC}"
else
    echo -e "${RED}❌ WWW HTTP ($WWW_HTTP_CODE): http://www.$DOMAIN${NC}"
fi

# Teste WWW HTTPS
WWW_HTTPS_CODE=$(curl -s -o /dev/null -w "%{http_code}" "https://www.$DOMAIN" 2>/dev/null)
if [ "$WWW_HTTPS_CODE" = "200" ]; then
    echo -e "${GREEN}✅ WWW HTTPS ($WWW_HTTPS_CODE): https://www.$DOMAIN${NC}"
else
    echo -e "${RED}❌ WWW HTTPS ($WWW_HTTPS_CODE): https://www.$DOMAIN${NC}"
fi

echo ""

# Verificar certificado SSL
echo -e "${BLUE}🔒 CERTIFICADO SSL${NC}"
echo "=================="

if command -v openssl &> /dev/null; then
    SSL_INFO=$(echo | openssl s_client -servername $DOMAIN -connect $DOMAIN:443 2>/dev/null | openssl x509 -noout -dates 2>/dev/null)
    
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✅ Certificado SSL válido${NC}"
        
        # Extrair data de expiração
        EXPIRY=$(echo "$SSL_INFO" | grep "notAfter" | cut -d= -f2)
        if [ -n "$EXPIRY" ]; then
            EXPIRY_EPOCH=$(date -d "$EXPIRY" +%s 2>/dev/null)
            NOW_EPOCH=$(date +%s)
            
            if [ -n "$EXPIRY_EPOCH" ]; then
                DAYS_LEFT=$(( ($EXPIRY_EPOCH - $NOW_EPOCH) / 86400 ))
                
                if [ $DAYS_LEFT -gt 30 ]; then
                    echo -e "${GREEN}✅ Expira em: $DAYS_LEFT dias${NC}"
                elif [ $DAYS_LEFT -gt 7 ]; then
                    echo -e "${YELLOW}⚠️  Expira em: $DAYS_LEFT dias${NC}"
                else
                    echo -e "${RED}❌ Expira em: $DAYS_LEFT dias (URGENTE)${NC}"
                fi
            fi
        fi
    else
        echo -e "${RED}❌ Certificado SSL inválido ou não encontrado${NC}"
    fi
else
    echo -e "${YELLOW}⚠️  OpenSSL não disponível para verificar certificado${NC}"
fi

echo ""

# Verificar redirecionamento HTTPS
echo -e "${BLUE}🔄 REDIRECIONAMENTO HTTPS${NC}"
echo "=========================="

REDIRECT_TEST=$(curl -s -I "http://$DOMAIN" | grep -i "location: https")
if [ -n "$REDIRECT_TEST" ]; then
    echo -e "${GREEN}✅ HTTP redireciona para HTTPS${NC}"
else
    echo -e "${YELLOW}⚠️  HTTP não redireciona para HTTPS${NC}"
fi

echo ""

# Resumo final
echo -e "${BLUE}📋 RESUMO${NC}"
echo "========="

ISSUES=0

# Verificar problemas
if [ "$DOMAIN_IP" != "$SERVER_IP" ]; then
    echo -e "${RED}❌ DNS do domínio principal incorreto${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ "$WWW_IP" != "$SERVER_IP" ]; then
    echo -e "${RED}❌ DNS do www incorreto${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ "$HTTP_CODE" != "200" ] && [ "$HTTP_CODE" != "301" ] && [ "$HTTP_CODE" != "302" ]; then
    echo -e "${RED}❌ Site não carrega via HTTP${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ "$HTTPS_CODE" != "200" ]; then
    echo -e "${RED}❌ Site não carrega via HTTPS${NC}"
    ISSUES=$((ISSUES + 1))
fi

if [ $ISSUES -eq 0 ]; then
    echo -e "${GREEN}🎉 TUDO FUNCIONANDO PERFEITAMENTE!${NC}"
    echo ""
    echo -e "${GREEN}✅ Seu domínio está configurado corretamente${NC}"
    echo -e "${GREEN}✅ DNS propagado e funcionando${NC}"
    echo -e "${GREEN}✅ Site acessível via HTTP e HTTPS${NC}"
    echo -e "${GREEN}✅ SSL configurado e válido${NC}"
else
    echo -e "${RED}⚠️  ENCONTRADOS $ISSUES PROBLEMAS${NC}"
    echo ""
    echo -e "${YELLOW}📋 PRÓXIMOS PASSOS:${NC}"
    
    if [ "$DOMAIN_IP" != "$SERVER_IP" ]; then
        echo "1. Verifique configuração DNS no GoDaddy"
        echo "   - Registro A para @ deve apontar para $SERVER_IP"
    fi
    
    if [ "$WWW_IP" != "$SERVER_IP" ]; then
        echo "2. Verifique configuração DNS para www no GoDaddy"
        echo "   - Registro A para www deve apontar para $SERVER_IP"
    fi
    
    if [ "$HTTP_CODE" != "200" ] && [ "$HTTP_CODE" != "301" ] && [ "$HTTP_CODE" != "302" ]; then
        echo "3. Verifique configuração do Apache no VPS"
        echo "   - Execute: systemctl status apache2"
        echo "   - Execute: apache2ctl configtest"
    fi
    
    if [ "$HTTPS_CODE" != "200" ]; then
        echo "4. Verifique configuração SSL"
        echo "   - Execute: certbot certificates"
        echo "   - Execute: ./configure_ssl.sh $DOMAIN seu@email.com"
    fi
fi

echo ""
echo "🔗 LINKS ÚTEIS:"
echo "- Verificar propagação: https://www.whatsmydns.net/#A/$DOMAIN"
echo "- Testar SSL: https://www.ssllabs.com/ssltest/analyze.html?d=$DOMAIN"
echo "- Testar velocidade: https://gtmetrix.com/"
echo ""
echo "📞 SUPORTE:"
echo "- GoDaddy: 0800 606 4656"
echo "- Hostgator: 0800 600 2000"
echo "- Villa Joias: suporte@villajoias.com.br"

