#!/bin/bash

# Script de Configuração de Virtual Host - Sistema Villa Joias
# Versão: 2.0 VPS Edition
# Uso: ./configure_domain.sh seudominio.com.br

DOMAIN=$1

if [ -z "$DOMAIN" ]; then
    echo "❌ Erro: Você deve especificar o domínio"
    echo "Uso: ./configure_domain.sh seudominio.com.br"
    exit 1
fi

if [ "$EUID" -ne 0 ]; then
    echo "❌ Este script deve ser executado como root (sudo)"
    exit 1
fi

echo "🌐 Configurando Virtual Host para: $DOMAIN"
echo "============================================"

# Desabilitar site padrão
a2dissite 000-default

# Criar configuração do Virtual Host
cat > /etc/apache2/sites-available/villa_joias.conf << EOF
<VirtualHost *:80>
    ServerName $DOMAIN
    ServerAlias www.$DOMAIN
    DocumentRoot /var/www/html/villa_joias
    
    <Directory /var/www/html/villa_joias>
        AllowOverride All
        Require all granted
        Options -Indexes
    </Directory>
    
    # Logs
    ErrorLog \${APACHE_LOG_DIR}/villa_joias_error.log
    CustomLog \${APACHE_LOG_DIR}/villa_joias_access.log combined
    
    # Segurança adicional
    <Files "config.php">
        Require all denied
    </Files>
    
    <Files "database.sql">
        Require all denied
    </Files>
    
    <Files "*.log">
        Require all denied
    </Files>
</VirtualHost>
EOF

# Habilitar site
a2ensite villa_joias

# Criar arquivo .htaccess
cat > /var/www/html/villa_joias/.htaccess << 'EOF'
# Compressão GZIP
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
    AddOutputFilterByType DEFLATE application/json
</IfModule>

# Cache de arquivos estáticos
<IfModule mod_expires.c>
    ExpiresActive on
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
    ExpiresByType image/ico "access plus 1 month"
    ExpiresByType image/icon "access plus 1 month"
    ExpiresByType text/plain "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType font/woff "access plus 1 month"
    ExpiresByType font/woff2 "access plus 1 month"
</IfModule>

# Headers de segurança
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
    Header always set Permissions-Policy "geolocation=(), microphone=(), camera=()"
</IfModule>

# Segurança de arquivos
<Files "config.php">
    Require all denied
</Files>

<Files "database.sql">
    Require all denied
</Files>

<Files "install.php">
    <RequireAll>
        Require ip 127.0.0.1
        Require ip ::1
    </RequireAll>
</Files>

<FilesMatch "\.(log|bak|backup|old)$">
    Require all denied
</FilesMatch>

# URL amigáveis
RewriteEngine On

# Forçar HTTPS (descomente após configurar SSL)
# RewriteCond %{HTTPS} off
# RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Redirecionar www para não-www (ou vice-versa)
RewriteCond %{HTTP_HOST} ^www\.(.*)$ [NC]
RewriteRule ^(.*)$ http://%1/$1 [R=301,L]

# Rotas da aplicação
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^api/(.*)$ api/index.php [QSA,L]

RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]
EOF

# Testar configuração
echo "🔍 Testando configuração do Apache..."
if apache2ctl configtest; then
    echo "✅ Configuração do Apache OK"
    
    # Reiniciar Apache
    systemctl restart apache2
    
    if systemctl is-active --quiet apache2; then
        echo "✅ Apache reiniciado com sucesso"
        echo ""
        echo "🎉 VIRTUAL HOST CONFIGURADO!"
        echo "============================"
        echo ""
        echo "🌐 Domínio: $DOMAIN"
        echo "📁 Pasta: /var/www/html/villa_joias"
        echo "📋 Próximos passos:"
        echo "   1. Faça upload dos arquivos do sistema"
        echo "   2. Acesse: http://$DOMAIN/install.php"
        echo "   3. Configure SSL: certbot --apache -d $DOMAIN -d www.$DOMAIN"
        echo ""
    else
        echo "❌ Erro ao reiniciar Apache"
        systemctl status apache2
    fi
else
    echo "❌ Erro na configuração do Apache"
    echo "Verifique os logs: tail -f /var/log/apache2/error.log"
fi

