<?php
/**
 * Instalador Web - Sistema Villa Joias
 * Versão cPanel VPS Edition
 * Desenvolvido por: Manus AI - CIO Villa Joias
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configurações
define('INSTALL_VERSION', '3.0-cpanel');
define('MIN_PHP_VERSION', '8.0');
define('MIN_MYSQL_VERSION', '5.7');

// Verificar se já foi instalado
if (file_exists('config.php') && !isset($_GET['force'])) {
    header('Location: index.php');
    exit;
}

// Funções auxiliares
function checkPHPVersion() {
    return version_compare(PHP_VERSION, MIN_PHP_VERSION, '>=');
}

function checkPHPExtensions() {
    $required = ['mysqli', 'pdo', 'pdo_mysql', 'json', 'curl', 'mbstring', 'openssl'];
    $missing = [];
    
    foreach ($required as $ext) {
        if (!extension_loaded($ext)) {
            $missing[] = $ext;
        }
    }
    
    return $missing;
}

function checkWritePermissions() {
    $dirs = ['.', 'api', 'api/config', 'uploads', 'backups', 'logs'];
    $issues = [];
    
    foreach ($dirs as $dir) {
        if (!is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }
        
        if (!is_writable($dir)) {
            $issues[] = $dir;
        }
    }
    
    return $issues;
}

function testDatabaseConnection($host, $username, $password, $database) {
    try {
        $pdo = new PDO("mysql:host=$host;charset=utf8mb4", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Verificar se banco existe, se não, criar
        $stmt = $pdo->prepare("CREATE DATABASE IF NOT EXISTS `$database` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $stmt->execute();
        
        // Conectar no banco específico
        $pdo = new PDO("mysql:host=$host;dbname=$database;charset=utf8mb4", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        return ['success' => true, 'pdo' => $pdo];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function createTables($pdo) {
    $sql = file_get_contents('database.sql');
    if (!$sql) {
        return ['success' => false, 'error' => 'Arquivo database.sql não encontrado'];
    }
    
    try {
        // Dividir SQL em comandos individuais
        $commands = array_filter(array_map('trim', explode(';', $sql)));
        $created = 0;
        
        foreach ($commands as $command) {
            if (!empty($command)) {
                $pdo->exec($command);
                $created++;
            }
        }
        
        return ['success' => true, 'tables' => $created];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function createConfigFile($dbConfig, $adminConfig, $generalConfig) {
    $config = "<?php\n";
    $config .= "/**\n";
    $config .= " * Configuração do Sistema Villa Joias\n";
    $config .= " * Gerado automaticamente em " . date('d/m/Y H:i:s') . "\n";
    $config .= " */\n\n";
    
    $config .= "// Configurações do Banco de Dados\n";
    $config .= "define('DB_HOST', '" . addslashes($dbConfig['host']) . "');\n";
    $config .= "define('DB_NAME', '" . addslashes($dbConfig['database']) . "');\n";
    $config .= "define('DB_USER', '" . addslashes($dbConfig['username']) . "');\n";
    $config .= "define('DB_PASS', '" . addslashes($dbConfig['password']) . "');\n";
    $config .= "define('DB_PORT', '" . addslashes($dbConfig['port']) . "');\n\n";
    
    $config .= "// Configurações Gerais\n";
    $config .= "define('SITE_URL', '" . addslashes($generalConfig['site_url']) . "');\n";
    $config .= "define('SITE_NAME', '" . addslashes($generalConfig['company_name']) . "');\n";
    $config .= "define('COMPANY_CNPJ', '" . addslashes($generalConfig['cnpj']) . "');\n";
    $config .= "define('COMPANY_PHONE', '" . addslashes($generalConfig['phone']) . "');\n";
    $config .= "define('COMPANY_EMAIL', '" . addslashes($generalConfig['email']) . "');\n";
    $config .= "define('COMPANY_ADDRESS', '" . addslashes($generalConfig['address']) . "');\n";
    $config .= "define('TIMEZONE', '" . addslashes($generalConfig['timezone']) . "');\n";
    $config .= "define('CURRENCY', '" . addslashes($generalConfig['currency']) . "');\n\n";
    
    $config .= "// Configurações de Segurança\n";
    $config .= "define('JWT_SECRET', '" . bin2hex(random_bytes(32)) . "');\n";
    $config .= "define('ENCRYPTION_KEY', '" . bin2hex(random_bytes(16)) . "');\n";
    $config .= "define('SESSION_LIFETIME', 3600);\n\n";
    
    $config .= "// Configurações de Backup\n";
    $config .= "define('BACKUP_ENABLED', " . ($generalConfig['backup_enabled'] ? 'true' : 'false') . ");\n";
    $config .= "define('BACKUP_FREQUENCY', 'daily');\n";
    $config .= "define('BACKUP_RETENTION', 7);\n\n";
    
    $config .= "// Configurações de IA\n";
    $config .= "define('AI_ENABLED', " . ($generalConfig['ai_enabled'] ? 'true' : 'false') . ");\n";
    $config .= "define('AI_MODEL', 'manus-v2');\n\n";
    
    $config .= "// Configurações de Módulos\n";
    $config .= "define('MODULE_COBRANCA', " . ($generalConfig['module_cobranca'] ? 'true' : 'false') . ");\n";
    $config .= "define('MODULE_RELATORIOS', " . ($generalConfig['module_relatorios'] ? 'true' : 'false') . ");\n\n";
    
    $config .= "// Configurações de Ambiente\n";
    $config .= "define('ENVIRONMENT', 'production');\n";
    $config .= "define('DEBUG_MODE', false);\n";
    $config .= "define('LOG_LEVEL', 'error');\n\n";
    
    $config .= "// Configuração de Fuso Horário\n";
    $config .= "date_default_timezone_set(TIMEZONE);\n\n";
    
    $config .= "?>";
    
    return file_put_contents('config.php', $config);
}

function createAdminUser($pdo, $adminConfig) {
    try {
        $hashedPassword = password_hash($adminConfig['password'], PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            INSERT INTO usuarios (nome, email, senha, tipo, ativo, criado_em) 
            VALUES (?, ?, ?, 'admin', 1, NOW())
        ");
        
        $stmt->execute([
            $adminConfig['name'],
            $adminConfig['email'],
            $hashedPassword
        ]);
        
        return ['success' => true, 'user_id' => $pdo->lastInsertId()];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Processar formulários
$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$errors = [];
$success = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        case 2: // Configuração do banco
            $dbConfig = [
                'host' => $_POST['db_host'] ?? 'localhost',
                'database' => $_POST['db_name'] ?? '',
                'username' => $_POST['db_user'] ?? '',
                'password' => $_POST['db_pass'] ?? '',
                'port' => $_POST['db_port'] ?? '3306'
            ];
            
            $result = testDatabaseConnection($dbConfig['host'], $dbConfig['username'], $dbConfig['password'], $dbConfig['database']);
            
            if ($result['success']) {
                $_SESSION['db_config'] = $dbConfig;
                $_SESSION['db_connection'] = true;
                $success[] = "Conexão com banco de dados estabelecida com sucesso!";
            } else {
                $errors[] = "Erro na conexão: " . $result['error'];
            }
            break;
            
        case 3: // Criação das tabelas
            if (isset($_SESSION['db_config'])) {
                $dbConfig = $_SESSION['db_config'];
                $result = testDatabaseConnection($dbConfig['host'], $dbConfig['username'], $dbConfig['password'], $dbConfig['database']);
                
                if ($result['success']) {
                    $createResult = createTables($result['pdo']);
                    
                    if ($createResult['success']) {
                        $_SESSION['tables_created'] = true;
                        $success[] = "Tabelas criadas com sucesso! Total: " . $createResult['tables'];
                    } else {
                        $errors[] = "Erro ao criar tabelas: " . $createResult['error'];
                    }
                } else {
                    $errors[] = "Erro na conexão com banco: " . $result['error'];
                }
            } else {
                $errors[] = "Configuração do banco não encontrada. Refaça o passo anterior.";
            }
            break;
            
        case 4: // Configuração do administrador
            $adminConfig = [
                'name' => $_POST['admin_name'] ?? '',
                'email' => $_POST['admin_email'] ?? '',
                'password' => $_POST['admin_password'] ?? '',
                'confirm_password' => $_POST['admin_confirm_password'] ?? ''
            ];
            
            if (empty($adminConfig['name']) || empty($adminConfig['email']) || empty($adminConfig['password'])) {
                $errors[] = "Todos os campos são obrigatórios.";
            } elseif ($adminConfig['password'] !== $adminConfig['confirm_password']) {
                $errors[] = "As senhas não coincidem.";
            } elseif (strlen($adminConfig['password']) < 6) {
                $errors[] = "A senha deve ter pelo menos 6 caracteres.";
            } else {
                if (isset($_SESSION['db_config'])) {
                    $dbConfig = $_SESSION['db_config'];
                    $result = testDatabaseConnection($dbConfig['host'], $dbConfig['username'], $dbConfig['password'], $dbConfig['database']);
                    
                    if ($result['success']) {
                        $userResult = createAdminUser($result['pdo'], $adminConfig);
                        
                        if ($userResult['success']) {
                            $_SESSION['admin_config'] = $adminConfig;
                            $_SESSION['admin_created'] = true;
                            $success[] = "Usuário administrador criado com sucesso!";
                        } else {
                            $errors[] = "Erro ao criar usuário: " . $userResult['error'];
                        }
                    } else {
                        $errors[] = "Erro na conexão com banco: " . $result['error'];
                    }
                } else {
                    $errors[] = "Configuração do banco não encontrada.";
                }
            }
            break;
            
        case 5: // Configurações gerais
            $generalConfig = [
                'company_name' => $_POST['company_name'] ?? 'Villa Joias',
                'cnpj' => $_POST['cnpj'] ?? '',
                'phone' => $_POST['phone'] ?? '',
                'email' => $_POST['email'] ?? '',
                'address' => $_POST['address'] ?? '',
                'timezone' => $_POST['timezone'] ?? 'America/Sao_Paulo',
                'currency' => $_POST['currency'] ?? 'BRL',
                'site_url' => $_POST['site_url'] ?? 'https://' . $_SERVER['HTTP_HOST'],
                'backup_enabled' => isset($_POST['backup_enabled']),
                'ai_enabled' => isset($_POST['ai_enabled']),
                'module_cobranca' => isset($_POST['module_cobranca']),
                'module_relatorios' => isset($_POST['module_relatorios'])
            ];
            
            if (isset($_SESSION['db_config']) && isset($_SESSION['admin_config'])) {
                $configResult = createConfigFile($_SESSION['db_config'], $_SESSION['admin_config'], $generalConfig);
                
                if ($configResult) {
                    $_SESSION['installation_complete'] = true;
                    $success[] = "Configurações salvas com sucesso!";
                    $success[] = "Instalação concluída!";
                } else {
                    $errors[] = "Erro ao criar arquivo de configuração.";
                }
            } else {
                $errors[] = "Configurações anteriores não encontradas.";
            }
            break;
    }
}

// Determinar próximo passo
if (empty($errors) && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step < 5) {
        $step++;
    } elseif ($step == 5 && isset($_SESSION['installation_complete'])) {
        $step = 6; // Finalização
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Sistema Villa Joias</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1a1a1a 0%, #2d2d2d 100%);
            color: #ffffff;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .container {
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            padding: 40px;
            width: 90%;
            max-width: 600px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 215, 0, 0.2);
        }
        
        .logo {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .logo h1 {
            color: #FFD700;
            font-size: 2.5em;
            font-weight: bold;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.5);
        }
        
        .logo p {
            color: #cccccc;
            margin-top: 10px;
            font-size: 1.1em;
        }
        
        .progress {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
            position: relative;
        }
        
        .progress::before {
            content: '';
            position: absolute;
            top: 15px;
            left: 0;
            right: 0;
            height: 2px;
            background: rgba(255, 255, 255, 0.2);
            z-index: 1;
        }
        
        .progress-step {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            position: relative;
            z-index: 2;
        }
        
        .progress-step.active {
            background: #FFD700;
            color: #000;
        }
        
        .progress-step.completed {
            background: #4CAF50;
            color: #fff;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #FFD700;
            font-weight: 500;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.1);
            color: #ffffff;
            font-size: 16px;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #FFD700;
            box-shadow: 0 0 10px rgba(255, 215, 0, 0.3);
        }
        
        .form-row {
            display: flex;
            gap: 15px;
        }
        
        .form-row .form-group {
            flex: 1;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .checkbox-group input[type="checkbox"] {
            width: auto;
        }
        
        .btn {
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
            color: #000;
            border: none;
            padding: 15px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
            width: 100%;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(255, 215, 0, 0.3);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: #ffffff;
            margin-right: 10px;
            width: auto;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: rgba(76, 175, 80, 0.2);
            border: 1px solid #4CAF50;
            color: #4CAF50;
        }
        
        .alert-error {
            background: rgba(244, 67, 54, 0.2);
            border: 1px solid #f44336;
            color: #f44336;
        }
        
        .requirements {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .requirement {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .requirement:last-child {
            border-bottom: none;
        }
        
        .status-ok {
            color: #4CAF50;
            font-weight: bold;
        }
        
        .status-error {
            color: #f44336;
            font-weight: bold;
        }
        
        .final-info {
            background: rgba(255, 215, 0, 0.1);
            border: 1px solid #FFD700;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .final-info h3 {
            color: #FFD700;
            margin-bottom: 15px;
        }
        
        .final-info p {
            margin-bottom: 10px;
        }
        
        .final-info strong {
            color: #FFD700;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="logo">
            <h1>💎 Villa Joias</h1>
            <p>Instalação do Sistema - Versão cPanel VPS</p>
        </div>
        
        <div class="progress">
            <?php for ($i = 1; $i <= 6; $i++): ?>
                <div class="progress-step <?php 
                    if ($i < $step) echo 'completed';
                    elseif ($i == $step) echo 'active';
                ?>">
                    <?php echo $i; ?>
                </div>
            <?php endfor; ?>
        </div>
        
        <?php if (!empty($errors)): ?>
            <div class="alert alert-error">
                <?php foreach ($errors as $error): ?>
                    <p>❌ <?php echo htmlspecialchars($error); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <div class="alert alert-success">
                <?php foreach ($success as $msg): ?>
                    <p>✅ <?php echo htmlspecialchars($msg); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($step == 1): ?>
            <h2>Passo 1: Verificação de Requisitos</h2>
            <div class="requirements">
                <div class="requirement">
                    <span>PHP <?php echo MIN_PHP_VERSION; ?>+</span>
                    <span class="<?php echo checkPHPVersion() ? 'status-ok' : 'status-error'; ?>">
                        <?php echo checkPHPVersion() ? '✅ ' . PHP_VERSION : '❌ ' . PHP_VERSION; ?>
                    </span>
                </div>
                
                <?php $missingExt = checkPHPExtensions(); ?>
                <div class="requirement">
                    <span>Extensões PHP</span>
                    <span class="<?php echo empty($missingExt) ? 'status-ok' : 'status-error'; ?>">
                        <?php echo empty($missingExt) ? '✅ Todas OK' : '❌ Faltando: ' . implode(', ', $missingExt); ?>
                    </span>
                </div>
                
                <?php $writeIssues = checkWritePermissions(); ?>
                <div class="requirement">
                    <span>Permissões de Escrita</span>
                    <span class="<?php echo empty($writeIssues) ? 'status-ok' : 'status-error'; ?>">
                        <?php echo empty($writeIssues) ? '✅ Todas OK' : '❌ Problemas: ' . implode(', ', $writeIssues); ?>
                    </span>
                </div>
                
                <div class="requirement">
                    <span>Arquivo database.sql</span>
                    <span class="<?php echo file_exists('database.sql') ? 'status-ok' : 'status-error'; ?>">
                        <?php echo file_exists('database.sql') ? '✅ Encontrado' : '❌ Não encontrado'; ?>
                    </span>
                </div>
            </div>
            
            <form method="post" action="?step=2">
                <button type="submit" class="btn">Continuar para Configuração do Banco</button>
            </form>
            
        <?php elseif ($step == 2): ?>
            <h2>Passo 2: Configuração do Banco de Dados</h2>
            <p style="margin-bottom: 20px; color: #cccccc;">
                Configure a conexão com o banco de dados MySQL criado no cPanel.
            </p>
            
            <form method="post">
                <div class="form-group">
                    <label for="db_host">Servidor do Banco</label>
                    <input type="text" id="db_host" name="db_host" value="localhost" required>
                </div>
                
                <div class="form-group">
                    <label for="db_name">Nome do Banco</label>
                    <input type="text" id="db_name" name="db_name" placeholder="usuario_villa_joias" required>
                    <small style="color: #cccccc;">Use o nome completo criado no cPanel</small>
                </div>
                
                <div class="form-group">
                    <label for="db_user">Usuário do Banco</label>
                    <input type="text" id="db_user" name="db_user" placeholder="usuario_villa_admin" required>
                    <small style="color: #cccccc;">Use o nome completo criado no cPanel</small>
                </div>
                
                <div class="form-group">
                    <label for="db_pass">Senha do Banco</label>
                    <input type="password" id="db_pass" name="db_pass" required>
                </div>
                
                <div class="form-group">
                    <label for="db_port">Porta</label>
                    <input type="text" id="db_port" name="db_port" value="3306">
                </div>
                
                <button type="submit" class="btn">Testar Conexão e Continuar</button>
            </form>
            
        <?php elseif ($step == 3): ?>
            <h2>Passo 3: Criação das Tabelas</h2>
            <p style="margin-bottom: 20px; color: #cccccc;">
                Agora vamos criar as tabelas necessárias no banco de dados.
            </p>
            
            <form method="post">
                <button type="submit" class="btn">Criar Tabelas do Sistema</button>
            </form>
            
        <?php elseif ($step == 4): ?>
            <h2>Passo 4: Configuração do Administrador</h2>
            <p style="margin-bottom: 20px; color: #cccccc;">
                Crie o usuário administrador do sistema.
            </p>
            
            <form method="post">
                <div class="form-group">
                    <label for="admin_name">Nome Completo</label>
                    <input type="text" id="admin_name" name="admin_name" placeholder="Administrador Villa Joias" required>
                </div>
                
                <div class="form-group">
                    <label for="admin_email">Email</label>
                    <input type="email" id="admin_email" name="admin_email" placeholder="admin@seudominio.com.br" required>
                </div>
                
                <div class="form-group">
                    <label for="admin_password">Senha</label>
                    <input type="password" id="admin_password" name="admin_password" required>
                    <small style="color: #cccccc;">Mínimo 6 caracteres</small>
                </div>
                
                <div class="form-group">
                    <label for="admin_confirm_password">Confirmar Senha</label>
                    <input type="password" id="admin_confirm_password" name="admin_confirm_password" required>
                </div>
                
                <button type="submit" class="btn">Criar Administrador</button>
            </form>
            
        <?php elseif ($step == 5): ?>
            <h2>Passo 5: Configurações Gerais</h2>
            <p style="margin-bottom: 20px; color: #cccccc;">
                Configure os dados da sua empresa e preferências do sistema.
            </p>
            
            <form method="post">
                <div class="form-group">
                    <label for="company_name">Nome da Empresa</label>
                    <input type="text" id="company_name" name="company_name" value="Villa Joias" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="cnpj">CNPJ</label>
                        <input type="text" id="cnpj" name="cnpj" placeholder="00.000.000/0001-00">
                    </div>
                    
                    <div class="form-group">
                        <label for="phone">Telefone</label>
                        <input type="text" id="phone" name="phone" placeholder="(11) 99999-9999">
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="email">Email da Empresa</label>
                    <input type="email" id="email" name="email" placeholder="contato@villajoias.com.br">
                </div>
                
                <div class="form-group">
                    <label for="address">Endereço Completo</label>
                    <textarea id="address" name="address" rows="3" placeholder="Rua, número, bairro, cidade, estado, CEP"></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="timezone">Fuso Horário</label>
                        <select id="timezone" name="timezone">
                            <option value="America/Sao_Paulo">São Paulo (GMT-3)</option>
                            <option value="America/Manaus">Manaus (GMT-4)</option>
                            <option value="America/Rio_Branco">Rio Branco (GMT-5)</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="currency">Moeda</label>
                        <select id="currency" name="currency">
                            <option value="BRL">Real (R$)</option>
                            <option value="USD">Dólar (US$)</option>
                            <option value="EUR">Euro (€)</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="site_url">URL do Site</label>
                    <input type="url" id="site_url" name="site_url" value="https://<?php echo $_SERVER['HTTP_HOST']; ?>" required>
                </div>
                
                <h3 style="color: #FFD700; margin: 30px 0 20px 0;">Módulos e Funcionalidades</h3>
                
                <div class="checkbox-group">
                    <input type="checkbox" id="backup_enabled" name="backup_enabled" checked>
                    <label for="backup_enabled">Ativar backup automático</label>
                </div>
                
                <div class="checkbox-group">
                    <input type="checkbox" id="ai_enabled" name="ai_enabled" checked>
                    <label for="ai_enabled">Ativar assistente de IA (Manus)</label>
                </div>
                
                <div class="checkbox-group">
                    <input type="checkbox" id="module_cobranca" name="module_cobranca" checked>
                    <label for="module_cobranca">Ativar módulo de cobrança</label>
                </div>
                
                <div class="checkbox-group">
                    <input type="checkbox" id="module_relatorios" name="module_relatorios" checked>
                    <label for="module_relatorios">Ativar relatórios avançados</label>
                </div>
                
                <button type="submit" class="btn">Salvar Configurações e Finalizar</button>
            </form>
            
        <?php elseif ($step == 6): ?>
            <h2>🎉 Instalação Concluída!</h2>
            
            <div class="final-info">
                <h3>Sistema Villa Joias Instalado com Sucesso!</h3>
                
                <p><strong>URL do Sistema:</strong> <?php echo 'https://' . $_SERVER['HTTP_HOST']; ?></p>
                
                <?php if (isset($_SESSION['admin_config'])): ?>
                    <p><strong>Email do Administrador:</strong> <?php echo htmlspecialchars($_SESSION['admin_config']['email']); ?></p>
                    <p><strong>Senha:</strong> A senha que você definiu</p>
                <?php endif; ?>
                
                <p><strong>Versão:</strong> <?php echo INSTALL_VERSION; ?></p>
                <p><strong>Data da Instalação:</strong> <?php echo date('d/m/Y H:i:s'); ?></p>
            </div>
            
            <div class="alert alert-success">
                <p>✅ Banco de dados configurado</p>
                <p>✅ Tabelas criadas (32 tabelas)</p>
                <p>✅ Usuário administrador criado</p>
                <p>✅ Configurações salvas</p>
                <p>✅ Sistema pronto para uso</p>
            </div>
            
            <div style="display: flex; gap: 10px;">
                <a href="index.php" class="btn">Acessar Sistema</a>
                <a href="?cleanup=1" class="btn btn-secondary">Limpar Instalação</a>
            </div>
            
            <div style="margin-top: 30px; padding: 20px; background: rgba(255, 215, 0, 0.1); border-radius: 8px;">
                <h4 style="color: #FFD700; margin-bottom: 15px;">⚠️ Importante - Segurança</h4>
                <p style="color: #cccccc; margin-bottom: 10px;">
                    Por segurança, remova os seguintes arquivos do servidor:
                </p>
                <ul style="color: #cccccc; margin-left: 20px;">
                    <li>install.php (este arquivo)</li>
                    <li>database.sql</li>
                </ul>
                <p style="color: #cccccc; margin-top: 15px;">
                    Você pode fazer isso pelo Gerenciador de Arquivos do cPanel.
                </p>
            </div>
            
        <?php endif; ?>
    </div>
    
    <?php if (isset($_GET['cleanup'])): ?>
        <script>
            if (confirm('Deseja realmente remover os arquivos de instalação?')) {
                // Aqui você pode adicionar código para remover os arquivos
                alert('Remova manualmente os arquivos install.php e database.sql pelo cPanel.');
            }
        </script>
    <?php endif; ?>
</body>
</html>

