#!/bin/bash

# Script de Monitoramento - Sistema Villa Joias
# Versão: 2.0 VPS Edition
# Uso: ./monitor.sh

echo "📊 MONITORAMENTO - SISTEMA VILLA JOIAS"
echo "======================================"
echo ""

# Cores
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Função para status
check_status() {
    if [ $1 -eq 0 ]; then
        echo -e "${GREEN}✅ $2${NC}"
    else
        echo -e "${RED}❌ $2${NC}"
    fi
}

# Informações do sistema
echo -e "${BLUE}🖥️  INFORMAÇÕES DO SISTEMA${NC}"
echo "=========================="
echo "Hostname: $(hostname)"
echo "OS: $(cat /etc/os-release | grep PRETTY_NAME | cut -d'"' -f2)"
echo "Kernel: $(uname -r)"
echo "Uptime: $(uptime -p)"
echo ""

# Status dos serviços
echo -e "${BLUE}🔧 STATUS DOS SERVIÇOS${NC}"
echo "====================="
systemctl is-active --quiet apache2
check_status $? "Apache Web Server"

systemctl is-active --quiet mysql
check_status $? "MySQL Database"

systemctl is-active --quiet ufw
check_status $? "UFW Firewall"

# Verificar se o site está respondendo
if curl -s -o /dev/null -w "%{http_code}" "http://localhost" | grep -q "200\|301\|302"; then
    echo -e "${GREEN}✅ Website respondendo${NC}"
else
    echo -e "${RED}❌ Website não está respondendo${NC}"
fi

echo ""

# Uso de recursos
echo -e "${BLUE}📈 USO DE RECURSOS${NC}"
echo "=================="

# CPU
CPU_USAGE=$(top -bn1 | grep "Cpu(s)" | awk '{print $2}' | cut -d'%' -f1)
echo "CPU: ${CPU_USAGE}%"

# Memória
MEMORY_INFO=$(free -h | grep "Mem:")
MEMORY_USED=$(echo $MEMORY_INFO | awk '{print $3}')
MEMORY_TOTAL=$(echo $MEMORY_INFO | awk '{print $2}')
MEMORY_PERCENT=$(free | grep Mem | awk '{printf("%.1f", $3/$2 * 100.0)}')
echo "Memória: ${MEMORY_USED}/${MEMORY_TOTAL} (${MEMORY_PERCENT}%)"

# Disco
DISK_INFO=$(df -h / | tail -1)
DISK_USED=$(echo $DISK_INFO | awk '{print $3}')
DISK_TOTAL=$(echo $DISK_INFO | awk '{print $2}')
DISK_PERCENT=$(echo $DISK_INFO | awk '{print $5}')
echo "Disco: ${DISK_USED}/${DISK_TOTAL} (${DISK_PERCENT})"

# Swap
SWAP_INFO=$(free -h | grep "Swap:")
if [ -n "$SWAP_INFO" ]; then
    SWAP_USED=$(echo $SWAP_INFO | awk '{print $3}')
    SWAP_TOTAL=$(echo $SWAP_INFO | awk '{print $2}')
    echo "Swap: ${SWAP_USED}/${SWAP_TOTAL}"
else
    echo "Swap: Não configurado"
fi

echo ""

# Conexões ativas
echo -e "${BLUE}🌐 CONEXÕES DE REDE${NC}"
echo "==================="
CONNECTIONS=$(netstat -an | grep :80 | grep ESTABLISHED | wc -l)
echo "Conexões HTTP ativas: $CONNECTIONS"

MYSQL_CONNECTIONS=$(mysql -u villa_admin -pVillaJoias2025! -e "SHOW STATUS LIKE 'Threads_connected';" 2>/dev/null | tail -1 | awk '{print $2}')
if [ -n "$MYSQL_CONNECTIONS" ]; then
    echo "Conexões MySQL ativas: $MYSQL_CONNECTIONS"
fi

echo ""

# Logs recentes
echo -e "${BLUE}📋 LOGS RECENTES${NC}"
echo "================"

# Erros do Apache
APACHE_ERRORS=$(tail -10 /var/log/apache2/error.log 2>/dev/null | wc -l)
if [ $APACHE_ERRORS -gt 0 ]; then
    echo -e "${YELLOW}⚠️  $APACHE_ERRORS erros recentes no Apache${NC}"
    echo "Últimos erros:"
    tail -3 /var/log/apache2/error.log 2>/dev/null | sed 's/^/   /'
else
    echo -e "${GREEN}✅ Nenhum erro recente no Apache${NC}"
fi

# Erros do MySQL
MYSQL_ERRORS=$(tail -10 /var/log/mysql/error.log 2>/dev/null | wc -l)
if [ $MYSQL_ERRORS -gt 0 ]; then
    echo -e "${YELLOW}⚠️  Verificar logs do MySQL${NC}"
else
    echo -e "${GREEN}✅ MySQL funcionando normalmente${NC}"
fi

echo ""

# Backup
echo -e "${BLUE}💾 STATUS DO BACKUP${NC}"
echo "=================="

if [ -f "/root/backup_villa_joias.sh" ]; then
    echo -e "${GREEN}✅ Script de backup configurado${NC}"
    
    # Último backup
    LAST_BACKUP=$(ls -t /root/backups/db_*.sql 2>/dev/null | head -1)
    if [ -n "$LAST_BACKUP" ]; then
        BACKUP_DATE=$(stat -c %y "$LAST_BACKUP" | cut -d' ' -f1)
        echo "Último backup: $BACKUP_DATE"
        
        # Verificar se backup é recente (últimas 25 horas)
        if find "$LAST_BACKUP" -mtime -1 | grep -q .; then
            echo -e "${GREEN}✅ Backup recente${NC}"
        else
            echo -e "${YELLOW}⚠️  Backup não é recente${NC}"
        fi
    else
        echo -e "${YELLOW}⚠️  Nenhum backup encontrado${NC}"
    fi
    
    # Espaço usado pelos backups
    BACKUP_SIZE=$(du -sh /root/backups 2>/dev/null | cut -f1)
    if [ -n "$BACKUP_SIZE" ]; then
        echo "Espaço usado: $BACKUP_SIZE"
    fi
else
    echo -e "${RED}❌ Script de backup não encontrado${NC}"
fi

echo ""

# SSL
echo -e "${BLUE}🔒 STATUS DO SSL${NC}"
echo "==============="

if [ -d "/etc/letsencrypt/live" ]; then
    DOMAINS=$(ls /etc/letsencrypt/live/ 2>/dev/null)
    if [ -n "$DOMAINS" ]; then
        for domain in $DOMAINS; do
            if [ "$domain" != "README" ]; then
                CERT_FILE="/etc/letsencrypt/live/$domain/cert.pem"
                if [ -f "$CERT_FILE" ]; then
                    EXPIRY=$(openssl x509 -enddate -noout -in "$CERT_FILE" | cut -d= -f2)
                    EXPIRY_EPOCH=$(date -d "$EXPIRY" +%s)
                    NOW_EPOCH=$(date +%s)
                    DAYS_LEFT=$(( ($EXPIRY_EPOCH - $NOW_EPOCH) / 86400 ))
                    
                    if [ $DAYS_LEFT -gt 30 ]; then
                        echo -e "${GREEN}✅ $domain: $DAYS_LEFT dias restantes${NC}"
                    elif [ $DAYS_LEFT -gt 7 ]; then
                        echo -e "${YELLOW}⚠️  $domain: $DAYS_LEFT dias restantes${NC}"
                    else
                        echo -e "${RED}❌ $domain: $DAYS_LEFT dias restantes (URGENTE)${NC}"
                    fi
                fi
            fi
        done
    else
        echo -e "${YELLOW}⚠️  Nenhum certificado SSL encontrado${NC}"
    fi
else
    echo -e "${YELLOW}⚠️  Let's Encrypt não configurado${NC}"
fi

echo ""

# Alertas
echo -e "${BLUE}🚨 ALERTAS${NC}"
echo "========="

ALERTS=0

# Verificar uso de disco
DISK_USAGE=$(df / | tail -1 | awk '{print $5}' | sed 's/%//')
if [ $DISK_USAGE -gt 80 ]; then
    echo -e "${RED}❌ Uso de disco alto: ${DISK_USAGE}%${NC}"
    ALERTS=$((ALERTS + 1))
fi

# Verificar uso de memória
MEMORY_USAGE=$(free | grep Mem | awk '{printf("%.0f", $3/$2 * 100.0)}')
if [ $MEMORY_USAGE -gt 80 ]; then
    echo -e "${RED}❌ Uso de memória alto: ${MEMORY_USAGE}%${NC}"
    ALERTS=$((ALERTS + 1))
fi

# Verificar load average
LOAD_AVG=$(uptime | awk -F'load average:' '{print $2}' | awk '{print $1}' | sed 's/,//')
LOAD_INT=$(echo "$LOAD_AVG" | cut -d'.' -f1)
if [ $LOAD_INT -gt 4 ]; then
    echo -e "${RED}❌ Load average alto: $LOAD_AVG${NC}"
    ALERTS=$((ALERTS + 1))
fi

if [ $ALERTS -eq 0 ]; then
    echo -e "${GREEN}✅ Nenhum alerta crítico${NC}"
fi

echo ""
echo "📊 Monitoramento concluído - $(date)"
echo "Para monitoramento contínuo: watch -n 30 ./monitor.sh"

