"""
Modelos de dados para cartões de crédito e dívidas
"""
from src.models.core import db
from datetime import datetime, date
from decimal import Decimal

class Cartao(db.Model):
    __tablename__ = 'cartoes'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    instituicao = db.Column(db.String(100))
    limite_total = db.Column(db.Numeric(15, 2), default=0.00)
    limite_disponivel = db.Column(db.Numeric(15, 2), default=0.00)
    dia_fechamento = db.Column(db.Integer, nullable=False)
    dia_vencimento = db.Column(db.Integer, nullable=False)
    ativo = db.Column(db.Boolean, default=True)
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def limite_usado(self):
        return float(self.limite_total) - float(self.limite_disponivel)
    
    def percentual_uso(self):
        if float(self.limite_total) > 0:
            return (self.limite_usado() / float(self.limite_total)) * 100
        return 0
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'instituicao': self.instituicao,
            'limite_total': float(self.limite_total),
            'limite_disponivel': float(self.limite_disponivel),
            'limite_usado': self.limite_usado(),
            'percentual_uso': round(self.percentual_uso(), 2),
            'dia_fechamento': self.dia_fechamento,
            'dia_vencimento': self.dia_vencimento,
            'ativo': self.ativo,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class FaturaCartao(db.Model):
    __tablename__ = 'faturas_cartao'
    
    id = db.Column(db.Integer, primary_key=True)
    cartao_id = db.Column(db.Integer, db.ForeignKey('cartoes.id'), nullable=False)
    mes_referencia = db.Column(db.Date, nullable=False)
    valor_total = db.Column(db.Numeric(15, 2), default=0.00)
    valor_pago = db.Column(db.Numeric(15, 2), default=0.00)
    valor_pendente = db.Column(db.Numeric(15, 2), default=0.00)
    data_vencimento = db.Column(db.Date, nullable=False)
    status = db.Column(db.Enum('aberta', 'paga_total', 'paga_parcial', 'vencida'), default='aberta')
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relacionamentos
    cartao = db.relationship('Cartao', backref='faturas')
    
    def percentual_pago(self):
        if float(self.valor_total) > 0:
            return (float(self.valor_pago) / float(self.valor_total)) * 100
        return 0
    
    def to_dict(self):
        return {
            'id': self.id,
            'cartao_id': self.cartao_id,
            'cartao_nome': self.cartao.nome if self.cartao else None,
            'mes_referencia': self.mes_referencia.isoformat(),
            'valor_total': float(self.valor_total),
            'valor_pago': float(self.valor_pago),
            'valor_pendente': float(self.valor_pendente),
            'percentual_pago': round(self.percentual_pago(), 2),
            'data_vencimento': self.data_vencimento.isoformat(),
            'status': self.status,
            'criado_em': self.criado_em.isoformat()
        }

class Divida(db.Model):
    __tablename__ = 'dividas'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(200), nullable=False)
    descricao = db.Column(db.Text)
    valor_total = db.Column(db.Numeric(15, 2), nullable=False)
    valor_pago = db.Column(db.Numeric(15, 2), default=0.00)
    valor_pendente = db.Column(db.Numeric(15, 2), nullable=False)
    numero_parcelas = db.Column(db.Integer, nullable=False)
    valor_parcela = db.Column(db.Numeric(15, 2), nullable=False)
    data_primeira_parcela = db.Column(db.Date, nullable=False)
    periodicidade = db.Column(db.Enum('semanal', 'quinzenal', 'mensal', 'bimestral', 'trimestral'), default='mensal')
    categoria_id = db.Column(db.Integer, db.ForeignKey('categorias.id'))
    subcategoria_id = db.Column(db.Integer, db.ForeignKey('subcategorias.id'))
    status = db.Column(db.Enum('ativa', 'quitada', 'cancelada'), default='ativa')
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    # Relacionamentos
    categoria = db.relationship('Categoria', backref='dividas')
    subcategoria = db.relationship('Subcategoria', backref='dividas')
    
    def percentual_pago(self):
        if float(self.valor_total) > 0:
            return (float(self.valor_pago) / float(self.valor_total)) * 100
        return 0
    
    def parcelas_pagas(self):
        return len([p for p in self.parcelas if p.status == 'paga'])
    
    def parcelas_pendentes(self):
        return len([p for p in self.parcelas if p.status == 'pendente'])
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'descricao': self.descricao,
            'valor_total': float(self.valor_total),
            'valor_pago': float(self.valor_pago),
            'valor_pendente': float(self.valor_pendente),
            'percentual_pago': round(self.percentual_pago(), 2),
            'numero_parcelas': self.numero_parcelas,
            'valor_parcela': float(self.valor_parcela),
            'data_primeira_parcela': self.data_primeira_parcela.isoformat(),
            'periodicidade': self.periodicidade,
            'categoria_id': self.categoria_id,
            'categoria_nome': self.categoria.nome if self.categoria else None,
            'subcategoria_id': self.subcategoria_id,
            'subcategoria_nome': self.subcategoria.nome if self.subcategoria else None,
            'status': self.status,
            'parcelas_pagas': self.parcelas_pagas(),
            'parcelas_pendentes': self.parcelas_pendentes(),
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class ParcelaDivida(db.Model):
    __tablename__ = 'parcelas_divida'
    
    id = db.Column(db.Integer, primary_key=True)
    divida_id = db.Column(db.Integer, db.ForeignKey('dividas.id'), nullable=False)
    numero_parcela = db.Column(db.Integer, nullable=False)
    data_vencimento = db.Column(db.Date, nullable=False)
    valor_previsto = db.Column(db.Numeric(15, 2), nullable=False)
    valor_pago = db.Column(db.Numeric(15, 2), default=0.00)
    data_pagamento = db.Column(db.Date)
    caixa_pagamento_id = db.Column(db.Integer, db.ForeignKey('caixas.id'))
    status = db.Column(db.Enum('pendente', 'paga', 'atrasada', 'cancelada'), default='pendente')
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relacionamentos
    divida = db.relationship('Divida', backref='parcelas')
    caixa_pagamento = db.relationship('Caixa', backref='pagamentos_dividas')
    
    def dias_atraso(self):
        if self.status == 'pendente' and self.data_vencimento < date.today():
            return (date.today() - self.data_vencimento).days
        return 0
    
    def esta_atrasada(self):
        return self.status == 'pendente' and self.data_vencimento < date.today()
    
    def to_dict(self):
        return {
            'id': self.id,
            'divida_id': self.divida_id,
            'divida_nome': self.divida.nome if self.divida else None,
            'numero_parcela': self.numero_parcela,
            'data_vencimento': self.data_vencimento.isoformat(),
            'valor_previsto': float(self.valor_previsto),
            'valor_pago': float(self.valor_pago),
            'data_pagamento': self.data_pagamento.isoformat() if self.data_pagamento else None,
            'caixa_pagamento_id': self.caixa_pagamento_id,
            'caixa_pagamento_nome': self.caixa_pagamento.nome if self.caixa_pagamento else None,
            'status': self.status,
            'dias_atraso': self.dias_atraso(),
            'esta_atrasada': self.esta_atrasada(),
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

