"""
Modelos de dados para o Sistema Villa Joias
"""
from flask_sqlalchemy import SQLAlchemy
from datetime import datetime, date
from decimal import Decimal
import json
from werkzeug.security import generate_password_hash, check_password_hash
from flask_jwt_extended import create_access_token, create_refresh_token

db = SQLAlchemy()

class Usuario(db.Model):
    __tablename__ = 'usuarios'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    email = db.Column(db.String(100), unique=True, nullable=False)
    senha_hash = db.Column(db.String(255), nullable=False)
    tipo_usuario = db.Column(db.Enum('administrador', 'gerente', 'operador', 'consulta', 'personalizado'), default='operador')
    ativo = db.Column(db.Boolean, default=True)
    ultimo_login = db.Column(db.DateTime)
    tentativas_login = db.Column(db.Integer, default=0)
    bloqueado_ate = db.Column(db.DateTime)
    token_2fa = db.Column(db.String(32))
    ativo_2fa = db.Column(db.Boolean, default=False)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def set_password(self, password):
        self.senha_hash = generate_password_hash(password)
    
    def check_password(self, password):
        return check_password_hash(self.senha_hash, password)
    
    def generate_tokens(self):
        access_token = create_access_token(identity=self.id)
        refresh_token = create_refresh_token(identity=self.id)
        return access_token, refresh_token
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'email': self.email,
            'tipo_usuario': self.tipo_usuario,
            'ativo': self.ativo,
            'ultimo_login': self.ultimo_login.isoformat() if self.ultimo_login else None,
            'criado_em': self.criado_em.isoformat()
        }

class Caixa(db.Model):
    __tablename__ = 'caixas'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    tipo = db.Column(db.Enum('capital_giro', 'reserva', 'conta_corrente', 'fisico', 'outros'), nullable=False)
    descricao = db.Column(db.Text)
    instituicao = db.Column(db.String(100))
    saldo_inicial = db.Column(db.Numeric(15, 2), default=0.00)
    saldo_atual = db.Column(db.Numeric(15, 2), default=0.00)
    limite_cheque_especial = db.Column(db.Numeric(15, 2), default=0.00)
    taxa_rendimento = db.Column(db.Numeric(8, 4), default=0.00)
    periodicidade_rendimento = db.Column(db.Enum('diario', 'mensal', 'trimestral', 'anual'), default='mensal')
    ativo = db.Column(db.Boolean, default=True)
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def saldo_disponivel(self):
        return float(self.saldo_atual) + float(self.limite_cheque_especial or 0)
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'tipo': self.tipo,
            'descricao': self.descricao,
            'instituicao': self.instituicao,
            'saldo_inicial': float(self.saldo_inicial),
            'saldo_atual': float(self.saldo_atual),
            'limite_cheque_especial': float(self.limite_cheque_especial or 0),
            'saldo_disponivel': self.saldo_disponivel(),
            'taxa_rendimento': float(self.taxa_rendimento or 0),
            'periodicidade_rendimento': self.periodicidade_rendimento,
            'ativo': self.ativo,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class Representante(db.Model):
    __tablename__ = 'representantes'
    
    id = db.Column(db.Integer, primary_key=True)
    codigo = db.Column(db.String(20), unique=True, nullable=False)
    nome = db.Column(db.String(100), nullable=False)
    email = db.Column(db.String(100))
    telefone = db.Column(db.String(20))
    regiao = db.Column(db.String(50))
    comissao_padrao = db.Column(db.Numeric(5, 2), default=0.00)
    meta_mensal = db.Column(db.Numeric(15, 2), default=0.00)
    ativo = db.Column(db.Boolean, default=True)
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def to_dict(self):
        return {
            'id': self.id,
            'codigo': self.codigo,
            'nome': self.nome,
            'email': self.email,
            'telefone': self.telefone,
            'regiao': self.regiao,
            'comissao_padrao': float(self.comissao_padrao),
            'meta_mensal': float(self.meta_mensal),
            'ativo': self.ativo,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class Cliente(db.Model):
    __tablename__ = 'clientes'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    representante_id = db.Column(db.Integer, db.ForeignKey('representantes.id'), nullable=False)
    telefone = db.Column(db.String(20))
    email = db.Column(db.String(100))
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    # Relacionamentos
    representante = db.relationship('Representante', backref='clientes')
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'representante_id': self.representante_id,
            'representante_nome': self.representante.nome if self.representante else None,
            'telefone': self.telefone,
            'email': self.email,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

class Categoria(db.Model):
    __tablename__ = 'categorias'
    
    id = db.Column(db.Integer, primary_key=True)
    nome = db.Column(db.String(100), nullable=False)
    descricao = db.Column(db.Text)
    cor = db.Column(db.String(7), default='#2C3E50')
    icone = db.Column(db.String(50))
    orcamento_mensal = db.Column(db.Numeric(15, 2), default=0.00)
    ativo = db.Column(db.Boolean, default=True)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    def to_dict(self):
        return {
            'id': self.id,
            'nome': self.nome,
            'descricao': self.descricao,
            'cor': self.cor,
            'icone': self.icone,
            'orcamento_mensal': float(self.orcamento_mensal),
            'ativo': self.ativo,
            'criado_em': self.criado_em.isoformat()
        }

class Subcategoria(db.Model):
    __tablename__ = 'subcategorias'
    
    id = db.Column(db.Integer, primary_key=True)
    categoria_id = db.Column(db.Integer, db.ForeignKey('categorias.id'), nullable=False)
    nome = db.Column(db.String(100), nullable=False)
    descricao = db.Column(db.Text)
    orcamento_mensal = db.Column(db.Numeric(15, 2), default=0.00)
    ativo = db.Column(db.Boolean, default=True)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    # Relacionamentos
    categoria = db.relationship('Categoria', backref='subcategorias')
    
    def to_dict(self):
        return {
            'id': self.id,
            'categoria_id': self.categoria_id,
            'categoria_nome': self.categoria.nome if self.categoria else None,
            'nome': self.nome,
            'descricao': self.descricao,
            'orcamento_mensal': float(self.orcamento_mensal),
            'ativo': self.ativo,
            'criado_em': self.criado_em.isoformat()
        }

class Transacao(db.Model):
    __tablename__ = 'transacoes'
    
    id = db.Column(db.BigInteger, primary_key=True)
    tipo = db.Column(db.Enum('entrada', 'saida', 'transferencia', 'rendimento', 'taxa', 'ajuste'), nullable=False)
    subtipo = db.Column(db.String(50))
    data_transacao = db.Column(db.Date, nullable=False)
    valor = db.Column(db.Numeric(15, 2), nullable=False)
    descricao = db.Column(db.Text, nullable=False)
    
    # Relacionamentos
    caixa_origem_id = db.Column(db.Integer, db.ForeignKey('caixas.id'))
    caixa_destino_id = db.Column(db.Integer, db.ForeignKey('caixas.id'))
    representante_id = db.Column(db.Integer, db.ForeignKey('representantes.id'))
    cliente_id = db.Column(db.Integer, db.ForeignKey('clientes.id'))
    categoria_id = db.Column(db.Integer, db.ForeignKey('categorias.id'))
    subcategoria_id = db.Column(db.Integer, db.ForeignKey('subcategorias.id'))
    
    # Dados específicos
    metodo_pagamento = db.Column(db.Enum('dinheiro', 'pix', 'transferencia', 'cartao_debito', 'cartao_credito', 'cheque', 'outros'))
    numero_parcelas = db.Column(db.Integer, default=1)
    parcela_atual = db.Column(db.Integer, default=1)
    valor_original = db.Column(db.Numeric(15, 2))
    
    # Controle
    status = db.Column(db.Enum('pendente', 'confirmado', 'cancelado', 'estornado'), default='confirmado')
    conciliado = db.Column(db.Boolean, default=False)
    data_conciliacao = db.Column(db.DateTime)
    
    # Auditoria
    observacoes = db.Column(db.Text)
    criado_em = db.Column(db.DateTime, default=datetime.utcnow)
    atualizado_em = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    criado_por = db.Column(db.Integer, db.ForeignKey('usuarios.id'))
    
    # Relacionamentos
    caixa_origem = db.relationship('Caixa', foreign_keys=[caixa_origem_id], backref='transacoes_origem')
    caixa_destino = db.relationship('Caixa', foreign_keys=[caixa_destino_id], backref='transacoes_destino')
    representante = db.relationship('Representante', backref='transacoes')
    cliente = db.relationship('Cliente', backref='transacoes')
    categoria = db.relationship('Categoria', backref='transacoes')
    subcategoria = db.relationship('Subcategoria', backref='transacoes')
    
    def to_dict(self):
        return {
            'id': self.id,
            'tipo': self.tipo,
            'subtipo': self.subtipo,
            'data_transacao': self.data_transacao.isoformat(),
            'valor': float(self.valor),
            'descricao': self.descricao,
            'caixa_origem_id': self.caixa_origem_id,
            'caixa_origem_nome': self.caixa_origem.nome if self.caixa_origem else None,
            'caixa_destino_id': self.caixa_destino_id,
            'caixa_destino_nome': self.caixa_destino.nome if self.caixa_destino else None,
            'representante_id': self.representante_id,
            'representante_nome': self.representante.nome if self.representante else None,
            'cliente_id': self.cliente_id,
            'cliente_nome': self.cliente.nome if self.cliente else None,
            'categoria_id': self.categoria_id,
            'categoria_nome': self.categoria.nome if self.categoria else None,
            'subcategoria_id': self.subcategoria_id,
            'subcategoria_nome': self.subcategoria.nome if self.subcategoria else None,
            'metodo_pagamento': self.metodo_pagamento,
            'numero_parcelas': self.numero_parcelas,
            'parcela_atual': self.parcela_atual,
            'valor_original': float(self.valor_original) if self.valor_original else None,
            'status': self.status,
            'conciliado': self.conciliado,
            'observacoes': self.observacoes,
            'criado_em': self.criado_em.isoformat()
        }

