"""
Rotas para gerenciamento de caixas
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Caixa, Transacao
from datetime import datetime
from decimal import Decimal

caixas_bp = Blueprint('caixas', __name__)

@caixas_bp.route('/caixas', methods=['GET'])
@jwt_required()
def listar_caixas():
    """Lista todos os caixas ativos"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 50, type=int)
        tipo = request.args.get('tipo')
        ativo = request.args.get('ativo', 'true').lower() == 'true'
        
        query = Caixa.query.filter_by(ativo=ativo)
        
        if tipo:
            query = query.filter_by(tipo=tipo)
        
        caixas = query.order_by(Caixa.nome).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'success': True,
            'data': [caixa.to_dict() for caixa in caixas.items],
            'pagination': {
                'page': page,
                'pages': caixas.pages,
                'per_page': per_page,
                'total': caixas.total
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/<int:caixa_id>', methods=['GET'])
@jwt_required()
def obter_caixa(caixa_id):
    """Obtém detalhes de um caixa específico"""
    try:
        caixa = Caixa.query.get_or_404(caixa_id)
        return jsonify({
            'success': True,
            'data': caixa.to_dict()
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas', methods=['POST'])
@jwt_required()
def criar_caixa():
    """Cria um novo caixa"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if not data.get('tipo'):
            return jsonify({'success': False, 'message': 'Tipo é obrigatório'}), 400
        
        tipos_validos = ['capital_giro', 'reserva', 'conta_corrente', 'fisico', 'outros']
        if data['tipo'] not in tipos_validos:
            return jsonify({'success': False, 'message': 'Tipo inválido'}), 400
        
        # Verificar se já existe caixa com o mesmo nome
        if Caixa.query.filter_by(nome=data['nome'], ativo=True).first():
            return jsonify({'success': False, 'message': 'Já existe um caixa com este nome'}), 400
        
        caixa = Caixa(
            nome=data['nome'],
            tipo=data['tipo'],
            descricao=data.get('descricao'),
            instituicao=data.get('instituicao'),
            saldo_inicial=Decimal(str(data.get('saldo_inicial', 0))),
            saldo_atual=Decimal(str(data.get('saldo_inicial', 0))),
            limite_cheque_especial=Decimal(str(data.get('limite_cheque_especial', 0))),
            taxa_rendimento=Decimal(str(data.get('taxa_rendimento', 0))),
            periodicidade_rendimento=data.get('periodicidade_rendimento', 'mensal'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(caixa)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Caixa criado com sucesso',
            'data': caixa.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/<int:caixa_id>', methods=['PUT'])
@jwt_required()
def atualizar_caixa(caixa_id):
    """Atualiza um caixa existente"""
    try:
        caixa = Caixa.query.get_or_404(caixa_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if 'tipo' in data:
            tipos_validos = ['capital_giro', 'reserva', 'conta_corrente', 'fisico', 'outros']
            if data['tipo'] not in tipos_validos:
                return jsonify({'success': False, 'message': 'Tipo inválido'}), 400
        
        # Verificar se já existe outro caixa com o mesmo nome
        if 'nome' in data and data['nome'] != caixa.nome:
            if Caixa.query.filter_by(nome=data['nome'], ativo=True).filter(Caixa.id != caixa_id).first():
                return jsonify({'success': False, 'message': 'Já existe um caixa com este nome'}), 400
        
        # Atualizar campos
        if 'nome' in data:
            caixa.nome = data['nome']
        if 'tipo' in data:
            caixa.tipo = data['tipo']
        if 'descricao' in data:
            caixa.descricao = data['descricao']
        if 'instituicao' in data:
            caixa.instituicao = data['instituicao']
        if 'limite_cheque_especial' in data:
            caixa.limite_cheque_especial = Decimal(str(data['limite_cheque_especial']))
        if 'taxa_rendimento' in data:
            caixa.taxa_rendimento = Decimal(str(data['taxa_rendimento']))
        if 'periodicidade_rendimento' in data:
            caixa.periodicidade_rendimento = data['periodicidade_rendimento']
        if 'observacoes' in data:
            caixa.observacoes = data['observacoes']
        if 'ativo' in data:
            caixa.ativo = data['ativo']
        
        caixa.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Caixa atualizado com sucesso',
            'data': caixa.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/<int:caixa_id>/saldo', methods=['GET'])
@jwt_required()
def obter_saldo_caixa(caixa_id):
    """Obtém o saldo atual de um caixa"""
    try:
        caixa = Caixa.query.get_or_404(caixa_id)
        
        return jsonify({
            'success': True,
            'data': {
                'id': caixa.id,
                'nome': caixa.nome,
                'saldo_atual': float(caixa.saldo_atual),
                'limite_cheque_especial': float(caixa.limite_cheque_especial or 0),
                'saldo_disponivel': caixa.saldo_disponivel()
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/resumo', methods=['GET'])
@jwt_required()
def resumo_caixas():
    """Obtém resumo de todos os caixas"""
    try:
        caixas = Caixa.query.filter_by(ativo=True).all()
        
        resumo = {
            'total_caixas': len(caixas),
            'saldo_total': 0,
            'saldo_disponivel_total': 0,
            'por_tipo': {}
        }
        
        for caixa in caixas:
            resumo['saldo_total'] += float(caixa.saldo_atual)
            resumo['saldo_disponivel_total'] += caixa.saldo_disponivel()
            
            if caixa.tipo not in resumo['por_tipo']:
                resumo['por_tipo'][caixa.tipo] = {
                    'quantidade': 0,
                    'saldo_total': 0,
                    'saldo_disponivel_total': 0
                }
            
            resumo['por_tipo'][caixa.tipo]['quantidade'] += 1
            resumo['por_tipo'][caixa.tipo]['saldo_total'] += float(caixa.saldo_atual)
            resumo['por_tipo'][caixa.tipo]['saldo_disponivel_total'] += caixa.saldo_disponivel()
        
        return jsonify({
            'success': True,
            'data': resumo
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/<int:caixa_id>/extrato', methods=['GET'])
@jwt_required()
def extrato_caixa(caixa_id):
    """Obtém extrato de movimentações de um caixa"""
    try:
        caixa = Caixa.query.get_or_404(caixa_id)
        
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 50, type=int)
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        
        # Query base para transações do caixa
        query = Transacao.query.filter(
            (Transacao.caixa_origem_id == caixa_id) | 
            (Transacao.caixa_destino_id == caixa_id)
        ).filter_by(status='confirmado')
        
        # Filtros de data
        if data_inicio:
            query = query.filter(Transacao.data_transacao >= datetime.strptime(data_inicio, '%Y-%m-%d').date())
        if data_fim:
            query = query.filter(Transacao.data_transacao <= datetime.strptime(data_fim, '%Y-%m-%d').date())
        
        transacoes = query.order_by(Transacao.data_transacao.desc(), Transacao.id.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        # Processar transações para o extrato
        extrato = []
        for transacao in transacoes.items:
            item = transacao.to_dict()
            
            # Determinar se é entrada ou saída para este caixa
            if transacao.caixa_destino_id == caixa_id:
                item['movimento'] = 'entrada'
                item['valor_movimento'] = float(transacao.valor)
            else:
                item['movimento'] = 'saida'
                item['valor_movimento'] = -float(transacao.valor)
            
            extrato.append(item)
        
        return jsonify({
            'success': True,
            'data': {
                'caixa': caixa.to_dict(),
                'extrato': extrato,
                'pagination': {
                    'page': page,
                    'pages': transacoes.pages,
                    'per_page': per_page,
                    'total': transacoes.total
                }
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@caixas_bp.route('/caixas/tipos', methods=['GET'])
@jwt_required()
def tipos_caixa():
    """Lista os tipos de caixa disponíveis"""
    tipos = [
        {'value': 'capital_giro', 'label': 'Capital de Giro'},
        {'value': 'reserva', 'label': 'Reserva/Aplicação'},
        {'value': 'conta_corrente', 'label': 'Conta Corrente'},
        {'value': 'fisico', 'label': 'Caixa Físico'},
        {'value': 'outros', 'label': 'Outros'}
    ]
    
    return jsonify({
        'success': True,
        'data': tipos
    })

