"""
Rotas para gerenciamento de cartões de crédito
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Transacao, Categoria, Subcategoria
from src.models.avancados import Cartao, FaturaCartao
from datetime import datetime, date
from decimal import Decimal
from sqlalchemy import func, extract
from dateutil.relativedelta import relativedelta

cartoes_bp = Blueprint('cartoes', __name__)

@cartoes_bp.route('/cartoes', methods=['GET'])
@jwt_required()
def listar_cartoes():
    """Lista todos os cartões"""
    try:
        ativo = request.args.get('ativo', 'true').lower() == 'true'
        
        cartoes = Cartao.query.filter_by(ativo=ativo).order_by(Cartao.nome).all()
        
        return jsonify({
            'success': True,
            'data': [cartao.to_dict() for cartao in cartoes]
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/<int:cartao_id>', methods=['GET'])
@jwt_required()
def obter_cartao(cartao_id):
    """Obtém detalhes de um cartão específico"""
    try:
        cartao = Cartao.query.get_or_404(cartao_id)
        return jsonify({
            'success': True,
            'data': cartao.to_dict()
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes', methods=['POST'])
@jwt_required()
def criar_cartao():
    """Cria um novo cartão"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if not data.get('dia_fechamento') or not (1 <= data['dia_fechamento'] <= 31):
            return jsonify({'success': False, 'message': 'Dia de fechamento deve estar entre 1 e 31'}), 400
        
        if not data.get('dia_vencimento') or not (1 <= data['dia_vencimento'] <= 31):
            return jsonify({'success': False, 'message': 'Dia de vencimento deve estar entre 1 e 31'}), 400
        
        # Verificar se já existe cartão com o mesmo nome
        if Cartao.query.filter_by(nome=data['nome'], ativo=True).first():
            return jsonify({'success': False, 'message': 'Já existe um cartão com este nome'}), 400
        
        limite_total = Decimal(str(data.get('limite_total', 0)))
        
        cartao = Cartao(
            nome=data['nome'],
            instituicao=data.get('instituicao'),
            limite_total=limite_total,
            limite_disponivel=limite_total,  # Inicialmente todo o limite está disponível
            dia_fechamento=data['dia_fechamento'],
            dia_vencimento=data['dia_vencimento'],
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(cartao)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Cartão criado com sucesso',
            'data': cartao.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/<int:cartao_id>', methods=['PUT'])
@jwt_required()
def atualizar_cartao(cartao_id):
    """Atualiza um cartão existente"""
    try:
        cartao = Cartao.query.get_or_404(cartao_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if 'dia_fechamento' in data and not (1 <= data['dia_fechamento'] <= 31):
            return jsonify({'success': False, 'message': 'Dia de fechamento deve estar entre 1 e 31'}), 400
        
        if 'dia_vencimento' in data and not (1 <= data['dia_vencimento'] <= 31):
            return jsonify({'success': False, 'message': 'Dia de vencimento deve estar entre 1 e 31'}), 400
        
        # Verificar se já existe outro cartão com o mesmo nome
        if 'nome' in data and data['nome'] != cartao.nome:
            if Cartao.query.filter_by(nome=data['nome'], ativo=True).filter(Cartao.id != cartao_id).first():
                return jsonify({'success': False, 'message': 'Já existe um cartão com este nome'}), 400
        
        # Atualizar campos
        if 'nome' in data:
            cartao.nome = data['nome']
        if 'instituicao' in data:
            cartao.instituicao = data['instituicao']
        if 'limite_total' in data:
            novo_limite = Decimal(str(data['limite_total']))
            diferenca = novo_limite - cartao.limite_total
            cartao.limite_total = novo_limite
            cartao.limite_disponivel += diferenca  # Ajustar limite disponível
        if 'dia_fechamento' in data:
            cartao.dia_fechamento = data['dia_fechamento']
        if 'dia_vencimento' in data:
            cartao.dia_vencimento = data['dia_vencimento']
        if 'observacoes' in data:
            cartao.observacoes = data['observacoes']
        if 'ativo' in data:
            cartao.ativo = data['ativo']
        
        cartao.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Cartão atualizado com sucesso',
            'data': cartao.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/<int:cartao_id>/despesas', methods=['POST'])
@jwt_required()
def criar_despesa_cartao():
    """Cria uma nova despesa no cartão"""
    try:
        cartao_id = request.view_args['cartao_id']
        data = request.get_json()
        
        # Validações
        if not data.get('valor') or float(data['valor']) <= 0:
            return jsonify({'success': False, 'message': 'Valor deve ser maior que zero'}), 400
        
        if not data.get('descricao'):
            return jsonify({'success': False, 'message': 'Descrição é obrigatória'}), 400
        
        if not data.get('categoria_id'):
            return jsonify({'success': False, 'message': 'Categoria é obrigatória'}), 400
        
        # Verificar se o cartão existe
        cartao = Cartao.query.get(cartao_id)
        if not cartao or not cartao.ativo:
            return jsonify({'success': False, 'message': 'Cartão inválido'}), 400
        
        # Verificar categoria
        categoria = Categoria.query.get(data['categoria_id'])
        if not categoria or not categoria.ativo:
            return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar subcategoria se fornecida
        if data.get('subcategoria_id'):
            subcategoria = Subcategoria.query.get(data['subcategoria_id'])
            if not subcategoria or not subcategoria.ativo or subcategoria.categoria_id != data['categoria_id']:
                return jsonify({'success': False, 'message': 'Subcategoria inválida'}), 400
        
        valor_total = Decimal(str(data['valor']))
        numero_parcelas = data.get('numero_parcelas', 1)
        
        # Verificar limite disponível
        if valor_total > cartao.limite_disponivel:
            return jsonify({
                'success': False, 
                'message': f'Limite insuficiente. Disponível: R$ {float(cartao.limite_disponivel):.2f}'
            }), 400
        
        # Criar transação principal
        transacao = Transacao(
            tipo='saida',
            subtipo='cartao_credito',
            data_transacao=datetime.strptime(data.get('data_transacao', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=valor_total,
            descricao=data['descricao'],
            categoria_id=data['categoria_id'],
            subcategoria_id=data.get('subcategoria_id'),
            cartao_id=cartao_id,
            metodo_pagamento='cartao_credito',
            numero_parcelas=numero_parcelas,
            parcela_atual=1,
            valor_original=valor_total,
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Se parcelado, criar transações para as outras parcelas
        if numero_parcelas > 1:
            valor_parcela = valor_total / numero_parcelas
            data_base = datetime.strptime(data.get('data_transacao', date.today().isoformat()), '%Y-%m-%d').date()
            
            for i in range(2, numero_parcelas + 1):
                data_parcela = data_base + relativedelta(months=i-1)
                
                transacao_parcela = Transacao(
                    tipo='saida',
                    subtipo='cartao_credito',
                    data_transacao=data_parcela,
                    valor=valor_parcela,
                    descricao=f"{data['descricao']} - Parcela {i}/{numero_parcelas}",
                    categoria_id=data['categoria_id'],
                    subcategoria_id=data.get('subcategoria_id'),
                    cartao_id=cartao_id,
                    metodo_pagamento='cartao_credito',
                    numero_parcelas=numero_parcelas,
                    parcela_atual=i,
                    valor_original=valor_total,
                    observacoes=data.get('observacoes'),
                    criado_por=get_jwt_identity()
                )
                
                db.session.add(transacao_parcela)
        
        # Atualizar limite disponível do cartão
        cartao.limite_disponivel -= valor_total
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Despesa registrada com sucesso',
            'data': transacao.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/<int:cartao_id>/faturas', methods=['GET'])
@jwt_required()
def listar_faturas_cartao(cartao_id):
    """Lista faturas de um cartão"""
    try:
        cartao = Cartao.query.get_or_404(cartao_id)
        
        faturas = FaturaCartao.query.filter_by(cartao_id=cartao_id).order_by(
            FaturaCartao.mes_referencia.desc()
        ).all()
        
        return jsonify({
            'success': True,
            'data': {
                'cartao': cartao.to_dict(),
                'faturas': [fatura.to_dict() for fatura in faturas]
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/<int:cartao_id>/fatura-atual', methods=['GET'])
@jwt_required()
def obter_fatura_atual(cartao_id):
    """Obtém a fatura atual do cartão"""
    try:
        cartao = Cartao.query.get_or_404(cartao_id)
        
        # Calcular mês de referência baseado no dia de fechamento
        hoje = date.today()
        if hoje.day <= cartao.dia_fechamento:
            mes_referencia = hoje.replace(day=1) - relativedelta(months=1)
        else:
            mes_referencia = hoje.replace(day=1)
        
        # Buscar ou criar fatura
        fatura = FaturaCartao.query.filter_by(
            cartao_id=cartao_id,
            mes_referencia=mes_referencia
        ).first()
        
        if not fatura:
            # Calcular data de vencimento
            data_vencimento = mes_referencia + relativedelta(months=1)
            data_vencimento = data_vencimento.replace(day=cartao.dia_vencimento)
            
            fatura = FaturaCartao(
                cartao_id=cartao_id,
                mes_referencia=mes_referencia,
                data_vencimento=data_vencimento
            )
            db.session.add(fatura)
        
        # Calcular valor total das despesas do período
        data_inicio = mes_referencia
        data_fim = mes_referencia + relativedelta(months=1) - relativedelta(days=1)
        
        total_despesas = Transacao.query.filter(
            Transacao.cartao_id == cartao_id,
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= data_inicio,
            Transacao.data_transacao <= data_fim
        ).with_entities(func.sum(Transacao.valor)).scalar() or 0
        
        fatura.valor_total = total_despesas
        fatura.valor_pendente = total_despesas - fatura.valor_pago
        
        # Atualizar status
        if fatura.valor_pago >= fatura.valor_total:
            fatura.status = 'paga_total'
        elif fatura.valor_pago > 0:
            fatura.status = 'paga_parcial'
        elif fatura.data_vencimento < hoje:
            fatura.status = 'vencida'
        else:
            fatura.status = 'aberta'
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'data': fatura.to_dict()
        })
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/faturas/<int:fatura_id>/pagar', methods=['POST'])
@jwt_required()
def pagar_fatura(fatura_id):
    """Registra pagamento de fatura"""
    try:
        fatura = FaturaCartao.query.get_or_404(fatura_id)
        data = request.get_json()
        
        # Validações
        if not data.get('valor_pagamento') or float(data['valor_pagamento']) <= 0:
            return jsonify({'success': False, 'message': 'Valor do pagamento deve ser maior que zero'}), 400
        
        if not data.get('caixa_origem_id'):
            return jsonify({'success': False, 'message': 'Caixa de origem é obrigatório'}), 400
        
        valor_pagamento = Decimal(str(data['valor_pagamento']))
        
        # Verificar se não excede o valor pendente
        if valor_pagamento > fatura.valor_pendente:
            return jsonify({
                'success': False, 
                'message': f'Valor excede o pendente. Máximo: R$ {float(fatura.valor_pendente):.2f}'
            }), 400
        
        # Criar transação de pagamento
        transacao = Transacao(
            tipo='saida',
            subtipo='pagamento_fatura',
            data_transacao=datetime.strptime(data.get('data_pagamento', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=valor_pagamento,
            descricao=f'Pagamento fatura {fatura.cartao.nome} - {fatura.mes_referencia.strftime("%m/%Y")}',
            caixa_origem_id=data['caixa_origem_id'],
            fatura_cartao_id=fatura_id,
            metodo_pagamento=data.get('metodo_pagamento', 'pix'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Atualizar fatura
        fatura.valor_pago += valor_pagamento
        fatura.valor_pendente -= valor_pagamento
        
        # Atualizar status
        if fatura.valor_pago >= fatura.valor_total:
            fatura.status = 'paga_total'
        else:
            fatura.status = 'paga_parcial'
        
        # Liberar limite do cartão
        fatura.cartao.limite_disponivel += valor_pagamento
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Pagamento registrado com sucesso',
            'data': {
                'transacao': transacao.to_dict(),
                'fatura': fatura.to_dict()
            }
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cartoes_bp.route('/cartoes/resumo', methods=['GET'])
@jwt_required()
def resumo_cartoes():
    """Obtém resumo de todos os cartões"""
    try:
        cartoes = Cartao.query.filter_by(ativo=True).all()
        
        resumo = {
            'total_cartoes': len(cartoes),
            'limite_total': 0,
            'limite_usado': 0,
            'limite_disponivel': 0,
            'faturas_pendentes': 0,
            'valor_faturas_pendentes': 0
        }
        
        for cartao in cartoes:
            resumo['limite_total'] += float(cartao.limite_total)
            resumo['limite_usado'] += cartao.limite_usado()
            resumo['limite_disponivel'] += float(cartao.limite_disponivel)
        
        # Contar faturas pendentes
        faturas_pendentes = FaturaCartao.query.filter(
            FaturaCartao.status.in_(['aberta', 'paga_parcial', 'vencida'])
        ).all()
        
        resumo['faturas_pendentes'] = len(faturas_pendentes)
        resumo['valor_faturas_pendentes'] = sum(float(f.valor_pendente) for f in faturas_pendentes)
        
        # Calcular percentual de uso
        if resumo['limite_total'] > 0:
            resumo['percentual_uso'] = round((resumo['limite_usado'] / resumo['limite_total']) * 100, 2)
        else:
            resumo['percentual_uso'] = 0
        
        return jsonify({
            'success': True,
            'data': resumo
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

