"""
Rotas para gerenciamento de inadimplentes e cobrança
"""
from flask import Blueprint, request, jsonify, send_file
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Usuario, Representante
from src.models.cobranca import (
    Inadimplente, CampanhaCobranca, AcordoCobranca, ParcelaAcordo,
    HistoricoCobranca, ConfiguracaoDesconto
)
from datetime import datetime, date
from decimal import Decimal
from sqlalchemy import func, and_, or_
from dateutil.relativedelta import relativedelta
import pandas as pd
import io
import json

cobranca_bp = Blueprint('cobranca', __name__)

@cobranca_bp.route('/inadimplentes', methods=['GET'])
@jwt_required()
def listar_inadimplentes():
    """Lista inadimplentes com filtros"""
    try:
        # Parâmetros de filtro
        status = request.args.get('status', 'ativo')
        prioridade = request.args.get('prioridade')
        representante_id = request.args.get('representante_id')
        spc_negativado = request.args.get('spc_negativado')
        
        query = Inadimplente.query
        
        if status != 'todos':
            query = query.filter_by(status=status)
        
        if prioridade:
            query = query.filter_by(prioridade=prioridade)
        
        if representante_id:
            query = query.filter_by(representante_id=representante_id)
        
        if spc_negativado is not None:
            query = query.filter_by(spc_negativado=spc_negativado.lower() == 'true')
        
        inadimplentes = query.order_by(Inadimplente.saldo_atual.desc()).all()
        
        # Atualizar saldos com juros
        for inadimplente in inadimplentes:
            inadimplente.atualizar_saldo()
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'data': [inadimplente.to_dict() for inadimplente in inadimplentes]
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/inadimplentes', methods=['POST'])
@jwt_required()
def criar_inadimplente():
    """Cria um novo inadimplente"""
    try:
        data = request.get_json()
        
        # Validações obrigatórias
        campos_obrigatorios = ['nome_devedor', 'cpf', 'saldo_original', 'telefone_01']
        for campo in campos_obrigatorios:
            if not data.get(campo):
                return jsonify({'success': False, 'message': f'{campo} é obrigatório'}), 400
        
        # Verificar se CPF já existe
        if Inadimplente.query.filter_by(cpf=data['cpf'], status='ativo').first():
            return jsonify({'success': False, 'message': 'CPF já cadastrado como inadimplente ativo'}), 400
        
        # Verificar representante se fornecido
        if data.get('representante_id'):
            representante = Representante.query.get(data['representante_id'])
            if not representante or not representante.ativo:
                return jsonify({'success': False, 'message': 'Representante inválido'}), 400
        
        saldo_original = Decimal(str(data['saldo_original']))
        
        inadimplente = Inadimplente(
            operador_id=get_jwt_identity(),
            spc_negativado=data.get('spc_negativado', False),
            data_negativacao=datetime.strptime(data['data_negativacao'], '%Y-%m-%d').date() if data.get('data_negativacao') else None,
            cobranca_juridica=data.get('cobranca_juridica', False),
            representante_id=data.get('representante_id'),
            nome_devedor=data['nome_devedor'],
            cpf=data['cpf'],
            numero_contrato=data.get('numero_contrato'),
            saldo_original=saldo_original,
            saldo_atual=saldo_original,
            juros_multa_percentual=Decimal(str(data.get('juros_multa_percentual', 0))),
            telefone_01=data['telefone_01'],
            telefone_02=data.get('telefone_02'),
            cep=data.get('cep'),
            rua=data.get('rua'),
            numero_casa=data.get('numero_casa'),
            bairro=data.get('bairro'),
            cidade=data.get('cidade'),
            estado=data.get('estado'),
            prioridade=data.get('prioridade', 'media'),
            observacoes=data.get('observacoes')
        )
        
        db.session.add(inadimplente)
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Inadimplente cadastrado com sucesso',
            'data': inadimplente.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/inadimplentes/importar', methods=['POST'])
@jwt_required()
def importar_inadimplentes():
    """Importa inadimplentes via Excel"""
    try:
        if 'arquivo' not in request.files:
            return jsonify({'success': False, 'message': 'Arquivo não encontrado'}), 400
        
        arquivo = request.files['arquivo']
        if arquivo.filename == '':
            return jsonify({'success': False, 'message': 'Nenhum arquivo selecionado'}), 400
        
        # Ler Excel
        df = pd.read_excel(arquivo)
        
        # Validar colunas obrigatórias
        colunas_obrigatorias = ['nome_devedor', 'cpf', 'saldo_original', 'telefone_01']
        for coluna in colunas_obrigatorias:
            if coluna not in df.columns:
                return jsonify({'success': False, 'message': f'Coluna {coluna} não encontrada'}), 400
        
        inadimplentes_criados = 0
        erros = []
        
        for index, row in df.iterrows():
            try:
                # Verificar se CPF já existe
                if Inadimplente.query.filter_by(cpf=str(row['cpf']), status='ativo').first():
                    erros.append(f"Linha {index + 2}: CPF {row['cpf']} já cadastrado")
                    continue
                
                inadimplente = Inadimplente(
                    operador_id=get_jwt_identity(),
                    spc_negativado=bool(row.get('spc_negativado', False)),
                    cobranca_juridica=bool(row.get('cobranca_juridica', False)),
                    nome_devedor=str(row['nome_devedor']),
                    cpf=str(row['cpf']),
                    numero_contrato=str(row.get('numero_contrato', '')),
                    saldo_original=Decimal(str(row['saldo_original'])),
                    saldo_atual=Decimal(str(row['saldo_original'])),
                    juros_multa_percentual=Decimal(str(row.get('juros_multa_percentual', 0))),
                    telefone_01=str(row['telefone_01']),
                    telefone_02=str(row.get('telefone_02', '')),
                    cep=str(row.get('cep', '')),
                    rua=str(row.get('rua', '')),
                    numero_casa=str(row.get('numero_casa', '')),
                    bairro=str(row.get('bairro', '')),
                    cidade=str(row.get('cidade', '')),
                    estado=str(row.get('estado', '')),
                    observacoes=str(row.get('observacoes', ''))
                )
                
                db.session.add(inadimplente)
                inadimplentes_criados += 1
                
            except Exception as e:
                erros.append(f"Linha {index + 2}: {str(e)}")
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': f'{inadimplentes_criados} inadimplentes importados com sucesso',
            'data': {
                'criados': inadimplentes_criados,
                'erros': erros
            }
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/inadimplentes/template-excel', methods=['GET'])
@jwt_required()
def download_template_excel():
    """Download do template Excel para importação"""
    try:
        # Criar DataFrame com colunas do template
        template_data = {
            'nome_devedor': ['João Silva', 'Maria Santos'],
            'cpf': ['12345678901', '98765432100'],
            'saldo_original': [1500.00, 2300.50],
            'telefone_01': ['11999999999', '11888888888'],
            'telefone_02': ['', '11777777777'],
            'numero_contrato': ['CONT001', 'CONT002'],
            'spc_negativado': [True, False],
            'cobranca_juridica': [False, False],
            'juros_multa_percentual': [2.5, 3.0],
            'cep': ['01234567', '87654321'],
            'rua': ['Rua das Flores, 123', 'Av. Principal, 456'],
            'numero_casa': ['123', '456'],
            'bairro': ['Centro', 'Jardim'],
            'cidade': ['São Paulo', 'Rio de Janeiro'],
            'estado': ['SP', 'RJ'],
            'observacoes': ['Cliente antigo', 'Contato preferencial manhã']
        }
        
        df = pd.DataFrame(template_data)
        
        # Criar arquivo Excel em memória
        output = io.BytesIO()
        with pd.ExcelWriter(output, engine='openpyxl') as writer:
            df.to_excel(writer, sheet_name='Inadimplentes', index=False)
        
        output.seek(0)
        
        return send_file(
            output,
            mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            as_attachment=True,
            download_name='template_inadimplentes.xlsx'
        )
        
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/configuracao-descontos', methods=['GET'])
@jwt_required()
def obter_configuracao_descontos():
    """Obtém configuração ativa de descontos"""
    try:
        config = ConfiguracaoDesconto.query.filter_by(ativa=True).first()
        
        if not config:
            # Criar configuração padrão
            config = ConfiguracaoDesconto(
                nome='Configuração Padrão',
                faixa1_desconto=10.00,
                faixa2_desconto=15.00,
                faixa3_desconto=20.00,
                faixa4_desconto=25.00,
                faixa5_desconto=30.00,
                faixa6_desconto=35.00
            )
            db.session.add(config)
            db.session.commit()
        
        return jsonify({
            'success': True,
            'data': config.to_dict()
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/configuracao-descontos', methods=['PUT'])
@jwt_required()
def atualizar_configuracao_descontos():
    """Atualiza configuração de descontos"""
    try:
        data = request.get_json()
        
        config = ConfiguracaoDesconto.query.filter_by(ativa=True).first()
        if not config:
            return jsonify({'success': False, 'message': 'Configuração não encontrada'}), 404
        
        # Atualizar faixas
        faixas = data.get('faixas', [])
        for i, faixa in enumerate(faixas, 1):
            setattr(config, f'faixa{i}_min', Decimal(str(faixa['min'])))
            setattr(config, f'faixa{i}_max', Decimal(str(faixa['max'])))
            setattr(config, f'faixa{i}_desconto', Decimal(str(faixa['desconto'])))
        
        # Atualizar outras configurações
        if 'eliminar_juros_disponivel' in data:
            config.eliminar_juros_disponivel = data['eliminar_juros_disponivel']
        if 'desconto_maximo_automatico' in data:
            config.desconto_maximo_automatico = Decimal(str(data['desconto_maximo_automatico']))
        if 'priorizar_pagamento_vista' in data:
            config.priorizar_pagamento_vista = data['priorizar_pagamento_vista']
        if 'tentativas_vista_antes_parcelamento' in data:
            config.tentativas_vista_antes_parcelamento = data['tentativas_vista_antes_parcelamento']
        if 'chave_pix' in data:
            config.chave_pix = data['chave_pix']
        if 'dados_bancarios' in data:
            config.dados_bancarios = data['dados_bancarios']
        
        config.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Configuração atualizada com sucesso',
            'data': config.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/acordos', methods=['POST'])
@jwt_required()
def criar_acordo():
    """Cria um novo acordo de cobrança"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('inadimplente_id'):
            return jsonify({'success': False, 'message': 'Inadimplente é obrigatório'}), 400
        
        if not data.get('valor_acordo') or float(data['valor_acordo']) <= 0:
            return jsonify({'success': False, 'message': 'Valor do acordo deve ser maior que zero'}), 400
        
        inadimplente = Inadimplente.query.get(data['inadimplente_id'])
        if not inadimplente:
            return jsonify({'success': False, 'message': 'Inadimplente não encontrado'}), 404
        
        valor_original = inadimplente.saldo_atual
        valor_acordo = Decimal(str(data['valor_acordo']))
        desconto_valor = valor_original - valor_acordo
        desconto_percentual = (desconto_valor / valor_original) * 100 if valor_original > 0 else 0
        
        # Verificar se é parcelado
        tipo_acordo = data.get('tipo_acordo', 'a_vista')
        numero_parcelas = int(data.get('numero_parcelas', 1)) if tipo_acordo == 'parcelado' else 1
        
        acordo = AcordoCobranca(
            inadimplente_id=data['inadimplente_id'],
            campanha_id=data.get('campanha_id'),
            valor_original=valor_original,
            valor_acordo=valor_acordo,
            desconto_percentual=desconto_percentual,
            desconto_valor=desconto_valor,
            tipo_acordo=tipo_acordo,
            numero_parcelas=numero_parcelas,
            valor_entrada=Decimal(str(data.get('valor_entrada', 0))),
            chave_pix=data.get('chave_pix'),
            dados_bancarios=data.get('dados_bancarios'),
            data_vencimento=datetime.strptime(data['data_vencimento'], '%Y-%m-%d').date() if data.get('data_vencimento') else None,
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(acordo)
        db.session.flush()  # Para obter o ID
        
        # Criar parcelas se for parcelado
        if tipo_acordo == 'parcelado' and numero_parcelas > 1:
            valor_parcela = (valor_acordo - acordo.valor_entrada) / (numero_parcelas - 1) if acordo.valor_entrada > 0 else valor_acordo / numero_parcelas
            data_vencimento = acordo.data_vencimento or date.today()
            
            for i in range(numero_parcelas):
                if i == 0 and acordo.valor_entrada > 0:
                    # Primeira parcela é a entrada
                    valor_parcela_atual = acordo.valor_entrada
                    data_vencimento_atual = data_vencimento
                else:
                    valor_parcela_atual = valor_parcela
                    data_vencimento_atual = data_vencimento + relativedelta(months=i if acordo.valor_entrada == 0 else i)
                
                parcela = ParcelaAcordo(
                    acordo_id=acordo.id,
                    numero_parcela=i + 1,
                    data_vencimento=data_vencimento_atual,
                    valor_parcela=valor_parcela_atual
                )
                db.session.add(parcela)
        else:
            # Acordo à vista - uma parcela única
            parcela = ParcelaAcordo(
                acordo_id=acordo.id,
                numero_parcela=1,
                data_vencimento=acordo.data_vencimento or date.today(),
                valor_parcela=valor_acordo
            )
            db.session.add(parcela)
        
        # Atualizar status do inadimplente
        inadimplente.status = 'acordo_fechado'
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Acordo criado com sucesso',
            'data': acordo.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/relatorios/performance-cobranca', methods=['GET'])
@jwt_required()
def relatorio_performance_cobranca():
    """Relatório de performance de cobrança"""
    try:
        # Parâmetros
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        
        if not data_inicio or not data_fim:
            return jsonify({'success': False, 'message': 'Período é obrigatório'}), 400
        
        data_inicio = datetime.strptime(data_inicio, '%Y-%m-%d').date()
        data_fim = datetime.strptime(data_fim, '%Y-%m-%d').date()
        
        # Inadimplentes no período
        total_inadimplentes = Inadimplente.query.filter(
            Inadimplente.data_registro >= data_inicio,
            Inadimplente.data_registro <= data_fim
        ).count()
        
        # Acordos fechados no período
        acordos = AcordoCobranca.query.filter(
            AcordoCobranca.data_acordo >= data_inicio,
            AcordoCobranca.data_acordo <= data_fim
        ).all()
        
        total_acordos = len(acordos)
        valor_total_cobrado = sum(float(acordo.valor_original) for acordo in acordos)
        valor_total_recuperado = sum(float(acordo.valor_acordo) for acordo in acordos)
        valor_total_desconto = sum(float(acordo.desconto_valor) for acordo in acordos)
        
        # Taxa de conversão
        taxa_conversao = (total_acordos / total_inadimplentes * 100) if total_inadimplentes > 0 else 0
        
        # Acordos à vista vs parcelados
        acordos_vista = len([a for a in acordos if a.tipo_acordo == 'a_vista'])
        acordos_parcelados = len([a for a in acordos if a.tipo_acordo == 'parcelado'])
        
        # Desconto médio
        desconto_medio = (valor_total_desconto / valor_total_cobrado * 100) if valor_total_cobrado > 0 else 0
        
        # Valor médio recuperado
        valor_medio_recuperado = valor_total_recuperado / total_acordos if total_acordos > 0 else 0
        
        # Performance por representante
        performance_representantes = db.session.query(
            Representante.nome,
            func.count(AcordoCobranca.id).label('total_acordos'),
            func.sum(AcordoCobranca.valor_original).label('valor_cobrado'),
            func.sum(AcordoCobranca.valor_acordo).label('valor_recuperado')
        ).join(
            Inadimplente, Inadimplente.representante_id == Representante.id
        ).join(
            AcordoCobranca, AcordoCobranca.inadimplente_id == Inadimplente.id
        ).filter(
            AcordoCobranca.data_acordo >= data_inicio,
            AcordoCobranca.data_acordo <= data_fim
        ).group_by(
            Representante.nome
        ).all()
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {
                    'inicio': data_inicio.isoformat(),
                    'fim': data_fim.isoformat()
                },
                'resumo_geral': {
                    'total_inadimplentes': total_inadimplentes,
                    'total_acordos': total_acordos,
                    'taxa_conversao': round(taxa_conversao, 2),
                    'valor_total_cobrado': valor_total_cobrado,
                    'valor_total_recuperado': valor_total_recuperado,
                    'valor_total_desconto': valor_total_desconto,
                    'percentual_recuperacao': round((valor_total_recuperado / valor_total_cobrado * 100), 2) if valor_total_cobrado > 0 else 0,
                    'desconto_medio': round(desconto_medio, 2),
                    'valor_medio_recuperado': round(valor_medio_recuperado, 2)
                },
                'tipos_acordo': {
                    'acordos_vista': acordos_vista,
                    'acordos_parcelados': acordos_parcelados,
                    'percentual_vista': round((acordos_vista / total_acordos * 100), 2) if total_acordos > 0 else 0
                },
                'performance_representantes': [
                    {
                        'nome': rep.nome,
                        'total_acordos': rep.total_acordos,
                        'valor_cobrado': float(rep.valor_cobrado),
                        'valor_recuperado': float(rep.valor_recuperado),
                        'taxa_recuperacao': round((float(rep.valor_recuperado) / float(rep.valor_cobrado) * 100), 2) if rep.valor_cobrado > 0 else 0
                    } for rep in performance_representantes
                ]
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/relatorios/parcelas-acordo', methods=['GET'])
@jwt_required()
def relatorio_parcelas_acordo():
    """Relatório de controle de parcelas de acordos"""
    try:
        # Parcelas pendentes
        parcelas_pendentes = ParcelaAcordo.query.filter_by(status='pendente').count()
        
        # Parcelas atrasadas
        parcelas_atrasadas = ParcelaAcordo.query.filter(
            ParcelaAcordo.status == 'pendente',
            ParcelaAcordo.data_vencimento < date.today()
        ).count()
        
        # Parcelas do mês
        inicio_mes = date.today().replace(day=1)
        fim_mes = inicio_mes + relativedelta(months=1) - relativedelta(days=1)
        
        parcelas_mes = ParcelaAcordo.query.filter(
            ParcelaAcordo.data_vencimento >= inicio_mes,
            ParcelaAcordo.data_vencimento <= fim_mes
        ).all()
        
        valor_parcelas_mes = sum(float(p.valor_parcela) for p in parcelas_mes)
        parcelas_pagas_mes = len([p for p in parcelas_mes if p.status in ['pago', 'validado']])
        
        # Taxa de pagamento
        total_parcelas_vencidas = ParcelaAcordo.query.filter(
            ParcelaAcordo.data_vencimento < date.today()
        ).count()
        
        parcelas_pagas_vencidas = ParcelaAcordo.query.filter(
            ParcelaAcordo.data_vencimento < date.today(),
            ParcelaAcordo.status.in_(['pago', 'validado'])
        ).count()
        
        taxa_pagamento = (parcelas_pagas_vencidas / total_parcelas_vencidas * 100) if total_parcelas_vencidas > 0 else 0
        
        # Próximos vencimentos (próximos 7 dias)
        proximos_vencimentos = ParcelaAcordo.query.filter(
            ParcelaAcordo.status == 'pendente',
            ParcelaAcordo.data_vencimento >= date.today(),
            ParcelaAcordo.data_vencimento <= date.today() + relativedelta(days=7)
        ).order_by(ParcelaAcordo.data_vencimento).all()
        
        return jsonify({
            'success': True,
            'data': {
                'resumo': {
                    'parcelas_pendentes': parcelas_pendentes,
                    'parcelas_atrasadas': parcelas_atrasadas,
                    'valor_parcelas_mes': valor_parcelas_mes,
                    'parcelas_pagas_mes': parcelas_pagas_mes,
                    'taxa_pagamento': round(taxa_pagamento, 2)
                },
                'proximos_vencimentos': [parcela.to_dict() for parcela in proximos_vencimentos]
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@cobranca_bp.route('/dashboard-cobranca', methods=['GET'])
@jwt_required()
def dashboard_cobranca():
    """Dashboard principal de cobrança"""
    try:
        hoje = date.today()
        inicio_mes = hoje.replace(day=1)
        
        # Métricas principais
        total_inadimplentes_ativos = Inadimplente.query.filter_by(status='ativo').count()
        valor_total_inadimplencia = Inadimplente.query.filter_by(status='ativo').with_entities(
            func.sum(Inadimplente.saldo_atual)
        ).scalar() or 0
        
        # Acordos do mês
        acordos_mes = AcordoCobranca.query.filter(
            AcordoCobranca.data_acordo >= inicio_mes
        ).count()
        
        valor_recuperado_mes = AcordoCobranca.query.filter(
            AcordoCobranca.data_acordo >= inicio_mes
        ).with_entities(func.sum(AcordoCobranca.valor_acordo)).scalar() or 0
        
        # Taxa de conversão do mês
        inadimplentes_mes = Inadimplente.query.filter(
            Inadimplente.data_registro >= inicio_mes
        ).count()
        
        taxa_conversao_mes = (acordos_mes / inadimplentes_mes * 100) if inadimplentes_mes > 0 else 0
        
        # Parcelas vencendo hoje
        parcelas_hoje = ParcelaAcordo.query.filter(
            ParcelaAcordo.data_vencimento == hoje,
            ParcelaAcordo.status == 'pendente'
        ).count()
        
        # Top 5 maiores inadimplências
        top_inadimplentes = Inadimplente.query.filter_by(status='ativo').order_by(
            Inadimplente.saldo_atual.desc()
        ).limit(5).all()
        
        return jsonify({
            'success': True,
            'data': {
                'metricas_principais': {
                    'total_inadimplentes_ativos': total_inadimplentes_ativos,
                    'valor_total_inadimplencia': float(valor_total_inadimplencia),
                    'acordos_mes': acordos_mes,
                    'valor_recuperado_mes': float(valor_recuperado_mes),
                    'taxa_conversao_mes': round(taxa_conversao_mes, 2),
                    'parcelas_vencendo_hoje': parcelas_hoje
                },
                'top_inadimplentes': [inadimplente.to_dict() for inadimplente in top_inadimplentes]
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

