"""
Rotas para gerenciamento de dívidas e parcelas
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Categoria, Subcategoria, Caixa, Transacao
from src.models.avancados import Divida, ParcelaDivida
from datetime import datetime, date
from decimal import Decimal
from dateutil.relativedelta import relativedelta
from sqlalchemy import func

dividas_bp = Blueprint('dividas', __name__)

@dividas_bp.route('/dividas', methods=['GET'])
@jwt_required()
def listar_dividas():
    """Lista todas as dívidas"""
    try:
        status = request.args.get('status', 'ativa')
        
        dividas = Divida.query.filter_by(status=status).order_by(Divida.nome).all()
        
        return jsonify({
            'success': True,
            'data': [divida.to_dict() for divida in dividas]
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/dividas/<int:divida_id>', methods=['GET'])
@jwt_required()
def obter_divida(divida_id):
    """Obtém detalhes de uma dívida específica"""
    try:
        divida = Divida.query.get_or_404(divida_id)
        
        # Incluir parcelas
        parcelas = ParcelaDivida.query.filter_by(divida_id=divida_id).order_by(
            ParcelaDivida.numero_parcela
        ).all()
        
        data = divida.to_dict()
        data['parcelas'] = [parcela.to_dict() for parcela in parcelas]
        
        return jsonify({
            'success': True,
            'data': data
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/dividas', methods=['POST'])
@jwt_required()
def criar_divida():
    """Cria uma nova dívida"""
    try:
        data = request.get_json()
        
        # Validações
        if not data.get('nome'):
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        if not data.get('valor_total') or float(data['valor_total']) <= 0:
            return jsonify({'success': False, 'message': 'Valor total deve ser maior que zero'}), 400
        
        if not data.get('numero_parcelas') or int(data['numero_parcelas']) <= 0:
            return jsonify({'success': False, 'message': 'Número de parcelas deve ser maior que zero'}), 400
        
        if not data.get('data_primeira_parcela'):
            return jsonify({'success': False, 'message': 'Data da primeira parcela é obrigatória'}), 400
        
        # Verificar categoria se fornecida
        if data.get('categoria_id'):
            categoria = Categoria.query.get(data['categoria_id'])
            if not categoria or not categoria.ativo:
                return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar subcategoria se fornecida
        if data.get('subcategoria_id'):
            subcategoria = Subcategoria.query.get(data['subcategoria_id'])
            if not subcategoria or not subcategoria.ativo:
                return jsonify({'success': False, 'message': 'Subcategoria inválida'}), 400
            if data.get('categoria_id') and subcategoria.categoria_id != data['categoria_id']:
                return jsonify({'success': False, 'message': 'Subcategoria não pertence à categoria selecionada'}), 400
        
        valor_total = Decimal(str(data['valor_total']))
        numero_parcelas = int(data['numero_parcelas'])
        valor_parcela = valor_total / numero_parcelas
        
        # Criar dívida
        divida = Divida(
            nome=data['nome'],
            descricao=data.get('descricao'),
            valor_total=valor_total,
            valor_pendente=valor_total,
            numero_parcelas=numero_parcelas,
            valor_parcela=valor_parcela,
            data_primeira_parcela=datetime.strptime(data['data_primeira_parcela'], '%Y-%m-%d').date(),
            periodicidade=data.get('periodicidade', 'mensal'),
            categoria_id=data.get('categoria_id'),
            subcategoria_id=data.get('subcategoria_id'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(divida)
        db.session.flush()  # Para obter o ID da dívida
        
        # Criar parcelas
        data_parcela = divida.data_primeira_parcela
        for i in range(1, numero_parcelas + 1):
            parcela = ParcelaDivida(
                divida_id=divida.id,
                numero_parcela=i,
                data_vencimento=data_parcela,
                valor_previsto=valor_parcela
            )
            
            db.session.add(parcela)
            
            # Calcular próxima data baseada na periodicidade
            if divida.periodicidade == 'semanal':
                data_parcela += relativedelta(weeks=1)
            elif divida.periodicidade == 'quinzenal':
                data_parcela += relativedelta(weeks=2)
            elif divida.periodicidade == 'mensal':
                data_parcela += relativedelta(months=1)
            elif divida.periodicidade == 'bimestral':
                data_parcela += relativedelta(months=2)
            elif divida.periodicidade == 'trimestral':
                data_parcela += relativedelta(months=3)
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Dívida criada com sucesso',
            'data': divida.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/dividas/<int:divida_id>', methods=['PUT'])
@jwt_required()
def atualizar_divida(divida_id):
    """Atualiza uma dívida existente"""
    try:
        divida = Divida.query.get_or_404(divida_id)
        data = request.get_json()
        
        # Validações
        if 'nome' in data and not data['nome']:
            return jsonify({'success': False, 'message': 'Nome é obrigatório'}), 400
        
        # Verificar categoria se fornecida
        if 'categoria_id' in data and data['categoria_id']:
            categoria = Categoria.query.get(data['categoria_id'])
            if not categoria or not categoria.ativo:
                return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar subcategoria se fornecida
        if 'subcategoria_id' in data and data['subcategoria_id']:
            subcategoria = Subcategoria.query.get(data['subcategoria_id'])
            if not subcategoria or not subcategoria.ativo:
                return jsonify({'success': False, 'message': 'Subcategoria inválida'}), 400
        
        # Atualizar campos
        if 'nome' in data:
            divida.nome = data['nome']
        if 'descricao' in data:
            divida.descricao = data['descricao']
        if 'categoria_id' in data:
            divida.categoria_id = data['categoria_id']
        if 'subcategoria_id' in data:
            divida.subcategoria_id = data['subcategoria_id']
        if 'observacoes' in data:
            divida.observacoes = data['observacoes']
        if 'status' in data:
            divida.status = data['status']
        
        divida.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Dívida atualizada com sucesso',
            'data': divida.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/parcelas/<int:parcela_id>/pagar', methods=['POST'])
@jwt_required()
def pagar_parcela(parcela_id):
    """Registra pagamento de uma parcela"""
    try:
        parcela = ParcelaDivida.query.get_or_404(parcela_id)
        data = request.get_json()
        
        # Validações
        if parcela.status == 'paga':
            return jsonify({'success': False, 'message': 'Parcela já foi paga'}), 400
        
        if not data.get('valor_pago') or float(data['valor_pago']) <= 0:
            return jsonify({'success': False, 'message': 'Valor pago deve ser maior que zero'}), 400
        
        if not data.get('caixa_pagamento_id'):
            return jsonify({'success': False, 'message': 'Caixa de pagamento é obrigatório'}), 400
        
        # Verificar se o caixa existe
        caixa = Caixa.query.get(data['caixa_pagamento_id'])
        if not caixa or not caixa.ativo:
            return jsonify({'success': False, 'message': 'Caixa inválido'}), 400
        
        valor_pago = Decimal(str(data['valor_pago']))
        
        # Verificar se há saldo suficiente no caixa
        if valor_pago > caixa.saldo_atual:
            return jsonify({
                'success': False, 
                'message': f'Saldo insuficiente no caixa. Disponível: R$ {float(caixa.saldo_atual):.2f}'
            }), 400
        
        # Criar transação de pagamento
        transacao = Transacao(
            tipo='saida',
            subtipo='pagamento_divida',
            data_transacao=datetime.strptime(data.get('data_pagamento', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=valor_pago,
            descricao=f'Pagamento {parcela.divida.nome} - Parcela {parcela.numero_parcela}/{parcela.divida.numero_parcelas}',
            caixa_origem_id=data['caixa_pagamento_id'],
            categoria_id=parcela.divida.categoria_id,
            subcategoria_id=parcela.divida.subcategoria_id,
            metodo_pagamento=data.get('metodo_pagamento', 'pix'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Atualizar parcela
        parcela.valor_pago = valor_pago
        parcela.data_pagamento = transacao.data_transacao
        parcela.caixa_pagamento_id = data['caixa_pagamento_id']
        parcela.status = 'paga'
        parcela.observacoes = data.get('observacoes')
        parcela.atualizado_em = datetime.utcnow()
        
        # Atualizar dívida
        divida = parcela.divida
        divida.valor_pago += valor_pago
        divida.valor_pendente -= valor_pago
        
        # Verificar se a dívida foi quitada
        if divida.valor_pago >= divida.valor_total:
            divida.status = 'quitada'
        
        # Atualizar saldo do caixa
        caixa.saldo_atual -= valor_pago
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Pagamento registrado com sucesso',
            'data': {
                'transacao': transacao.to_dict(),
                'parcela': parcela.to_dict(),
                'divida': divida.to_dict()
            }
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/parcelas/vencimentos', methods=['GET'])
@jwt_required()
def proximos_vencimentos():
    """Lista próximos vencimentos de parcelas"""
    try:
        dias = int(request.args.get('dias', 30))
        
        data_limite = date.today() + relativedelta(days=dias)
        
        parcelas = ParcelaDivida.query.join(Divida).filter(
            ParcelaDivida.status == 'pendente',
            ParcelaDivida.data_vencimento <= data_limite,
            Divida.status == 'ativa'
        ).order_by(ParcelaDivida.data_vencimento).all()
        
        # Agrupar por data
        vencimentos = {}
        for parcela in parcelas:
            data_str = parcela.data_vencimento.isoformat()
            if data_str not in vencimentos:
                vencimentos[data_str] = []
            vencimentos[data_str].append(parcela.to_dict())
        
        return jsonify({
            'success': True,
            'data': {
                'periodo_dias': dias,
                'total_parcelas': len(parcelas),
                'valor_total': sum(float(p.valor_previsto) for p in parcelas),
                'vencimentos': vencimentos
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/parcelas/atrasadas', methods=['GET'])
@jwt_required()
def parcelas_atrasadas():
    """Lista parcelas em atraso"""
    try:
        parcelas = ParcelaDivida.query.join(Divida).filter(
            ParcelaDivida.status == 'pendente',
            ParcelaDivida.data_vencimento < date.today(),
            Divida.status == 'ativa'
        ).order_by(ParcelaDivida.data_vencimento).all()
        
        # Atualizar status para atrasada
        for parcela in parcelas:
            if parcela.status == 'pendente':
                parcela.status = 'atrasada'
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'data': {
                'total_parcelas': len(parcelas),
                'valor_total': sum(float(p.valor_previsto) for p in parcelas),
                'parcelas': [parcela.to_dict() for parcela in parcelas]
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@dividas_bp.route('/dividas/resumo', methods=['GET'])
@jwt_required()
def resumo_dividas():
    """Obtém resumo de todas as dívidas"""
    try:
        # Dívidas ativas
        dividas_ativas = Divida.query.filter_by(status='ativa').all()
        
        resumo = {
            'total_dividas': len(dividas_ativas),
            'valor_total': sum(float(d.valor_total) for d in dividas_ativas),
            'valor_pago': sum(float(d.valor_pago) for d in dividas_ativas),
            'valor_pendente': sum(float(d.valor_pendente) for d in dividas_ativas),
            'parcelas_pendentes': 0,
            'parcelas_atrasadas': 0,
            'valor_parcelas_mes': 0
        }
        
        # Calcular percentual pago
        if resumo['valor_total'] > 0:
            resumo['percentual_pago'] = round((resumo['valor_pago'] / resumo['valor_total']) * 100, 2)
        else:
            resumo['percentual_pago'] = 0
        
        # Contar parcelas
        parcelas_pendentes = ParcelaDivida.query.join(Divida).filter(
            ParcelaDivida.status.in_(['pendente', 'atrasada']),
            Divida.status == 'ativa'
        ).all()
        
        resumo['parcelas_pendentes'] = len([p for p in parcelas_pendentes if p.status == 'pendente'])
        resumo['parcelas_atrasadas'] = len([p for p in parcelas_pendentes if p.status == 'atrasada'])
        
        # Parcelas do mês atual
        inicio_mes = date.today().replace(day=1)
        fim_mes = inicio_mes + relativedelta(months=1) - relativedelta(days=1)
        
        parcelas_mes = ParcelaDivida.query.join(Divida).filter(
            ParcelaDivida.data_vencimento >= inicio_mes,
            ParcelaDivida.data_vencimento <= fim_mes,
            ParcelaDivida.status.in_(['pendente', 'atrasada']),
            Divida.status == 'ativa'
        ).all()
        
        resumo['valor_parcelas_mes'] = sum(float(p.valor_previsto) for p in parcelas_mes)
        
        return jsonify({
            'success': True,
            'data': resumo
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

