"""
Rotas para relatórios avançados e análises
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required
from src.models.core import db, Transacao, Caixa, Representante, Cliente
from src.models.avancados import Cartao, Divida, ParcelaDivida
from datetime import datetime, date
from decimal import Decimal
from sqlalchemy import func, extract, and_, or_
from dateutil.relativedelta import relativedelta
import calendar

relatorios_bp = Blueprint('relatorios', __name__)

@relatorios_bp.route('/relatorios/dashboard', methods=['GET'])
@jwt_required()
def dashboard_principal():
    """Dados principais para o dashboard"""
    try:
        # Período padrão: mês atual
        hoje = date.today()
        inicio_mes = hoje.replace(day=1)
        fim_mes = inicio_mes + relativedelta(months=1) - relativedelta(days=1)
        
        # Saldo total dos caixas
        saldo_total = Caixa.query.filter_by(ativo=True).with_entities(
            func.sum(Caixa.saldo_atual)
        ).scalar() or 0
        
        # Faturamento do mês (entradas)
        faturamento_mes = Transacao.query.filter(
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= inicio_mes,
            Transacao.data_transacao <= fim_mes
        ).with_entities(func.sum(Transacao.valor)).scalar() or 0
        
        # Despesas do mês (saídas)
        despesas_mes = Transacao.query.filter(
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= inicio_mes,
            Transacao.data_transacao <= fim_mes
        ).with_entities(func.sum(Transacao.valor)).scalar() or 0
        
        # Inadimplência (valor total de dívidas ativas)
        inadimplencia = Divida.query.filter_by(status='ativa').with_entities(
            func.sum(Divida.valor_pendente)
        ).scalar() or 0
        
        # Comparação com mês anterior
        inicio_mes_anterior = inicio_mes - relativedelta(months=1)
        fim_mes_anterior = inicio_mes - relativedelta(days=1)
        
        faturamento_mes_anterior = Transacao.query.filter(
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= inicio_mes_anterior,
            Transacao.data_transacao <= fim_mes_anterior
        ).with_entities(func.sum(Transacao.valor)).scalar() or 0
        
        # Calcular crescimento
        if float(faturamento_mes_anterior) > 0:
            crescimento = ((float(faturamento_mes) - float(faturamento_mes_anterior)) / float(faturamento_mes_anterior)) * 100
        else:
            crescimento = 0
        
        # Top 3 representantes do mês
        top_representantes = db.session.query(
            Representante.id,
            Representante.nome,
            func.sum(Transacao.valor).label('total')
        ).join(
            Transacao, Transacao.representante_id == Representante.id
        ).filter(
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= inicio_mes,
            Transacao.data_transacao <= fim_mes
        ).group_by(
            Representante.id, Representante.nome
        ).order_by(
            func.sum(Transacao.valor).desc()
        ).limit(3).all()
        
        return jsonify({
            'success': True,
            'data': {
                'saldo_total': float(saldo_total),
                'faturamento_mes': float(faturamento_mes),
                'despesas_mes': float(despesas_mes),
                'saldo_liquido': float(faturamento_mes) - float(despesas_mes),
                'inadimplencia': float(inadimplencia),
                'crescimento_percentual': round(crescimento, 2),
                'top_representantes': [
                    {
                        'id': rep.id,
                        'nome': rep.nome,
                        'total': float(rep.total)
                    } for rep in top_representantes
                ],
                'periodo': {
                    'inicio': inicio_mes.isoformat(),
                    'fim': fim_mes.isoformat()
                }
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@relatorios_bp.route('/relatorios/fluxo-caixa', methods=['GET'])
@jwt_required()
def relatorio_fluxo_caixa():
    """Relatório de fluxo de caixa por período"""
    try:
        # Parâmetros
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        caixa_id = request.args.get('caixa_id')
        
        if not data_inicio or not data_fim:
            return jsonify({'success': False, 'message': 'Período é obrigatório'}), 400
        
        data_inicio = datetime.strptime(data_inicio, '%Y-%m-%d').date()
        data_fim = datetime.strptime(data_fim, '%Y-%m-%d').date()
        
        # Query base
        query = Transacao.query.filter(
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= data_inicio,
            Transacao.data_transacao <= data_fim
        )
        
        if caixa_id:
            query = query.filter(
                or_(
                    Transacao.caixa_origem_id == caixa_id,
                    Transacao.caixa_destino_id == caixa_id
                )
            )
        
        transacoes = query.order_by(Transacao.data_transacao.desc()).all()
        
        # Agrupar por data
        fluxo_diario = {}
        total_entradas = 0
        total_saidas = 0
        
        for transacao in transacoes:
            data_str = transacao.data_transacao.isoformat()
            
            if data_str not in fluxo_diario:
                fluxo_diario[data_str] = {
                    'data': data_str,
                    'entradas': 0,
                    'saidas': 0,
                    'saldo': 0,
                    'transacoes': []
                }
            
            if transacao.tipo == 'entrada':
                fluxo_diario[data_str]['entradas'] += float(transacao.valor)
                total_entradas += float(transacao.valor)
            else:
                fluxo_diario[data_str]['saidas'] += float(transacao.valor)
                total_saidas += float(transacao.valor)
            
            fluxo_diario[data_str]['transacoes'].append(transacao.to_dict())
        
        # Calcular saldo diário
        for data_str in sorted(fluxo_diario.keys()):
            fluxo_diario[data_str]['saldo'] = fluxo_diario[data_str]['entradas'] - fluxo_diario[data_str]['saidas']
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {
                    'inicio': data_inicio.isoformat(),
                    'fim': data_fim.isoformat()
                },
                'resumo': {
                    'total_entradas': total_entradas,
                    'total_saidas': total_saidas,
                    'saldo_liquido': total_entradas - total_saidas
                },
                'fluxo_diario': list(fluxo_diario.values())
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@relatorios_bp.route('/relatorios/representantes', methods=['GET'])
@jwt_required()
def relatorio_representantes():
    """Relatório de performance dos representantes"""
    try:
        # Parâmetros
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        
        if not data_inicio or not data_fim:
            return jsonify({'success': False, 'message': 'Período é obrigatório'}), 400
        
        data_inicio = datetime.strptime(data_inicio, '%Y-%m-%d').date()
        data_fim = datetime.strptime(data_fim, '%Y-%m-%d').date()
        
        # Performance por representante
        performance = db.session.query(
            Representante.id,
            Representante.nome,
            Representante.codigo,
            func.count(Transacao.id).label('total_transacoes'),
            func.sum(Transacao.valor).label('total_valor'),
            func.avg(Transacao.valor).label('ticket_medio'),
            func.min(Transacao.data_transacao).label('primeira_transacao'),
            func.max(Transacao.data_transacao).label('ultima_transacao')
        ).join(
            Transacao, Transacao.representante_id == Representante.id
        ).filter(
            Transacao.tipo == 'entrada',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= data_inicio,
            Transacao.data_transacao <= data_fim
        ).group_by(
            Representante.id, Representante.nome, Representante.codigo
        ).order_by(
            func.sum(Transacao.valor).desc()
        ).all()
        
        # Calcular total geral para percentuais
        total_geral = sum(float(p.total_valor) for p in performance)
        
        # Formatar dados
        dados_representantes = []
        for p in performance:
            percentual = (float(p.total_valor) / total_geral * 100) if total_geral > 0 else 0
            
            dados_representantes.append({
                'id': p.id,
                'nome': p.nome,
                'codigo': p.codigo,
                'total_transacoes': p.total_transacoes,
                'total_valor': float(p.total_valor),
                'ticket_medio': float(p.ticket_medio),
                'percentual_total': round(percentual, 2),
                'primeira_transacao': p.primeira_transacao.isoformat(),
                'ultima_transacao': p.ultima_transacao.isoformat()
            })
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {
                    'inicio': data_inicio.isoformat(),
                    'fim': data_fim.isoformat()
                },
                'resumo': {
                    'total_representantes': len(dados_representantes),
                    'total_geral': total_geral,
                    'ticket_medio_geral': total_geral / len(dados_representantes) if dados_representantes else 0
                },
                'representantes': dados_representantes
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@relatorios_bp.route('/relatorios/comparativo-mensal', methods=['GET'])
@jwt_required()
def relatorio_comparativo_mensal():
    """Relatório comparativo entre meses"""
    try:
        # Parâmetros
        mes1 = request.args.get('mes1')  # YYYY-MM
        mes2 = request.args.get('mes2')  # YYYY-MM
        
        if not mes1 or not mes2:
            return jsonify({'success': False, 'message': 'Dois meses são obrigatórios para comparação'}), 400
        
        # Converter para datas
        ano1, mes_num1 = map(int, mes1.split('-'))
        ano2, mes_num2 = map(int, mes2.split('-'))
        
        inicio_mes1 = date(ano1, mes_num1, 1)
        fim_mes1 = date(ano1, mes_num1, calendar.monthrange(ano1, mes_num1)[1])
        
        inicio_mes2 = date(ano2, mes_num2, 1)
        fim_mes2 = date(ano2, mes_num2, calendar.monthrange(ano2, mes_num2)[1])
        
        def obter_dados_mes(inicio, fim):
            entradas = Transacao.query.filter(
                Transacao.tipo == 'entrada',
                Transacao.status == 'confirmado',
                Transacao.data_transacao >= inicio,
                Transacao.data_transacao <= fim
            ).with_entities(func.sum(Transacao.valor)).scalar() or 0
            
            saidas = Transacao.query.filter(
                Transacao.tipo == 'saida',
                Transacao.status == 'confirmado',
                Transacao.data_transacao >= inicio,
                Transacao.data_transacao <= fim
            ).with_entities(func.sum(Transacao.valor)).scalar() or 0
            
            # Performance por representante
            representantes = db.session.query(
                Representante.nome,
                func.sum(Transacao.valor).label('total')
            ).join(
                Transacao, Transacao.representante_id == Representante.id
            ).filter(
                Transacao.tipo == 'entrada',
                Transacao.status == 'confirmado',
                Transacao.data_transacao >= inicio,
                Transacao.data_transacao <= fim
            ).group_by(
                Representante.nome
            ).all()
            
            return {
                'entradas': float(entradas),
                'saidas': float(saidas),
                'saldo_liquido': float(entradas) - float(saidas),
                'representantes': {rep.nome: float(rep.total) for rep in representantes}
            }
        
        dados_mes1 = obter_dados_mes(inicio_mes1, fim_mes1)
        dados_mes2 = obter_dados_mes(inicio_mes2, fim_mes2)
        
        # Calcular variações
        def calcular_variacao(valor1, valor2):
            if valor2 != 0:
                return ((valor1 - valor2) / valor2) * 100
            return 0
        
        variacao_entradas = calcular_variacao(dados_mes1['entradas'], dados_mes2['entradas'])
        variacao_saidas = calcular_variacao(dados_mes1['saidas'], dados_mes2['saidas'])
        variacao_saldo = calcular_variacao(dados_mes1['saldo_liquido'], dados_mes2['saldo_liquido'])
        
        return jsonify({
            'success': True,
            'data': {
                'mes1': {
                    'periodo': mes1,
                    'dados': dados_mes1
                },
                'mes2': {
                    'periodo': mes2,
                    'dados': dados_mes2
                },
                'variacoes': {
                    'entradas': round(variacao_entradas, 2),
                    'saidas': round(variacao_saidas, 2),
                    'saldo_liquido': round(variacao_saldo, 2)
                }
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@relatorios_bp.route('/relatorios/categorias', methods=['GET'])
@jwt_required()
def relatorio_categorias():
    """Relatório de despesas por categoria"""
    try:
        # Parâmetros
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        
        if not data_inicio or not data_fim:
            return jsonify({'success': False, 'message': 'Período é obrigatório'}), 400
        
        data_inicio = datetime.strptime(data_inicio, '%Y-%m-%d').date()
        data_fim = datetime.strptime(data_fim, '%Y-%m-%d').date()
        
        # Despesas por categoria
        from src.models.core import Categoria, Subcategoria
        
        despesas_categoria = db.session.query(
            Categoria.id,
            Categoria.nome.label('categoria_nome'),
            Subcategoria.id.label('subcategoria_id'),
            Subcategoria.nome.label('subcategoria_nome'),
            func.count(Transacao.id).label('total_transacoes'),
            func.sum(Transacao.valor).label('total_valor')
        ).join(
            Transacao, Transacao.categoria_id == Categoria.id
        ).outerjoin(
            Subcategoria, Transacao.subcategoria_id == Subcategoria.id
        ).filter(
            Transacao.tipo == 'saida',
            Transacao.status == 'confirmado',
            Transacao.data_transacao >= data_inicio,
            Transacao.data_transacao <= data_fim
        ).group_by(
            Categoria.id, Categoria.nome, Subcategoria.id, Subcategoria.nome
        ).order_by(
            func.sum(Transacao.valor).desc()
        ).all()
        
        # Organizar dados por categoria
        categorias = {}
        total_geral = 0
        
        for despesa in despesas_categoria:
            categoria_id = despesa.id
            categoria_nome = despesa.categoria_nome
            
            if categoria_id not in categorias:
                categorias[categoria_id] = {
                    'id': categoria_id,
                    'nome': categoria_nome,
                    'total_valor': 0,
                    'total_transacoes': 0,
                    'subcategorias': {}
                }
            
            valor = float(despesa.total_valor)
            categorias[categoria_id]['total_valor'] += valor
            categorias[categoria_id]['total_transacoes'] += despesa.total_transacoes
            total_geral += valor
            
            # Subcategoria
            if despesa.subcategoria_id:
                subcategoria_id = despesa.subcategoria_id
                categorias[categoria_id]['subcategorias'][subcategoria_id] = {
                    'id': subcategoria_id,
                    'nome': despesa.subcategoria_nome,
                    'total_valor': valor,
                    'total_transacoes': despesa.total_transacoes
                }
        
        # Calcular percentuais
        for categoria in categorias.values():
            categoria['percentual'] = round((categoria['total_valor'] / total_geral * 100), 2) if total_geral > 0 else 0
            
            for subcategoria in categoria['subcategorias'].values():
                subcategoria['percentual'] = round((subcategoria['total_valor'] / categoria['total_valor'] * 100), 2) if categoria['total_valor'] > 0 else 0
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': {
                    'inicio': data_inicio.isoformat(),
                    'fim': data_fim.isoformat()
                },
                'resumo': {
                    'total_categorias': len(categorias),
                    'total_geral': total_geral
                },
                'categorias': list(categorias.values())
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@relatorios_bp.route('/relatorios/evolucao-mensal', methods=['GET'])
@jwt_required()
def relatorio_evolucao_mensal():
    """Relatório de evolução mensal dos últimos 12 meses"""
    try:
        # Últimos 12 meses
        hoje = date.today()
        meses = []
        
        for i in range(12):
            mes_data = hoje - relativedelta(months=i)
            inicio_mes = mes_data.replace(day=1)
            fim_mes = inicio_mes + relativedelta(months=1) - relativedelta(days=1)
            
            # Dados do mês
            entradas = Transacao.query.filter(
                Transacao.tipo == 'entrada',
                Transacao.status == 'confirmado',
                Transacao.data_transacao >= inicio_mes,
                Transacao.data_transacao <= fim_mes
            ).with_entities(func.sum(Transacao.valor)).scalar() or 0
            
            saidas = Transacao.query.filter(
                Transacao.tipo == 'saida',
                Transacao.status == 'confirmado',
                Transacao.data_transacao >= inicio_mes,
                Transacao.data_transacao <= fim_mes
            ).with_entities(func.sum(Transacao.valor)).scalar() or 0
            
            meses.append({
                'mes': mes_data.strftime('%Y-%m'),
                'mes_nome': mes_data.strftime('%B %Y'),
                'entradas': float(entradas),
                'saidas': float(saidas),
                'saldo_liquido': float(entradas) - float(saidas)
            })
        
        # Reverter para ordem cronológica
        meses.reverse()
        
        return jsonify({
            'success': True,
            'data': {
                'periodo': '12 meses',
                'meses': meses
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

