"""
Rotas para gerenciamento de transações (entradas e saídas)
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity
from src.models.core import db, Transacao, Caixa, Representante, Cliente, Categoria, Subcategoria
from datetime import datetime, date
from decimal import Decimal
from sqlalchemy import and_, or_, extract, func

transacoes_bp = Blueprint('transacoes', __name__)

@transacoes_bp.route('/transacoes', methods=['GET'])
@jwt_required()
def listar_transacoes():
    """Lista transações com filtros"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 50, type=int)
        tipo = request.args.get('tipo')
        caixa_id = request.args.get('caixa_id', type=int)
        representante_id = request.args.get('representante_id', type=int)
        categoria_id = request.args.get('categoria_id', type=int)
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        status = request.args.get('status', 'confirmado')
        
        query = Transacao.query.filter_by(status=status)
        
        # Filtros
        if tipo:
            query = query.filter_by(tipo=tipo)
        if caixa_id:
            query = query.filter(or_(
                Transacao.caixa_origem_id == caixa_id,
                Transacao.caixa_destino_id == caixa_id
            ))
        if representante_id:
            query = query.filter_by(representante_id=representante_id)
        if categoria_id:
            query = query.filter_by(categoria_id=categoria_id)
        if data_inicio:
            query = query.filter(Transacao.data_transacao >= datetime.strptime(data_inicio, '%Y-%m-%d').date())
        if data_fim:
            query = query.filter(Transacao.data_transacao <= datetime.strptime(data_fim, '%Y-%m-%d').date())
        
        transacoes = query.order_by(Transacao.data_transacao.desc(), Transacao.id.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'success': True,
            'data': [transacao.to_dict() for transacao in transacoes.items],
            'pagination': {
                'page': page,
                'pages': transacoes.pages,
                'per_page': per_page,
                'total': transacoes.total
            }
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transacoes/<int:transacao_id>', methods=['GET'])
@jwt_required()
def obter_transacao(transacao_id):
    """Obtém detalhes de uma transação específica"""
    try:
        transacao = Transacao.query.get_or_404(transacao_id)
        return jsonify({
            'success': True,
            'data': transacao.to_dict()
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/entradas', methods=['POST'])
@jwt_required()
def criar_entrada():
    """Cria uma nova entrada de dinheiro"""
    try:
        data = request.get_json()
        
        # Validações obrigatórias
        if not data.get('valor') or float(data['valor']) <= 0:
            return jsonify({'success': False, 'message': 'Valor deve ser maior que zero'}), 400
        
        if not data.get('descricao'):
            return jsonify({'success': False, 'message': 'Descrição é obrigatória'}), 400
        
        if not data.get('caixa_destino_id'):
            return jsonify({'success': False, 'message': 'Caixa de destino é obrigatório'}), 400
        
        # Verificar se o caixa existe
        caixa_destino = Caixa.query.get(data['caixa_destino_id'])
        if not caixa_destino or not caixa_destino.ativo:
            return jsonify({'success': False, 'message': 'Caixa de destino inválido'}), 400
        
        # Validar representante se fornecido
        if data.get('representante_id'):
            representante = Representante.query.get(data['representante_id'])
            if not representante or not representante.ativo:
                return jsonify({'success': False, 'message': 'Representante inválido'}), 400
        
        # Validar cliente se fornecido
        if data.get('cliente_id'):
            cliente = Cliente.query.get(data['cliente_id'])
            if not cliente:
                return jsonify({'success': False, 'message': 'Cliente inválido'}), 400
        
        # Criar transação
        transacao = Transacao(
            tipo='entrada',
            subtipo=data.get('subtipo', 'pagamento'),
            data_transacao=datetime.strptime(data.get('data_transacao', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=Decimal(str(data['valor'])),
            descricao=data['descricao'],
            caixa_destino_id=data['caixa_destino_id'],
            representante_id=data.get('representante_id'),
            cliente_id=data.get('cliente_id'),
            metodo_pagamento=data.get('metodo_pagamento', 'pix'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Atualizar saldo do caixa
        caixa_destino.saldo_atual += Decimal(str(data['valor']))
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Entrada registrada com sucesso',
            'data': transacao.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/saidas', methods=['POST'])
@jwt_required()
def criar_saida():
    """Cria uma nova saída de dinheiro"""
    try:
        data = request.get_json()
        
        # Validações obrigatórias
        if not data.get('valor') or float(data['valor']) <= 0:
            return jsonify({'success': False, 'message': 'Valor deve ser maior que zero'}), 400
        
        if not data.get('descricao'):
            return jsonify({'success': False, 'message': 'Descrição é obrigatória'}), 400
        
        if not data.get('caixa_origem_id'):
            return jsonify({'success': False, 'message': 'Caixa de origem é obrigatório'}), 400
        
        if not data.get('categoria_id'):
            return jsonify({'success': False, 'message': 'Categoria é obrigatória'}), 400
        
        # Verificar se o caixa existe e tem saldo suficiente
        caixa_origem = Caixa.query.get(data['caixa_origem_id'])
        if not caixa_origem or not caixa_origem.ativo:
            return jsonify({'success': False, 'message': 'Caixa de origem inválido'}), 400
        
        valor_saida = Decimal(str(data['valor']))
        saldo_disponivel = caixa_origem.saldo_atual + (caixa_origem.limite_cheque_especial or 0)
        
        if valor_saida > saldo_disponivel:
            return jsonify({
                'success': False, 
                'message': f'Saldo insuficiente. Disponível: R$ {float(saldo_disponivel):.2f}'
            }), 400
        
        # Verificar categoria
        categoria = Categoria.query.get(data['categoria_id'])
        if not categoria or not categoria.ativo:
            return jsonify({'success': False, 'message': 'Categoria inválida'}), 400
        
        # Verificar subcategoria se fornecida
        if data.get('subcategoria_id'):
            subcategoria = Subcategoria.query.get(data['subcategoria_id'])
            if not subcategoria or not subcategoria.ativo or subcategoria.categoria_id != data['categoria_id']:
                return jsonify({'success': False, 'message': 'Subcategoria inválida'}), 400
        
        # Criar transação
        transacao = Transacao(
            tipo='saida',
            subtipo=data.get('subtipo', 'despesa'),
            data_transacao=datetime.strptime(data.get('data_transacao', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=valor_saida,
            descricao=data['descricao'],
            caixa_origem_id=data['caixa_origem_id'],
            categoria_id=data['categoria_id'],
            subcategoria_id=data.get('subcategoria_id'),
            metodo_pagamento=data.get('metodo_pagamento', 'pix'),
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Atualizar saldo do caixa
        caixa_origem.saldo_atual -= valor_saida
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Saída registrada com sucesso',
            'data': transacao.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transferencias', methods=['POST'])
@jwt_required()
def criar_transferencia():
    """Cria uma transferência entre caixas"""
    try:
        data = request.get_json()
        
        # Validações obrigatórias
        if not data.get('valor') or float(data['valor']) <= 0:
            return jsonify({'success': False, 'message': 'Valor deve ser maior que zero'}), 400
        
        if not data.get('caixa_origem_id'):
            return jsonify({'success': False, 'message': 'Caixa de origem é obrigatório'}), 400
        
        if not data.get('caixa_destino_id'):
            return jsonify({'success': False, 'message': 'Caixa de destino é obrigatório'}), 400
        
        if data['caixa_origem_id'] == data['caixa_destino_id']:
            return jsonify({'success': False, 'message': 'Caixa de origem e destino devem ser diferentes'}), 400
        
        # Verificar caixas
        caixa_origem = Caixa.query.get(data['caixa_origem_id'])
        caixa_destino = Caixa.query.get(data['caixa_destino_id'])
        
        if not caixa_origem or not caixa_origem.ativo:
            return jsonify({'success': False, 'message': 'Caixa de origem inválido'}), 400
        
        if not caixa_destino or not caixa_destino.ativo:
            return jsonify({'success': False, 'message': 'Caixa de destino inválido'}), 400
        
        # Verificar saldo
        valor_transferencia = Decimal(str(data['valor']))
        saldo_disponivel = caixa_origem.saldo_atual + (caixa_origem.limite_cheque_especial or 0)
        
        if valor_transferencia > saldo_disponivel:
            return jsonify({
                'success': False, 
                'message': f'Saldo insuficiente. Disponível: R$ {float(saldo_disponivel):.2f}'
            }), 400
        
        # Criar transação
        transacao = Transacao(
            tipo='transferencia',
            subtipo='transferencia_interna',
            data_transacao=datetime.strptime(data.get('data_transacao', date.today().isoformat()), '%Y-%m-%d').date(),
            valor=valor_transferencia,
            descricao=data.get('descricao', f'Transferência de {caixa_origem.nome} para {caixa_destino.nome}'),
            caixa_origem_id=data['caixa_origem_id'],
            caixa_destino_id=data['caixa_destino_id'],
            observacoes=data.get('observacoes'),
            criado_por=get_jwt_identity()
        )
        
        db.session.add(transacao)
        
        # Atualizar saldos
        caixa_origem.saldo_atual -= valor_transferencia
        caixa_destino.saldo_atual += valor_transferencia
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Transferência realizada com sucesso',
            'data': transacao.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transacoes/<int:transacao_id>', methods=['PUT'])
@jwt_required()
def atualizar_transacao(transacao_id):
    """Atualiza uma transação existente"""
    try:
        transacao = Transacao.query.get_or_404(transacao_id)
        data = request.get_json()
        
        # Não permitir alteração de transações já conciliadas
        if transacao.conciliado:
            return jsonify({'success': False, 'message': 'Não é possível alterar transação já conciliada'}), 400
        
        valor_anterior = transacao.valor
        
        # Atualizar campos permitidos
        if 'descricao' in data:
            transacao.descricao = data['descricao']
        if 'observacoes' in data:
            transacao.observacoes = data['observacoes']
        if 'metodo_pagamento' in data:
            transacao.metodo_pagamento = data['metodo_pagamento']
        
        # Se alterar valor, recalcular saldos
        if 'valor' in data and float(data['valor']) > 0:
            novo_valor = Decimal(str(data['valor']))
            diferenca = novo_valor - valor_anterior
            
            # Atualizar saldos dos caixas
            if transacao.tipo == 'entrada' and transacao.caixa_destino:
                transacao.caixa_destino.saldo_atual += diferenca
            elif transacao.tipo == 'saida' and transacao.caixa_origem:
                # Verificar se há saldo suficiente para a diferença
                if diferenca > 0:
                    saldo_disponivel = transacao.caixa_origem.saldo_atual + (transacao.caixa_origem.limite_cheque_especial or 0)
                    if diferenca > saldo_disponivel:
                        return jsonify({
                            'success': False, 
                            'message': f'Saldo insuficiente para o aumento. Disponível: R$ {float(saldo_disponivel):.2f}'
                        }), 400
                
                transacao.caixa_origem.saldo_atual -= diferenca
            elif transacao.tipo == 'transferencia':
                if transacao.caixa_origem and transacao.caixa_destino:
                    # Verificar saldo para aumento
                    if diferenca > 0:
                        saldo_disponivel = transacao.caixa_origem.saldo_atual + (transacao.caixa_origem.limite_cheque_especial or 0)
                        if diferenca > saldo_disponivel:
                            return jsonify({
                                'success': False, 
                                'message': f'Saldo insuficiente para o aumento. Disponível: R$ {float(saldo_disponivel):.2f}'
                            }), 400
                    
                    transacao.caixa_origem.saldo_atual -= diferenca
                    transacao.caixa_destino.saldo_atual += diferenca
            
            transacao.valor = novo_valor
        
        transacao.atualizado_em = datetime.utcnow()
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Transação atualizada com sucesso',
            'data': transacao.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transacoes/<int:transacao_id>/cancelar', methods=['POST'])
@jwt_required()
def cancelar_transacao(transacao_id):
    """Cancela uma transação"""
    try:
        transacao = Transacao.query.get_or_404(transacao_id)
        
        if transacao.status == 'cancelado':
            return jsonify({'success': False, 'message': 'Transação já está cancelada'}), 400
        
        if transacao.conciliado:
            return jsonify({'success': False, 'message': 'Não é possível cancelar transação já conciliada'}), 400
        
        # Reverter saldos dos caixas
        if transacao.tipo == 'entrada' and transacao.caixa_destino:
            transacao.caixa_destino.saldo_atual -= transacao.valor
        elif transacao.tipo == 'saida' and transacao.caixa_origem:
            transacao.caixa_origem.saldo_atual += transacao.valor
        elif transacao.tipo == 'transferencia':
            if transacao.caixa_origem and transacao.caixa_destino:
                transacao.caixa_origem.saldo_atual += transacao.valor
                transacao.caixa_destino.saldo_atual -= transacao.valor
        
        transacao.status = 'cancelado'
        transacao.atualizado_em = datetime.utcnow()
        
        db.session.commit()
        
        return jsonify({
            'success': True,
            'message': 'Transação cancelada com sucesso',
            'data': transacao.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transacoes/resumo', methods=['GET'])
@jwt_required()
def resumo_transacoes():
    """Obtém resumo das transações por período"""
    try:
        data_inicio = request.args.get('data_inicio')
        data_fim = request.args.get('data_fim')
        
        # Definir período padrão (mês atual)
        if not data_inicio:
            hoje = date.today()
            data_inicio = date(hoje.year, hoje.month, 1).isoformat()
        if not data_fim:
            data_fim = date.today().isoformat()
        
        # Query base
        query = Transacao.query.filter(
            Transacao.data_transacao >= datetime.strptime(data_inicio, '%Y-%m-%d').date(),
            Transacao.data_transacao <= datetime.strptime(data_fim, '%Y-%m-%d').date(),
            Transacao.status == 'confirmado'
        )
        
        # Calcular totais
        total_entradas = query.filter_by(tipo='entrada').with_entities(func.sum(Transacao.valor)).scalar() or 0
        total_saidas = query.filter_by(tipo='saida').with_entities(func.sum(Transacao.valor)).scalar() or 0
        total_transferencias = query.filter_by(tipo='transferencia').with_entities(func.sum(Transacao.valor)).scalar() or 0
        
        # Contar transações
        count_entradas = query.filter_by(tipo='entrada').count()
        count_saidas = query.filter_by(tipo='saida').count()
        count_transferencias = query.filter_by(tipo='transferencia').count()
        
        resumo = {
            'periodo': {
                'data_inicio': data_inicio,
                'data_fim': data_fim
            },
            'entradas': {
                'total': float(total_entradas),
                'quantidade': count_entradas,
                'ticket_medio': float(total_entradas / count_entradas) if count_entradas > 0 else 0
            },
            'saidas': {
                'total': float(total_saidas),
                'quantidade': count_saidas,
                'ticket_medio': float(total_saidas / count_saidas) if count_saidas > 0 else 0
            },
            'transferencias': {
                'total': float(total_transferencias),
                'quantidade': count_transferencias
            },
            'saldo_liquido': float(total_entradas - total_saidas),
            'total_movimentacao': float(total_entradas + total_saidas + total_transferencias)
        }
        
        return jsonify({
            'success': True,
            'data': resumo
        })
    except Exception as e:
        return jsonify({'success': False, 'message': str(e)}), 500

@transacoes_bp.route('/transacoes/metodos-pagamento', methods=['GET'])
@jwt_required()
def metodos_pagamento():
    """Lista os métodos de pagamento disponíveis"""
    metodos = [
        {'value': 'dinheiro', 'label': 'Dinheiro'},
        {'value': 'pix', 'label': 'PIX'},
        {'value': 'transferencia', 'label': 'Transferência'},
        {'value': 'cartao_debito', 'label': 'Cartão de Débito'},
        {'value': 'cartao_credito', 'label': 'Cartão de Crédito'},
        {'value': 'cheque', 'label': 'Cheque'},
        {'value': 'outros', 'label': 'Outros'}
    ]
    
    return jsonify({
        'success': True,
        'data': metodos
    })

