"""
Rotas para gerenciamento de usuários e permissões
"""
from flask import Blueprint, request, jsonify
from flask_jwt_extended import jwt_required, get_jwt_identity, create_access_token
from datetime import datetime, timedelta
from src.models.core import db
from src.models.usuarios import (
    Usuario, PermissaoUsuario, RestricaoUsuario, SessaoUsuario, 
    LogAtividade, ConfiguracaoSistema, criar_admin_padrao, criar_configuracoes_padrao
)
import secrets
import hashlib

usuarios_bp = Blueprint('usuarios', __name__)

# ==================== AUTENTICAÇÃO ====================

@usuarios_bp.route('/login', methods=['POST'])
def login():
    """Login do usuário"""
    try:
        data = request.get_json()
        email = data.get('email', '').strip().lower()
        senha = data.get('senha', '')
        
        if not email or not senha:
            return jsonify({"error": "Email e senha são obrigatórios"}), 400
        
        # Buscar usuário
        usuario = Usuario.query.filter_by(email=email).first()
        
        if not usuario or not usuario.verificar_senha(senha):
            # Log de tentativa de login inválida
            log = LogAtividade(
                usuario_id=usuario.id if usuario else None,
                acao='login_falhou',
                modulo='auth',
                detalhes={'email': email, 'motivo': 'credenciais_invalidas'},
                ip_address=request.remote_addr,
                user_agent=request.headers.get('User-Agent')
            )
            db.session.add(log)
            db.session.commit()
            
            return jsonify({"error": "Email ou senha inválidos"}), 401
        
        if not usuario.ativo:
            return jsonify({"error": "Usuário inativo"}), 401
        
        # Criar token de acesso
        token = create_access_token(
            identity=usuario.id,
            expires_delta=timedelta(hours=8)
        )
        
        # Criar sessão
        sessao_token = secrets.token_urlsafe(32)
        sessao = SessaoUsuario(
            usuario_id=usuario.id,
            token=sessao_token,
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent'),
            data_expiracao=datetime.utcnow() + timedelta(hours=8)
        )
        
        # Atualizar último login
        usuario.ultimo_login = datetime.utcnow()
        
        # Log de login bem-sucedido
        log = LogAtividade(
            usuario_id=usuario.id,
            acao='login',
            modulo='auth',
            detalhes={'ip': request.remote_addr},
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        
        db.session.add(sessao)
        db.session.add(log)
        db.session.commit()
        
        return jsonify({
            "success": True,
            "token": token,
            "usuario": usuario.to_dict(),
            "permissoes": obter_permissoes_usuario(usuario.id)
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/logout', methods=['POST'])
@jwt_required()
def logout():
    """Logout do usuário"""
    try:
        usuario_id = get_jwt_identity()
        
        # Desativar todas as sessões do usuário
        SessaoUsuario.query.filter_by(usuario_id=usuario_id, ativa=True).update({"ativa": False})
        
        # Log de logout
        log = LogAtividade(
            usuario_id=usuario_id,
            acao='logout',
            modulo='auth',
            detalhes={'ip': request.remote_addr},
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        
        db.session.add(log)
        db.session.commit()
        
        return jsonify({"success": True, "message": "Logout realizado com sucesso"})
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/perfil', methods=['GET'])
@jwt_required()
def obter_perfil():
    """Obtém perfil do usuário logado"""
    try:
        usuario_id = get_jwt_identity()
        usuario = Usuario.query.get(usuario_id)
        
        if not usuario:
            return jsonify({"error": "Usuário não encontrado"}), 404
        
        return jsonify({
            "success": True,
            "usuario": usuario.to_dict(),
            "permissoes": obter_permissoes_usuario(usuario_id)
        })
        
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ==================== GESTÃO DE USUÁRIOS ====================

@usuarios_bp.route('/usuarios', methods=['GET'])
@jwt_required()
def listar_usuarios():
    """Lista todos os usuários"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'visualizar'):
            return jsonify({"error": "Sem permissão para visualizar usuários"}), 403
        
        usuarios = Usuario.query.all()
        
        return jsonify({
            "success": True,
            "usuarios": [u.to_dict() for u in usuarios]
        })
        
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/usuarios', methods=['POST'])
@jwt_required()
def criar_usuario():
    """Cria novo usuário"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'criar'):
            return jsonify({"error": "Sem permissão para criar usuários"}), 403
        
        data = request.get_json()
        
        # Validações
        if not data.get('nome') or not data.get('email') or not data.get('senha'):
            return jsonify({"error": "Nome, email e senha são obrigatórios"}), 400
        
        # Verificar se email já existe
        if Usuario.query.filter_by(email=data['email'].strip().lower()).first():
            return jsonify({"error": "Email já está em uso"}), 400
        
        # Criar usuário
        novo_usuario = Usuario(
            nome=data['nome'].strip(),
            email=data['email'].strip().lower(),
            tipo_usuario=data.get('tipo_usuario', 'operador'),
            ativo=data.get('ativo', True),
            criado_por=usuario_id,
            observacoes=data.get('observacoes', '')
        )
        novo_usuario.set_senha(data['senha'])
        
        db.session.add(novo_usuario)
        db.session.flush()  # Para obter o ID
        
        # Criar permissões se especificadas
        permissoes = data.get('permissoes', [])
        for perm_data in permissoes:
            permissao = PermissaoUsuario(
                usuario_id=novo_usuario.id,
                modulo=perm_data['modulo'],
                visualizar=perm_data.get('visualizar', False),
                criar=perm_data.get('criar', False),
                editar=perm_data.get('editar', False),
                excluir=perm_data.get('excluir', False),
                exportar=perm_data.get('exportar', False),
                aprovar_desconto=perm_data.get('aprovar_desconto', False),
                ver_valores_altos=perm_data.get('ver_valores_altos', False),
                gerenciar_backup=perm_data.get('gerenciar_backup', False)
            )
            db.session.add(permissao)
        
        # Criar restrições se especificadas
        restricoes = data.get('restricoes', [])
        for rest_data in restricoes:
            restricao = RestricaoUsuario(
                usuario_id=novo_usuario.id,
                tipo_restricao=rest_data['tipo'],
                valor_restricao=rest_data['valor'],
                descricao=rest_data.get('descricao', '')
            )
            db.session.add(restricao)
        
        # Log da ação
        log = LogAtividade(
            usuario_id=usuario_id,
            acao='criar_usuario',
            modulo='usuarios',
            detalhes={'novo_usuario_id': novo_usuario.id, 'email': novo_usuario.email},
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        db.session.add(log)
        
        db.session.commit()
        
        return jsonify({
            "success": True,
            "message": "Usuário criado com sucesso",
            "usuario": novo_usuario.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/usuarios/<int:user_id>', methods=['PUT'])
@jwt_required()
def editar_usuario(user_id):
    """Edita usuário existente"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'editar'):
            return jsonify({"error": "Sem permissão para editar usuários"}), 403
        
        usuario = Usuario.query.get(user_id)
        if not usuario:
            return jsonify({"error": "Usuário não encontrado"}), 404
        
        data = request.get_json()
        
        # Atualizar dados básicos
        if 'nome' in data:
            usuario.nome = data['nome'].strip()
        if 'email' in data:
            email = data['email'].strip().lower()
            if email != usuario.email and Usuario.query.filter_by(email=email).first():
                return jsonify({"error": "Email já está em uso"}), 400
            usuario.email = email
        if 'tipo_usuario' in data:
            usuario.tipo_usuario = data['tipo_usuario']
        if 'ativo' in data:
            usuario.ativo = data['ativo']
        if 'observacoes' in data:
            usuario.observacoes = data['observacoes']
        if 'senha' in data and data['senha']:
            usuario.set_senha(data['senha'])
        
        # Atualizar permissões se especificadas
        if 'permissoes' in data:
            # Remover permissões existentes
            PermissaoUsuario.query.filter_by(usuario_id=user_id).delete()
            
            # Criar novas permissões
            for perm_data in data['permissoes']:
                permissao = PermissaoUsuario(
                    usuario_id=user_id,
                    modulo=perm_data['modulo'],
                    visualizar=perm_data.get('visualizar', False),
                    criar=perm_data.get('criar', False),
                    editar=perm_data.get('editar', False),
                    excluir=perm_data.get('excluir', False),
                    exportar=perm_data.get('exportar', False),
                    aprovar_desconto=perm_data.get('aprovar_desconto', False),
                    ver_valores_altos=perm_data.get('ver_valores_altos', False),
                    gerenciar_backup=perm_data.get('gerenciar_backup', False)
                )
                db.session.add(permissao)
        
        # Atualizar restrições se especificadas
        if 'restricoes' in data:
            # Remover restrições existentes
            RestricaoUsuario.query.filter_by(usuario_id=user_id).delete()
            
            # Criar novas restrições
            for rest_data in data['restricoes']:
                restricao = RestricaoUsuario(
                    usuario_id=user_id,
                    tipo_restricao=rest_data['tipo'],
                    valor_restricao=rest_data['valor'],
                    descricao=rest_data.get('descricao', '')
                )
                db.session.add(restricao)
        
        # Log da ação
        log = LogAtividade(
            usuario_id=usuario_id,
            acao='editar_usuario',
            modulo='usuarios',
            detalhes={'usuario_editado_id': user_id, 'campos_alterados': list(data.keys())},
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        db.session.add(log)
        
        db.session.commit()
        
        return jsonify({
            "success": True,
            "message": "Usuário atualizado com sucesso",
            "usuario": usuario.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/usuarios/<int:user_id>', methods=['DELETE'])
@jwt_required()
def excluir_usuario(user_id):
    """Exclui usuário"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'excluir'):
            return jsonify({"error": "Sem permissão para excluir usuários"}), 403
        
        # Não permitir excluir a si mesmo
        if user_id == usuario_id:
            return jsonify({"error": "Não é possível excluir seu próprio usuário"}), 400
        
        usuario = Usuario.query.get(user_id)
        if not usuario:
            return jsonify({"error": "Usuário não encontrado"}), 404
        
        # Não permitir excluir administrador se for o único
        if usuario.tipo_usuario == 'administrador':
            admin_count = Usuario.query.filter_by(tipo_usuario='administrador', ativo=True).count()
            if admin_count <= 1:
                return jsonify({"error": "Não é possível excluir o último administrador"}), 400
        
        # Log da ação antes de excluir
        log = LogAtividade(
            usuario_id=usuario_id,
            acao='excluir_usuario',
            modulo='usuarios',
            detalhes={'usuario_excluido_id': user_id, 'email': usuario.email},
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        db.session.add(log)
        
        # Excluir usuário (cascade vai remover permissões e restrições)
        db.session.delete(usuario)
        db.session.commit()
        
        return jsonify({
            "success": True,
            "message": "Usuário excluído com sucesso"
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ==================== PERMISSÕES ====================

@usuarios_bp.route('/usuarios/<int:user_id>/permissoes', methods=['GET'])
@jwt_required()
def obter_permissoes_usuario_route(user_id):
    """Obtém permissões de um usuário"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'visualizar'):
            return jsonify({"error": "Sem permissão para visualizar permissões"}), 403
        
        permissoes = obter_permissoes_usuario(user_id)
        
        return jsonify({
            "success": True,
            "permissoes": permissoes
        })
        
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ==================== LOGS DE ATIVIDADE ====================

@usuarios_bp.route('/logs', methods=['GET'])
@jwt_required()
def obter_logs():
    """Obtém logs de atividade"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'usuarios', 'visualizar'):
            return jsonify({"error": "Sem permissão para visualizar logs"}), 403
        
        # Parâmetros de filtro
        limite = request.args.get('limite', 100, type=int)
        usuario_filtro = request.args.get('usuario_id', type=int)
        modulo_filtro = request.args.get('modulo')
        acao_filtro = request.args.get('acao')
        
        query = LogAtividade.query
        
        if usuario_filtro:
            query = query.filter_by(usuario_id=usuario_filtro)
        if modulo_filtro:
            query = query.filter_by(modulo=modulo_filtro)
        if acao_filtro:
            query = query.filter_by(acao=acao_filtro)
        
        logs = query.order_by(LogAtividade.data_acao.desc()).limit(limite).all()
        
        # Incluir nome do usuário
        logs_com_usuario = []
        for log in logs:
            log_dict = log.to_dict()
            if log.usuario:
                log_dict['usuario_nome'] = log.usuario.nome
            logs_com_usuario.append(log_dict)
        
        return jsonify({
            "success": True,
            "logs": logs_com_usuario
        })
        
    except Exception as e:
        return jsonify({"error": str(e)}), 500

# ==================== CONFIGURAÇÕES DO SISTEMA ====================

@usuarios_bp.route('/configuracoes', methods=['GET'])
@jwt_required()
def obter_configuracoes():
    """Obtém configurações do sistema"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'configuracoes', 'visualizar'):
            return jsonify({"error": "Sem permissão para visualizar configurações"}), 403
        
        categoria = request.args.get('categoria')
        
        query = ConfiguracaoSistema.query
        if categoria:
            query = query.filter_by(categoria=categoria)
        
        configuracoes = query.all()
        
        return jsonify({
            "success": True,
            "configuracoes": [c.to_dict() for c in configuracoes]
        })
        
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@usuarios_bp.route('/configuracoes/<int:config_id>', methods=['PUT'])
@jwt_required()
def atualizar_configuracao(config_id):
    """Atualiza configuração do sistema"""
    try:
        usuario_id = get_jwt_identity()
        
        # Verificar permissão
        if not verificar_permissao(usuario_id, 'configuracoes', 'editar'):
            return jsonify({"error": "Sem permissão para editar configurações"}), 403
        
        config = ConfiguracaoSistema.query.get(config_id)
        if not config:
            return jsonify({"error": "Configuração não encontrada"}), 404
        
        if not config.editavel:
            return jsonify({"error": "Esta configuração não pode ser editada"}), 400
        
        data = request.get_json()
        valor_antigo = config.get_valor()
        
        config.set_valor(data['valor'])
        config.data_atualizacao = datetime.utcnow()
        
        # Log da ação
        log = LogAtividade(
            usuario_id=usuario_id,
            acao='atualizar_configuracao',
            modulo='configuracoes',
            detalhes={
                'chave': config.chave,
                'valor_antigo': valor_antigo,
                'valor_novo': config.get_valor()
            },
            ip_address=request.remote_addr,
            user_agent=request.headers.get('User-Agent')
        )
        db.session.add(log)
        
        db.session.commit()
        
        return jsonify({
            "success": True,
            "message": "Configuração atualizada com sucesso",
            "configuracao": config.to_dict()
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ==================== FUNÇÕES AUXILIARES ====================

def verificar_permissao(usuario_id, modulo, acao):
    """Verifica se o usuário tem permissão para uma ação"""
    usuario = Usuario.query.get(usuario_id)
    if not usuario:
        return False
    
    return usuario.tem_permissao(modulo, acao)

def obter_permissoes_usuario(usuario_id):
    """Obtém todas as permissões de um usuário"""
    usuario = Usuario.query.get(usuario_id)
    if not usuario:
        return {}
    
    if usuario.tipo_usuario == 'administrador':
        # Administrador tem todas as permissões
        modulos = ['caixas', 'transacoes', 'representantes', 'categorias', 'cartoes', 
                  'dividas', 'relatorios', 'cobranca', 'usuarios', 'configuracoes']
        
        permissoes = {}
        for modulo in modulos:
            permissoes[modulo] = {
                'visualizar': True,
                'criar': True,
                'editar': True,
                'excluir': True,
                'exportar': True,
                'aprovar_desconto': True,
                'ver_valores_altos': True,
                'gerenciar_backup': True
            }
        
        return permissoes
    
    # Usuário com permissões específicas
    permissoes_db = PermissaoUsuario.query.filter_by(usuario_id=usuario_id).all()
    permissoes = {}
    
    for perm in permissoes_db:
        permissoes[perm.modulo] = perm.to_dict()
    
    return permissoes

def registrar_atividade(usuario_id, acao, modulo, detalhes=None):
    """Registra atividade do usuário"""
    log = LogAtividade(
        usuario_id=usuario_id,
        acao=acao,
        modulo=modulo,
        detalhes=detalhes or {},
        ip_address=request.remote_addr if request else None,
        user_agent=request.headers.get('User-Agent') if request else None
    )
    db.session.add(log)

# ==================== INICIALIZAÇÃO ====================

@usuarios_bp.route('/inicializar', methods=['POST'])
def inicializar_sistema():
    """Inicializa o sistema com dados padrão"""
    try:
        # Criar administrador padrão
        admin = criar_admin_padrao()
        
        # Criar configurações padrão
        criar_configuracoes_padrao()
        
        return jsonify({
            "success": True,
            "message": "Sistema inicializado com sucesso",
            "admin_email": admin.email
        })
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

