"""
Serviço de integração com IA (Manus/Vi)
"""
import requests
import json
from datetime import datetime, date
from typing import Dict, List, Optional
from src.models.core import db, Usuario
from src.models.cobranca import (
    Inadimplente, AcordoCobranca, ConfiguracaoDesconto, 
    HistoricoCobranca, CampanhaCobranca
)

class ManusAIService:
    """Serviço para integração com Manus AI (CIO)"""
    
    def __init__(self):
        self.api_key = "manus_villa_joias_2025"
        self.base_url = "https://api.manus.ai/v1"
        self.user_whatsapp = None  # Configurado pelo usuário
        
    def configurar_whatsapp_usuario(self, numero_whatsapp: str):
        """Configura o WhatsApp do usuário para comunicação direta"""
        self.user_whatsapp = numero_whatsapp
        
    def analisar_fluxo_caixa(self, periodo_dias: int = 30) -> Dict:
        """Análise inteligente do fluxo de caixa"""
        try:
            # Simular análise da IA
            insights = {
                "tendencias": [
                    "Faturamento cresceu 12% nos últimos 30 dias",
                    "Despesas operacionais aumentaram 8% - revisar categoria energia",
                    "Representante João Silva está 40% acima da média"
                ],
                "alertas": [
                    "Caixa reserva pode render R$ 500 a mais se aplicado em CDB",
                    "3 clientes inadimplentes há mais de 60 dias precisam de ação urgente"
                ],
                "recomendacoes": [
                    "Transferir R$ 50.000 do caixa operacional para reserva",
                    "Renegociar contratos de energia elétrica",
                    "Implementar campanha de cobrança para inadimplentes antigos"
                ],
                "score_saude_financeira": 8.5,
                "projecao_30_dias": {
                    "entrada_estimada": 145000.00,
                    "saida_estimada": 98000.00,
                    "saldo_projetado": 47000.00
                }
            }
            
            return {
                "success": True,
                "data": insights,
                "timestamp": datetime.now().isoformat()
            }
            
        except Exception as e:
            return {"success": False, "error": str(e)}
    
    def gerar_dicas_estrategicas(self) -> List[str]:
        """Gera dicas estratégicas baseadas nos dados atuais"""
        dicas = [
            "💡 Representante João está 40% acima da média - replique a estratégia",
            "⚠️ Despesas operacionais subiram 15% - revisar categoria energia", 
            "💰 Caixa reserva pode render R$ 500 a mais se transferir para CDB",
            "🚀 3 clientes inadimplentes há 60 dias - hora de negociar",
            "📊 Taxa de conversão de cobrança está em 14,7% - meta é 15%",
            "🎯 Foque em acordos à vista - 68% de efetividade vs 32% parcelados"
        ]
        
        return dicas[:4]  # Retorna 4 dicas principais
    
    def enviar_notificacao_whatsapp(self, mensagem: str, tipo: str = "info") -> bool:
        """Envia notificação via WhatsApp para o usuário"""
        if not self.user_whatsapp:
            return False
            
        try:
            # Simular envio de WhatsApp
            print(f"📱 WhatsApp para {self.user_whatsapp}: {mensagem}")
            
            # Aqui seria a integração real com WhatsApp Business API
            # payload = {
            #     "messaging_product": "whatsapp",
            #     "to": self.user_whatsapp,
            #     "text": {"body": f"🏆 Manus - Villa Joias\n\n{mensagem}"}
            # }
            
            return True
            
        except Exception as e:
            print(f"Erro ao enviar WhatsApp: {e}")
            return False
    
    def processar_pergunta_usuario(self, pergunta: str) -> str:
        """Processa perguntas do usuário e retorna respostas contextualizadas"""
        pergunta_lower = pergunta.lower()
        
        # Análise de representantes
        if "representante" in pergunta_lower and ("pagou" in pergunta_lower or "vendeu" in pergunta_lower):
            return self._analisar_representante(pergunta)
        
        # Análise de saldos
        elif "saldo" in pergunta_lower or "caixa" in pergunta_lower:
            return self._analisar_saldos()
        
        # Análise de inadimplência
        elif "inadimpl" in pergunta_lower or "atraso" in pergunta_lower:
            return self._analisar_inadimplencia()
        
        # Análise geral
        elif "como" in pergunta_lower and ("está" in pergunta_lower or "estão" in pergunta_lower):
            return self._analise_geral()
        
        else:
            return "Posso ajudar com análises de representantes, saldos, inadimplência e performance geral. Que informação específica você precisa?"
    
    def _analisar_representante(self, pergunta: str) -> str:
        """Análise específica de representante"""
        # Simular dados de representante
        return """📊 **Análise de Representantes - Junho 2025**

🥇 **João Silva:** R$ 15.430,50 (40% acima da média)
🥈 **Maria Santos:** R$ 12.850,30 (25% acima da média)  
🥉 **Carlos Lima:** R$ 9.750,20 (média do time)

💡 **Insight:** João está usando estratégia de visitas presenciais que aumentou conversão em 35%. Recomendo replicar para outros representantes."""
    
    def _analisar_saldos(self) -> str:
        """Análise de saldos dos caixas"""
        return """💰 **Situação dos Caixas - Hoje**

🏦 **Caixa Principal:** R$ 85.430,50
💎 **Reserva CDB:** R$ 125.000,00 (rendendo 12% a.a.)
💳 **Conta Corrente:** R$ 15.750,30
💵 **Caixa Físico:** R$ 2.850,00

⚡ **Ação Recomendada:** Transferir R$ 20.000 do caixa principal para reserva CDB para otimizar rendimentos."""
    
    def _analisar_inadimplencia(self) -> str:
        """Análise de inadimplência"""
        return """🚨 **Status Inadimplência - Junho 2025**

📊 **Total:** R$ 485.750,80 (156 clientes)
⏰ **Críticos:** 23 clientes há mais de 60 dias
💰 **Recuperado este mês:** R$ 125.430,50
📈 **Taxa conversão:** 14,7% (meta: 15%)

🎯 **Prioridade:** 5 clientes com valores acima de R$ 10.000 precisam de ação imediata."""
    
    def _analise_geral(self) -> str:
        """Análise geral do negócio"""
        return """📈 **Villa Joias - Resumo Executivo**

✅ **Faturamento:** +12,5% vs mês anterior
⚠️ **Despesas:** +8% (atenção categoria energia)
💰 **Saldo Líquido:** R$ 47.280,55
🎯 **Score Saúde:** 8.5/10

🚀 **Próximas ações:** Otimizar custos operacionais e intensificar cobrança de inadimplentes antigos."""

class ViAIService:
    """Serviço para Vi AI (Assistente de Cobrança)"""
    
    def __init__(self):
        self.nome = "Vi"
        self.empresa = "Villa Joias"
        self.whatsapp_business_token = None
        self.configuracao_ativa = None
        
    def configurar_whatsapp_business(self, token: str, numero: str):
        """Configura WhatsApp Business API"""
        self.whatsapp_business_token = token
        self.numero_whatsapp = numero
        
    def obter_configuracao_desconto(self) -> Optional[ConfiguracaoDesconto]:
        """Obtém configuração ativa de descontos"""
        if not self.configuracao_ativa:
            self.configuracao_ativa = ConfiguracaoDesconto.query.filter_by(ativa=True).first()
        return self.configuracao_ativa
    
    def calcular_proposta_acordo(self, inadimplente: Inadimplente) -> Dict:
        """Calcula proposta de acordo baseada nas configurações"""
        config = self.obter_configuracao_desconto()
        if not config:
            return {"error": "Configuração de desconto não encontrada"}
        
        valor_atual = float(inadimplente.saldo_atual)
        desconto_percentual = config.obter_desconto_por_valor(valor_atual)
        
        # Estratégia: sempre priorizar pagamento à vista
        propostas = []
        
        # 1ª Proposta: À vista com desconto máximo
        desconto_vista = min(desconto_percentual + 10, config.desconto_maximo_automatico)
        valor_vista = valor_atual * (1 - desconto_vista / 100)
        
        propostas.append({
            "tipo": "a_vista",
            "desconto_percentual": desconto_vista,
            "valor_original": valor_atual,
            "valor_acordo": valor_vista,
            "economia": valor_atual - valor_vista,
            "prioridade": 1,
            "mensagem": f"🎯 OFERTA ESPECIAL! Quite hoje com {desconto_vista}% de desconto!"
        })
        
        # 2ª Proposta: À vista com desconto padrão (se primeira for rejeitada)
        if desconto_vista > desconto_percentual:
            valor_vista_2 = valor_atual * (1 - desconto_percentual / 100)
            propostas.append({
                "tipo": "a_vista",
                "desconto_percentual": desconto_percentual,
                "valor_original": valor_atual,
                "valor_acordo": valor_vista_2,
                "economia": valor_atual - valor_vista_2,
                "prioridade": 2,
                "mensagem": f"💰 Última chance! {desconto_percentual}% de desconto para pagamento à vista."
            })
        
        # 3ª Proposta: Parcelado (apenas se à vista for rejeitada múltiplas vezes)
        desconto_parcelado = max(desconto_percentual - 5, 0)
        valor_parcelado = valor_atual * (1 - desconto_parcelado / 100)
        
        propostas.append({
            "tipo": "parcelado",
            "desconto_percentual": desconto_parcelado,
            "valor_original": valor_atual,
            "valor_acordo": valor_parcelado,
            "economia": valor_atual - valor_parcelado,
            "parcelas_sugeridas": min(3, int(valor_parcelado / 500)),  # Máximo 3x
            "prioridade": 3,
            "mensagem": f"📅 Opção parcelada: {desconto_parcelado}% desconto em até 3x"
        })
        
        return {
            "inadimplente_id": inadimplente.id,
            "propostas": propostas,
            "estrategia": "priorizar_vista",
            "tentativas_vista_obrigatorias": config.tentativas_vista_antes_parcelamento
        }
    
    def gerar_mensagem_cobranca(self, inadimplente: Inadimplente, tipo_mensagem: str = "inicial") -> str:
        """Gera mensagem personalizada de cobrança"""
        nome = inadimplente.nome_devedor.split()[0]  # Primeiro nome
        valor = inadimplente.saldo_atual
        dias_atraso = inadimplente.dias_em_atraso()
        
        if tipo_mensagem == "inicial":
            return f"""Oi {nome}! 👋

Aqui é a Vi, da {self.empresa}. Tudo bem?

Estou entrando em contato sobre seu acordo conosco no valor de R$ {valor:,.2f}.

Temos uma proposta especial para você quitar hoje mesmo! 🎯

Posso te explicar melhor?"""
        
        elif tipo_mensagem == "proposta_vista":
            proposta = self.calcular_proposta_acordo(inadimplente)
            melhor_proposta = proposta["propostas"][0]
            
            return f"""💰 {nome}, tenho uma SUPER OFERTA para você!

💎 Valor original: R$ {valor:,.2f}
🎯 Pagamento à vista HOJE: R$ {melhor_proposta['valor_acordo']:,.2f}
💸 Sua economia: R$ {melhor_proposta['economia']:,.2f}

⚡ Esta oferta é válida APENAS HOJE!

Chave PIX: {self.obter_configuracao_desconto().chave_pix or '[CONFIGURAR_PIX]'}

Vamos fechar? 🤝"""
        
        elif tipo_mensagem == "urgencia":
            return f"""🚨 {nome}, ÚLTIMA OPORTUNIDADE!

Seu acordo de R$ {valor:,.2f} está há {dias_atraso} dias em atraso.

💥 DESCONTO ESPECIAL válido até às 18h de hoje!

Não perca esta chance de regularizar sua situação com economia! 

Me chama que explico tudo! 📞"""
        
        elif tipo_mensagem == "parcelamento":
            return f"""📅 {nome}, que tal uma opção parcelada?

Como o pagamento à vista não rolou, preparei uma alternativa:

💳 Parcelamento em até 3x
💰 Ainda com desconto especial
📱 Parcelas via PIX

Quer saber os valores? 🤔"""
        
        return f"Oi {nome}! Vi da {self.empresa}. Vamos conversar sobre seu acordo?"
    
    def processar_resposta_cliente(self, inadimplente_id: int, resposta: str) -> Dict:
        """Processa resposta do cliente e define próxima ação"""
        resposta_lower = resposta.lower()
        
        # Respostas positivas
        if any(palavra in resposta_lower for palavra in ["sim", "aceito", "quero", "vamos", "ok", "pode"]):
            return {
                "acao": "enviar_dados_pagamento",
                "tipo": "positiva",
                "proxima_mensagem": "gerar_dados_pix"
            }
        
        # Respostas negativas
        elif any(palavra in resposta_lower for palavra in ["não", "nao", "agora não", "depois", "sem condições"]):
            return {
                "acao": "reagendar_contato",
                "tipo": "negativa", 
                "proxima_mensagem": "proposta_alternativa"
            }
        
        # Pedido de mais informações
        elif any(palavra in resposta_lower for palavra in ["como", "quanto", "valor", "desconto", "parcela"]):
            return {
                "acao": "explicar_proposta",
                "tipo": "duvida",
                "proxima_mensagem": "detalhar_acordo"
            }
        
        # Resposta neutra
        else:
            return {
                "acao": "esclarecer_duvida",
                "tipo": "neutra",
                "proxima_mensagem": "reformular_proposta"
            }
    
    def registrar_tentativa_cobranca(self, inadimplente_id: int, mensagem: str, resposta: str = None) -> bool:
        """Registra tentativa de cobrança no histórico"""
        try:
            historico = HistoricoCobranca(
                inadimplente_id=inadimplente_id,
                tipo_contato='whatsapp',
                mensagem_enviada=mensagem,
                resposta_cliente=resposta,
                status_tentativa='enviado' if not resposta else 'respondido',
                resultado='sem_resposta' if not resposta else 'respondido',
                automatico=True
            )
            
            db.session.add(historico)
            db.session.commit()
            return True
            
        except Exception as e:
            print(f"Erro ao registrar tentativa: {e}")
            return False
    
    def executar_campanha_automatica(self, campanha_id: int) -> Dict:
        """Executa campanha de cobrança automatizada"""
        try:
            campanha = CampanhaCobranca.query.get(campanha_id)
            if not campanha or not campanha.ativa:
                return {"error": "Campanha não encontrada ou inativa"}
            
            # Filtrar inadimplentes elegíveis
            query = Inadimplente.query.filter_by(status='ativo')
            
            if campanha.valor_minimo:
                query = query.filter(Inadimplente.saldo_atual >= campanha.valor_minimo)
            
            if campanha.valor_maximo:
                query = query.filter(Inadimplente.saldo_atual <= campanha.valor_maximo)
            
            if campanha.apenas_nao_negativados:
                query = query.filter_by(spc_negativado=False)
            
            inadimplentes = query.limit(50).all()  # Limite de 50 por execução
            
            resultados = {
                "campanha_id": campanha_id,
                "total_elegíveis": len(inadimplentes),
                "mensagens_enviadas": 0,
                "erros": []
            }
            
            for inadimplente in inadimplentes:
                try:
                    mensagem = self.gerar_mensagem_cobranca(inadimplente, "inicial")
                    
                    # Simular envio (aqui seria a integração real com WhatsApp)
                    print(f"📱 Enviando para {inadimplente.telefone_01}: {mensagem[:50]}...")
                    
                    self.registrar_tentativa_cobranca(inadimplente.id, mensagem)
                    resultados["mensagens_enviadas"] += 1
                    
                except Exception as e:
                    resultados["erros"].append(f"Erro com {inadimplente.nome_devedor}: {str(e)}")
            
            return resultados
            
        except Exception as e:
            return {"error": str(e)}

# Instâncias globais dos serviços
manus_service = ManusAIService()
vi_service = ViAIService()

