import pytest
import json
from datetime import datetime, timedelta
import sys
import os

# Adicionar o diretório src ao path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', 'src'))

class TestPerformance:
    """Testes de performance do sistema"""
    
    def test_performance_listagem_transacoes(self, client, auth_headers):
        """Teste de performance na listagem de transações"""
        import time
        
        # Criar dados de teste em massa
        caixa_response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Performance',
                'tipo': 'capital_giro',
                'saldo_inicial': 10000.00
            },
            headers=auth_headers
        )
        caixa_data = json.loads(caixa_response.data)
        caixa_id = caixa_data['id']
        
        rep_response = client.post('/api/representantes', 
            json={
                'nome': 'Rep Performance',
                'codigo': 'PERF001'
            },
            headers=auth_headers
        )
        rep_data = json.loads(rep_response.data)
        rep_id = rep_data['id']
        
        # Criar múltiplas transações
        start_time = time.time()
        for i in range(50):
            client.post('/api/transacoes', 
                json={
                    'tipo': 'entrada',
                    'valor': 100.00,
                    'caixa_id': caixa_id,
                    'representante_id': rep_id,
                    'descricao': f'Transação {i}',
                    'metodo_pagamento': 'pix'
                },
                headers=auth_headers
            )
        creation_time = time.time() - start_time
        
        # Testar listagem
        start_time = time.time()
        response = client.get('/api/transacoes', headers=auth_headers)
        listing_time = time.time() - start_time
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert len(data) >= 50
        
        # Performance deve ser aceitável (menos de 2 segundos para listagem)
        assert listing_time < 2.0, f"Listagem muito lenta: {listing_time:.2f}s"
        print(f"Performance - Criação: {creation_time:.2f}s, Listagem: {listing_time:.2f}s")
    
    def test_performance_relatorios(self, client, auth_headers):
        """Teste de performance dos relatórios"""
        import time
        
        start_time = time.time()
        response = client.get('/api/relatorios/dashboard', headers=auth_headers)
        dashboard_time = time.time() - start_time
        
        assert response.status_code == 200
        assert dashboard_time < 3.0, f"Dashboard muito lento: {dashboard_time:.2f}s"
        
        start_time = time.time()
        response = client.get('/api/relatorios/fluxo-caixa', headers=auth_headers)
        fluxo_time = time.time() - start_time
        
        assert response.status_code == 200
        assert fluxo_time < 2.0, f"Relatório de fluxo muito lento: {fluxo_time:.2f}s"
        
        print(f"Performance Relatórios - Dashboard: {dashboard_time:.2f}s, Fluxo: {fluxo_time:.2f}s")

class TestIntegracao:
    """Testes de integração entre módulos"""
    
    def test_fluxo_completo_venda(self, client, auth_headers):
        """Teste do fluxo completo de uma venda"""
        # 1. Criar representante
        rep_response = client.post('/api/representantes', 
            json={
                'nome': 'Vendedor Teste',
                'codigo': 'VT001',
                'telefone': '11999999999'
            },
            headers=auth_headers
        )
        assert rep_response.status_code == 201
        rep_data = json.loads(rep_response.data)
        rep_id = rep_data['id']
        
        # 2. Criar caixa
        caixa_response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Vendas',
                'tipo': 'capital_giro',
                'saldo_inicial': 0.00
            },
            headers=auth_headers
        )
        assert caixa_response.status_code == 201
        caixa_data = json.loads(caixa_response.data)
        caixa_id = caixa_data['id']
        
        # 3. Registrar entrada (pagamento do cliente)
        entrada_response = client.post('/api/transacoes', 
            json={
                'tipo': 'entrada',
                'valor': 1500.00,
                'caixa_id': caixa_id,
                'representante_id': rep_id,
                'descricao': 'Venda de joias',
                'metodo_pagamento': 'pix',
                'cliente_nome': 'Cliente Teste'
            },
            headers=auth_headers
        )
        assert entrada_response.status_code == 201
        
        # 4. Verificar saldo do caixa
        caixa_response = client.get(f'/api/caixas/{caixa_id}', headers=auth_headers)
        caixa_data = json.loads(caixa_response.data)
        assert caixa_data['saldo_atual'] == 1500.00
        
        # 5. Verificar relatório do representante
        rel_response = client.get(f'/api/relatorios/representantes/{rep_id}', headers=auth_headers)
        assert rel_response.status_code == 200
        rel_data = json.loads(rel_response.data)
        assert rel_data['total_vendas'] >= 1500.00
    
    def test_fluxo_cobranca_acordo(self, client, auth_headers):
        """Teste do fluxo completo de cobrança e acordo"""
        # 1. Criar representante
        rep_response = client.post('/api/representantes', 
            json={'nome': 'Rep Cobrança', 'codigo': 'RC001'},
            headers=auth_headers
        )
        rep_data = json.loads(rep_response.data)
        rep_id = rep_data['id']
        
        # 2. Criar inadimplente
        inad_response = client.post('/api/cobranca/inadimplentes', 
            json={
                'nome': 'Devedor Teste',
                'cpf': '12345678901',
                'telefone_01': '11999999999',
                'representante_id': rep_id,
                'saldo_original': 1000.00,
                'saldo_atual': 1200.00,
                'numero_contrato': 'CONT001'
            },
            headers=auth_headers
        )
        assert inad_response.status_code == 201
        inad_data = json.loads(inad_response.data)
        inad_id = inad_data['id']
        
        # 3. Criar acordo
        acordo_response = client.post('/api/cobranca/acordos', 
            json={
                'inadimplente_id': inad_id,
                'valor_original': 1200.00,
                'valor_acordo': 800.00,
                'desconto_percentual': 33.33,
                'tipo_pagamento': 'a_vista',
                'parcelas': 1
            },
            headers=auth_headers
        )
        assert acordo_response.status_code == 201
        acordo_data = json.loads(acordo_response.data)
        
        # 4. Criar caixa para recebimento
        caixa_response = client.post('/api/caixas', 
            json={
                'nome': 'Caixa Cobrança',
                'tipo': 'capital_giro',
                'saldo_inicial': 0.00
            },
            headers=auth_headers
        )
        caixa_data = json.loads(caixa_response.data)
        caixa_id = caixa_data['id']
        
        # 5. Registrar pagamento do acordo
        pagamento_response = client.post('/api/transacoes', 
            json={
                'tipo': 'entrada',
                'valor': 800.00,
                'caixa_id': caixa_id,
                'descricao': f'Pagamento acordo - {inad_data["nome"]}',
                'metodo_pagamento': 'pix',
                'acordo_id': acordo_data['id']
            },
            headers=auth_headers
        )
        assert pagamento_response.status_code == 201
        
        # 6. Verificar saldo do caixa
        caixa_response = client.get(f'/api/caixas/{caixa_id}', headers=auth_headers)
        caixa_data = json.loads(caixa_response.data)
        assert caixa_data['saldo_atual'] == 800.00

class TestSeguranca:
    """Testes de segurança"""
    
    def test_acesso_sem_token(self, client):
        """Teste de acesso sem token de autenticação"""
        response = client.get('/api/caixas')
        assert response.status_code == 401
        
        response = client.post('/api/transacoes', json={})
        assert response.status_code == 401
        
        response = client.get('/api/relatorios/dashboard')
        assert response.status_code == 401
    
    def test_token_invalido(self, client):
        """Teste com token inválido"""
        headers = {'Authorization': 'Bearer token_invalido'}
        
        response = client.get('/api/caixas', headers=headers)
        assert response.status_code == 401
        
        response = client.post('/api/transacoes', json={}, headers=headers)
        assert response.status_code == 401
    
    def test_validacao_dados_entrada(self, client, auth_headers):
        """Teste de validação de dados de entrada"""
        # Teste com dados inválidos para caixa
        response = client.post('/api/caixas', 
            json={
                'nome': '',  # Nome vazio
                'tipo': 'tipo_invalido',  # Tipo inválido
                'saldo_inicial': -100  # Saldo negativo
            },
            headers=auth_headers
        )
        assert response.status_code == 400
        
        # Teste com dados inválidos para transação
        response = client.post('/api/transacoes', 
            json={
                'tipo': 'tipo_invalido',  # Tipo inválido
                'valor': -50,  # Valor negativo
                'caixa_id': 99999  # ID inexistente
            },
            headers=auth_headers
        )
        assert response.status_code == 400
    
    def test_sql_injection_protection(self, client, auth_headers):
        """Teste de proteção contra SQL injection"""
        # Tentar SQL injection no nome da caixa
        response = client.post('/api/caixas', 
            json={
                'nome': "'; DROP TABLE caixas; --",
                'tipo': 'capital_giro',
                'saldo_inicial': 100.00
            },
            headers=auth_headers
        )
        
        # Deve criar normalmente (string é tratada como texto)
        assert response.status_code == 201
        
        # Verificar se tabela ainda existe
        response = client.get('/api/caixas', headers=auth_headers)
        assert response.status_code == 200

class TestBackup:
    """Testes do sistema de backup"""
    
    def test_criar_backup(self, client, auth_headers):
        """Teste de criação de backup"""
        response = client.post('/api/backup/create', 
            json={'tipo': 'manual'},
            headers=auth_headers
        )
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert 'backup_id' in data
        assert data['status'] == 'sucesso'
    
    def test_listar_backups(self, client, auth_headers):
        """Teste de listagem de backups"""
        # Criar um backup primeiro
        client.post('/api/backup/create', 
            json={'tipo': 'manual'},
            headers=auth_headers
        )
        
        response = client.get('/api/backup/list', headers=auth_headers)
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert len(data) >= 1
    
    def test_status_backup(self, client, auth_headers):
        """Teste de status do sistema de backup"""
        response = client.get('/api/backup/status', headers=auth_headers)
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert 'total_backups' in data
        assert 'espaco_usado' in data
        assert 'ultimo_backup' in data

class TestIA:
    """Testes da integração com IA"""
    
    def test_chat_manus(self, client, auth_headers):
        """Teste do chat com Manus"""
        response = client.post('/api/ai/chat', 
            json={
                'mensagem': 'Como está o faturamento este mês?',
                'tipo': 'manus'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert 'resposta' in data
        assert len(data['resposta']) > 0
    
    def test_analise_financeira(self, client, auth_headers):
        """Teste de análise financeira da IA"""
        response = client.post('/api/ai/analise', 
            json={
                'tipo': 'fluxo_caixa',
                'periodo': '30_dias'
            },
            headers=auth_headers
        )
        
        assert response.status_code == 200
        data = json.loads(response.data)
        assert 'insights' in data
        assert 'recomendacoes' in data

if __name__ == '__main__':
    pytest.main([__file__, '-v'])

