import React, { useState, useEffect } from 'react';
import { 
  Shield, 
  Database, 
  Download, 
  Upload, 
  Trash2, 
  RefreshCw, 
  Clock, 
  HardDrive,
  Lock,
  AlertTriangle,
  CheckCircle,
  Settings,
  FileText,
  Calendar
} from 'lucide-react';

const BackupModule = () => {
  const [activeTab, setActiveTab] = useState('dashboard');
  const [backups, setBackups] = useState({
    daily: [],
    weekly: [],
    monthly: [],
    manual: []
  });
  const [status, setStatus] = useState(null);
  const [logs, setLogs] = useState([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    loadBackupData();
  }, []);

  const loadBackupData = async () => {
    try {
      setLoading(true);
      
      // Carregar lista de backups
      const backupsResponse = await fetch('/api/backup/list', {
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      });
      const backupsData = await backupsResponse.json();
      
      if (backupsData.success) {
        setBackups(backupsData.backups);
      }
      
      // Carregar status
      const statusResponse = await fetch('/api/backup/status', {
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      });
      const statusData = await statusResponse.json();
      
      if (statusData.success) {
        setStatus(statusData.status);
      }
      
    } catch (error) {
      console.error('Erro ao carregar dados de backup:', error);
    } finally {
      setLoading(false);
    }
  };

  const createBackup = async (type = 'manual') => {
    try {
      setLoading(true);
      
      const response = await fetch('/api/backup/create', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          type: type,
          include_database: true,
          include_files: true
        })
      });
      
      const data = await response.json();
      
      if (data.success) {
        alert('Backup criado com sucesso!');
        loadBackupData();
      } else {
        alert(`Erro ao criar backup: ${data.message}`);
      }
      
    } catch (error) {
      alert('Erro ao criar backup');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const downloadBackup = async (backupName, backupPath) => {
    try {
      const response = await fetch(`/api/backup/download/${backupName}`, {
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      });
      
      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = backupName;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      } else {
        alert('Erro ao baixar backup');
      }
      
    } catch (error) {
      alert('Erro ao baixar backup');
      console.error(error);
    }
  };

  const deleteBackup = async (backupPath) => {
    if (!confirm('Tem certeza que deseja excluir este backup?')) {
      return;
    }
    
    try {
      const response = await fetch('/api/backup/delete', {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({ backup_file: backupPath })
      });
      
      const data = await response.json();
      
      if (data.success) {
        alert('Backup excluído com sucesso!');
        loadBackupData();
      } else {
        alert(`Erro ao excluir backup: ${data.message}`);
      }
      
    } catch (error) {
      alert('Erro ao excluir backup');
      console.error(error);
    }
  };

  const cleanupOldBackups = async () => {
    if (!confirm('Tem certeza que deseja remover backups antigos?')) {
      return;
    }
    
    try {
      setLoading(true);
      
      const response = await fetch('/api/backup/cleanup', {
        method: 'POST',
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      });
      
      const data = await response.json();
      
      if (data.success) {
        alert('Limpeza concluída com sucesso!');
        loadBackupData();
      } else {
        alert(`Erro na limpeza: ${data.message}`);
      }
      
    } catch (error) {
      alert('Erro na limpeza');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  const StatusCard = ({ title, value, subtitle, icon: Icon, color = "blue" }) => (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-start justify-between">
        <div className="flex-1">
          <p className="text-sm font-medium text-gray-600">{title}</p>
          <p className={`text-2xl font-bold text-${color}-600 mt-1`}>{value}</p>
          {subtitle && (
            <p className="text-sm text-gray-500 mt-1">{subtitle}</p>
          )}
        </div>
        <div className={`p-3 bg-${color}-50 rounded-lg`}>
          <Icon className={`h-6 w-6 text-${color}-600`} />
        </div>
      </div>
    </div>
  );

  const BackupTable = ({ backups, title, type }) => (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200">
      <div className="px-6 py-4 border-b border-gray-200">
        <h3 className="text-lg font-semibold text-gray-900">{title}</h3>
      </div>
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Nome do Arquivo
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Tamanho
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Data de Criação
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Ações
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {backups.length === 0 ? (
              <tr>
                <td colSpan="5" className="px-6 py-4 text-center text-gray-500">
                  Nenhum backup encontrado
                </td>
              </tr>
            ) : (
              backups.map((backup, index) => (
                <tr key={index} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    {backup.name}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {(backup.size / 1024 / 1024).toFixed(2)} MB
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {new Date(backup.created).toLocaleString('pt-BR')}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    {backup.integrity_ok ? (
                      <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                        <CheckCircle className="h-3 w-3 mr-1" />
                        Íntegro
                      </span>
                    ) : (
                      <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                        <AlertTriangle className="h-3 w-3 mr-1" />
                        Corrompido
                      </span>
                    )}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                    <button
                      onClick={() => downloadBackup(backup.name, backup.path)}
                      className="text-blue-600 hover:text-blue-900"
                      title="Download"
                    >
                      <Download className="h-4 w-4" />
                    </button>
                    <button
                      onClick={() => deleteBackup(backup.path)}
                      className="text-red-600 hover:text-red-900"
                      title="Excluir"
                    >
                      <Trash2 className="h-4 w-4" />
                    </button>
                  </td>
                </tr>
              ))
            )}
          </tbody>
        </table>
      </div>
    </div>
  );

  const Dashboard = () => (
    <div className="space-y-6">
      {/* Cards de Status */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <StatusCard
          title="Total de Backups"
          value={status?.total_backups || 0}
          icon={Database}
          color="blue"
        />
        <StatusCard
          title="Espaço Utilizado"
          value={status?.total_size_mb ? `${status.total_size_mb} MB` : '0 MB'}
          icon={HardDrive}
          color="green"
        />
        <StatusCard
          title="Último Backup"
          value={status?.last_backup ? new Date(status.last_backup).toLocaleDateString('pt-BR') : 'Nunca'}
          icon={Clock}
          color="yellow"
        />
        <StatusCard
          title="Criptografia"
          value={status?.encryption_enabled ? 'Ativa' : 'Inativa'}
          icon={Lock}
          color={status?.encryption_enabled ? 'green' : 'red'}
        />
      </div>

      {/* Uso do Disco */}
      {status?.disk_usage && (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Uso do Disco</h3>
          <div className="space-y-4">
            <div className="flex justify-between text-sm">
              <span>Espaço Total: {status.disk_usage.total_space_gb} GB</span>
              <span>Livre: {status.disk_usage.free_space_gb} GB</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div 
                className="bg-blue-600 h-2 rounded-full" 
                style={{ width: `${status.disk_usage.usage_percent}%` }}
              ></div>
            </div>
            <p className="text-sm text-gray-600">
              {status.disk_usage.usage_percent}% utilizado
            </p>
          </div>
        </div>
      )}

      {/* Ações Rápidas */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Ações Rápidas</h3>
        <div className="flex flex-wrap gap-4">
          <button
            onClick={() => createBackup('manual')}
            disabled={loading}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center disabled:opacity-50"
          >
            <Database className="h-4 w-4 mr-2" />
            {loading ? 'Criando...' : 'Backup Manual'}
          </button>
          <button
            onClick={cleanupOldBackups}
            disabled={loading}
            className="bg-yellow-600 text-white px-4 py-2 rounded-lg hover:bg-yellow-700 transition-colors flex items-center disabled:opacity-50"
          >
            <Trash2 className="h-4 w-4 mr-2" />
            Limpar Antigos
          </button>
          <button
            onClick={loadBackupData}
            disabled={loading}
            className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors flex items-center disabled:opacity-50"
          >
            <RefreshCw className="h-4 w-4 mr-2" />
            Atualizar
          </button>
        </div>
      </div>
    </div>
  );

  const BackupsList = () => (
    <div className="space-y-6">
      <BackupTable backups={backups.manual} title="Backups Manuais" type="manual" />
      <BackupTable backups={backups.daily} title="Backups Diários" type="daily" />
      <BackupTable backups={backups.weekly} title="Backups Semanais" type="weekly" />
      <BackupTable backups={backups.monthly} title="Backups Mensais" type="monthly" />
    </div>
  );

  const Header = () => (
    <div className="bg-white shadow-sm border-b border-gray-200 px-6 py-4">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">
            {activeTab === 'dashboard' && 'Dashboard de Backup'}
            {activeTab === 'backups' && 'Lista de Backups'}
            {activeTab === 'logs' && 'Logs do Sistema'}
            {activeTab === 'config' && 'Configurações'}
          </h1>
          <p className="text-sm text-gray-600 mt-1">
            Sistema de backup e segurança Villa Joias
          </p>
        </div>
        <div className="flex items-center space-x-4">
          <button 
            onClick={() => createBackup('manual')}
            disabled={loading}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center disabled:opacity-50"
          >
            <Database className="h-4 w-4 mr-2" />
            Novo Backup
          </button>
          <div className="w-8 h-8 bg-gray-300 rounded-full"></div>
        </div>
      </div>
    </div>
  );

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="ml-64">
        <Header />
        
        {/* Navegação */}
        <div className="bg-white border-b border-gray-200">
          <nav className="px-6">
            <div className="flex space-x-8">
              {[
                { id: 'dashboard', label: 'Dashboard', icon: Shield },
                { id: 'backups', label: 'Backups', icon: Database },
                { id: 'logs', label: 'Logs', icon: FileText },
                { id: 'config', label: 'Configurações', icon: Settings }
              ].map(({ id, label, icon: Icon }) => (
                <button
                  key={id}
                  onClick={() => setActiveTab(id)}
                  className={`flex items-center px-1 py-4 border-b-2 font-medium text-sm ${
                    activeTab === id
                      ? 'border-yellow-500 text-yellow-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  <Icon className="h-4 w-4 mr-2" />
                  {label}
                </button>
              ))}
            </div>
          </nav>
        </div>

        <main className="p-6">
          {activeTab === 'dashboard' && <Dashboard />}
          {activeTab === 'backups' && <BackupsList />}
          {activeTab === 'logs' && (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Logs do Sistema</h3>
              <p className="text-gray-600">Funcionalidade de logs em desenvolvimento...</p>
            </div>
          )}
          {activeTab === 'config' && (
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Configurações de Backup</h3>
              <p className="text-gray-600">Configurações em desenvolvimento...</p>
            </div>
          )}
        </main>
      </div>
    </div>
  );
};

export default BackupModule;

