import React, { useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import {
  BarChart3,
  CreditCard,
  Users,
  FileText,
  Phone,
  Settings,
  LogOut,
  Menu,
  X,
  Wallet,
  TrendingUp,
  Calculator,
  Shield,
  MessageSquare,
  Bell,
  ChevronDown,
  Home
} from 'lucide-react';

const Sidebar = ({ children }) => {
  const [isOpen, setIsOpen] = useState(true);
  const [activeSubmenu, setActiveSubmenu] = useState(null);
  const location = useLocation();
  const navigate = useNavigate();

  const menuItems = [
    {
      id: 'dashboard',
      label: 'Dashboard',
      icon: Home,
      path: '/dashboard',
      color: 'text-yellow-600'
    },
    {
      id: 'caixas',
      label: 'Caixas',
      icon: Wallet,
      path: '/caixas',
      color: 'text-blue-600',
      submenu: [
        { label: 'Gerenciar Caixas', path: '/caixas' },
        { label: 'Transferências', path: '/caixas/transferencias' },
        { label: 'Extratos', path: '/caixas/extratos' }
      ]
    },
    {
      id: 'transacoes',
      label: 'Transações',
      icon: TrendingUp,
      path: '/transacoes',
      color: 'text-green-600',
      submenu: [
        { label: 'Nova Entrada', path: '/transacoes/entrada' },
        { label: 'Nova Saída', path: '/transacoes/saida' },
        { label: 'Histórico', path: '/transacoes/historico' }
      ]
    },
    {
      id: 'representantes',
      label: 'Representantes',
      icon: Users,
      path: '/representantes',
      color: 'text-purple-600',
      submenu: [
        { label: 'Gerenciar', path: '/representantes' },
        { label: 'Performance', path: '/representantes/performance' },
        { label: 'Comissões', path: '/representantes/comissoes' }
      ]
    },
    {
      id: 'cartoes',
      label: 'Cartões',
      icon: CreditCard,
      path: '/cartoes',
      color: 'text-indigo-600',
      submenu: [
        { label: 'Meus Cartões', path: '/cartoes' },
        { label: 'Nova Despesa', path: '/cartoes/despesa' },
        { label: 'Faturas', path: '/cartoes/faturas' }
      ]
    },
    {
      id: 'cobranca',
      label: 'Cobrança',
      icon: Phone,
      path: '/cobranca',
      color: 'text-red-600',
      submenu: [
        { label: 'Inadimplentes', path: '/cobranca' },
        { label: 'Campanhas', path: '/cobranca/campanhas' },
        { label: 'Relatórios', path: '/cobranca/relatorios' }
      ]
    },
    {
      id: 'relatorios',
      label: 'Relatórios',
      icon: FileText,
      path: '/relatorios',
      color: 'text-orange-600',
      submenu: [
        { label: 'Fluxo de Caixa', path: '/relatorios/fluxo' },
        { label: 'Comparativos', path: '/relatorios/comparativos' },
        { label: 'Exportar', path: '/relatorios/exportar' }
      ]
    },
    {
      id: 'usuarios',
      label: 'Usuários',
      icon: Shield,
      path: '/usuarios',
      color: 'text-gray-600'
    },
    {
      id: 'backup',
      label: 'Backup',
      icon: Database,
      path: '/backup',
      color: 'text-blue-600',
      submenu: [
        { label: 'Dashboard', path: '/backup' },
        { label: 'Criar Backup', path: '/backup/create' },
        { label: 'Restaurar', path: '/backup/restore' }
      ]
    },
    {
      id: 'configuracoes',
      label: 'Configurações',
      icon: Settings,
      path: '/configuracoes',
      color: 'text-gray-600',
      submenu: [
        { label: 'Geral', path: '/configuracoes' },
        { label: 'Integrações', path: '/configuracoes/integracoes' }
      ]
    }
  ];

  const handleMenuClick = (item) => {
    if (item.submenu) {
      setActiveSubmenu(activeSubmenu === item.id ? null : item.id);
    } else {
      navigate(item.path);
      setActiveSubmenu(null);
    }
  };

  const handleSubmenuClick = (path) => {
    navigate(path);
    setActiveSubmenu(null);
  };

  const handleLogout = () => {
    localStorage.removeItem('villa_joias_token');
    window.location.reload();
  };

  const isActivePath = (path) => {
    return location.pathname === path || location.pathname.startsWith(path + '/');
  };

  return (
    <div className="flex h-screen bg-gray-100">
      {/* Sidebar */}
      <div className={`${isOpen ? 'w-64' : 'w-16'} bg-gray-900 text-white transition-all duration-300 flex flex-col`}>
        {/* Header */}
        <div className="p-4 border-b border-gray-700">
          <div className="flex items-center justify-between">
            {isOpen && (
              <div className="flex items-center">
                <div className="w-10 h-10 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-xl flex items-center justify-center mr-3">
                  <span className="text-black font-bold text-lg">VJ</span>
                </div>
                <div>
                  <h1 className="text-lg font-bold">Villa Joias</h1>
                  <p className="text-xs text-gray-400">Sistema Financeiro</p>
                </div>
              </div>
            )}
            <button
              onClick={() => setIsOpen(!isOpen)}
              className="p-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              {isOpen ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
            </button>
          </div>
        </div>

        {/* Menu Items */}
        <nav className="flex-1 p-4 space-y-2 overflow-y-auto">
          {menuItems.map((item) => {
            const Icon = item.icon;
            const isActive = isActivePath(item.path);
            const hasSubmenu = item.submenu && item.submenu.length > 0;
            const isSubmenuOpen = activeSubmenu === item.id;

            return (
              <div key={item.id}>
                <button
                  onClick={() => handleMenuClick(item)}
                  className={`w-full flex items-center justify-between p-3 rounded-lg transition-all duration-200 ${
                    isActive
                      ? 'bg-yellow-600 text-black font-medium'
                      : 'text-gray-300 hover:bg-gray-700 hover:text-white'
                  }`}
                >
                  <div className="flex items-center">
                    <Icon className={`h-5 w-5 ${isActive ? 'text-black' : item.color}`} />
                    {isOpen && (
                      <span className="ml-3 text-sm font-medium">{item.label}</span>
                    )}
                  </div>
                  {isOpen && hasSubmenu && (
                    <ChevronDown 
                      className={`h-4 w-4 transition-transform ${
                        isSubmenuOpen ? 'rotate-180' : ''
                      }`} 
                    />
                  )}
                </button>

                {/* Submenu */}
                {isOpen && hasSubmenu && isSubmenuOpen && (
                  <div className="ml-8 mt-2 space-y-1">
                    {item.submenu.map((subItem, index) => (
                      <button
                        key={index}
                        onClick={() => handleSubmenuClick(subItem.path)}
                        className={`w-full text-left p-2 rounded-md text-sm transition-colors ${
                          isActivePath(subItem.path)
                            ? 'bg-yellow-600 text-black'
                            : 'text-gray-400 hover:bg-gray-700 hover:text-white'
                        }`}
                      >
                        {subItem.label}
                      </button>
                    ))}
                  </div>
                )}
              </div>
            );
          })}
        </nav>

        {/* Vi Assistant Status */}
        <div className="p-4 border-t border-gray-700">
          <div className={`flex items-center ${isOpen ? 'justify-between' : 'justify-center'}`}>
            {isOpen && (
              <div className="flex items-center">
                <div className="w-8 h-8 bg-gradient-to-br from-purple-500 to-pink-500 rounded-full flex items-center justify-center mr-3">
                  <MessageSquare className="h-4 w-4 text-white" />
                </div>
                <div>
                  <p className="text-sm font-medium">Vi Assistant</p>
                  <div className="flex items-center">
                    <div className="w-2 h-2 bg-green-400 rounded-full mr-2"></div>
                    <span className="text-xs text-gray-400">Online</span>
                  </div>
                </div>
              </div>
            )}
            <div className="flex items-center space-x-2">
              <button className="p-2 rounded-lg hover:bg-gray-700 transition-colors">
                <Bell className="h-4 w-4 text-gray-400" />
              </button>
              {!isOpen && (
                <div className="w-2 h-2 bg-green-400 rounded-full"></div>
              )}
            </div>
          </div>
        </div>

        {/* User Menu */}
        <div className="p-4 border-t border-gray-700">
          <div className={`flex items-center ${isOpen ? 'justify-between' : 'justify-center'}`}>
            {isOpen && (
              <div className="flex items-center">
                <div className="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                  <span className="text-white text-sm font-medium">A</span>
                </div>
                <div>
                  <p className="text-sm font-medium">Admin</p>
                  <p className="text-xs text-gray-400">Administrador</p>
                </div>
              </div>
            )}
            <button
              onClick={handleLogout}
              className="p-2 rounded-lg hover:bg-gray-700 transition-colors"
              title="Sair"
            >
              <LogOut className="h-4 w-4 text-gray-400" />
            </button>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Top Bar */}
        <header className="bg-white shadow-sm border-b border-gray-200 px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <h2 className="text-xl font-semibold text-gray-900">
                {menuItems.find(item => isActivePath(item.path))?.label || 'Dashboard'}
              </h2>
            </div>
            
            <div className="flex items-center space-x-4">
              {/* Notificações */}
              <button className="relative p-2 text-gray-400 hover:text-gray-600 transition-colors">
                <Bell className="h-5 w-5" />
                <span className="absolute top-1 right-1 w-2 h-2 bg-red-500 rounded-full"></span>
              </button>

              {/* Status do Sistema */}
              <div className="flex items-center space-x-2 text-sm">
                <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                <span className="text-gray-600">Sistema Online</span>
              </div>

              {/* Data/Hora */}
              <div className="text-sm text-gray-600">
                {new Date().toLocaleDateString('pt-BR', {
                  weekday: 'long',
                  year: 'numeric',
                  month: 'long',
                  day: 'numeric'
                })}
              </div>
            </div>
          </div>
        </header>

        {/* Page Content */}
        <main className="flex-1 overflow-y-auto bg-gray-50">
          {children}
        </main>
      </div>
    </div>
  );
};

export default Sidebar;

