import React, { useState, useEffect } from 'react';
import { Users, Shield, Settings, Eye, Edit, Trash2, Plus, Search, Filter } from 'lucide-react';

const UsuariosModule = () => {
  const [usuarios, setUsuarios] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [editingUser, setEditingUser] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterType, setFilterType] = useState('todos');

  // Dados simulados
  const usuariosSimulados = [
    {
      id: 1,
      nome: 'Administrador Villa Joias',
      email: 'admin@villajoias.com',
      tipo_usuario: 'administrador',
      ativo: true,
      ultimo_login: '2025-06-19T17:30:00',
      data_criacao: '2025-01-01T00:00:00'
    },
    {
      id: 2,
      nome: 'Maria Silva',
      email: 'maria@villajoias.com',
      tipo_usuario: 'gerente',
      ativo: true,
      ultimo_login: '2025-06-19T16:45:00',
      data_criacao: '2025-02-15T00:00:00'
    },
    {
      id: 3,
      nome: 'João Santos',
      email: 'joao@villajoias.com',
      tipo_usuario: 'operador',
      ativo: true,
      ultimo_login: '2025-06-19T15:20:00',
      data_criacao: '2025-03-10T00:00:00'
    },
    {
      id: 4,
      nome: 'Ana Costa',
      email: 'ana@villajoias.com',
      tipo_usuario: 'consulta',
      ativo: false,
      ultimo_login: '2025-06-15T10:30:00',
      data_criacao: '2025-04-05T00:00:00'
    }
  ];

  useEffect(() => {
    // Simular carregamento
    setTimeout(() => {
      setUsuarios(usuariosSimulados);
      setLoading(false);
    }, 1000);
  }, []);

  const tiposUsuario = {
    'administrador': { label: 'Administrador', color: 'bg-red-100 text-red-800', icon: '👑' },
    'gerente': { label: 'Gerente', color: 'bg-blue-100 text-blue-800', icon: '👔' },
    'operador': { label: 'Operador', color: 'bg-green-100 text-green-800', icon: '👤' },
    'consulta': { label: 'Consulta', color: 'bg-gray-100 text-gray-800', icon: '👁️' },
    'personalizado': { label: 'Personalizado', color: 'bg-purple-100 text-purple-800', icon: '⚙️' }
  };

  const formatarData = (dataString) => {
    const data = new Date(dataString);
    return data.toLocaleDateString('pt-BR') + ' às ' + data.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
  };

  const usuariosFiltrados = usuarios.filter(usuario => {
    const matchSearch = usuario.nome.toLowerCase().includes(searchTerm.toLowerCase()) ||
                       usuario.email.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchFilter = filterType === 'todos' || 
                       (filterType === 'ativos' && usuario.ativo) ||
                       (filterType === 'inativos' && !usuario.ativo) ||
                       usuario.tipo_usuario === filterType;
    
    return matchSearch && matchFilter;
  });

  const ModalUsuario = () => {
    const [formData, setFormData] = useState({
      nome: '',
      email: '',
      tipo_usuario: 'operador',
      ativo: true,
      senha: '',
      observacoes: ''
    });

    useEffect(() => {
      if (editingUser) {
        setFormData({
          nome: editingUser.nome,
          email: editingUser.email,
          tipo_usuario: editingUser.tipo_usuario,
          ativo: editingUser.ativo,
          senha: '',
          observacoes: editingUser.observacoes || ''
        });
      }
    }, [editingUser]);

    const handleSubmit = (e) => {
      e.preventDefault();
      
      if (editingUser) {
        // Atualizar usuário
        setUsuarios(prev => prev.map(u => 
          u.id === editingUser.id 
            ? { ...u, ...formData, senha: undefined }
            : u
        ));
      } else {
        // Criar novo usuário
        const novoUsuario = {
          id: Date.now(),
          ...formData,
          data_criacao: new Date().toISOString(),
          ultimo_login: null
        };
        setUsuarios(prev => [...prev, novoUsuario]);
      }
      
      setShowModal(false);
      setEditingUser(null);
      setFormData({
        nome: '',
        email: '',
        tipo_usuario: 'operador',
        ativo: true,
        senha: '',
        observacoes: ''
      });
    };

    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div className="bg-white rounded-lg p-6 w-full max-w-md max-h-[90vh] overflow-y-auto">
          <div className="flex justify-between items-center mb-4">
            <h3 className="text-lg font-semibold text-gray-900">
              {editingUser ? 'Editar Usuário' : 'Novo Usuário'}
            </h3>
            <button
              onClick={() => {
                setShowModal(false);
                setEditingUser(null);
              }}
              className="text-gray-400 hover:text-gray-600"
            >
              ✕
            </button>
          </div>

          <form onSubmit={handleSubmit} className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Nome Completo
              </label>
              <input
                type="text"
                value={formData.nome}
                onChange={(e) => setFormData(prev => ({ ...prev, nome: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
                required
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Email
              </label>
              <input
                type="email"
                value={formData.email}
                onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
                required
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Tipo de Usuário
              </label>
              <select
                value={formData.tipo_usuario}
                onChange={(e) => setFormData(prev => ({ ...prev, tipo_usuario: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
              >
                {Object.entries(tiposUsuario).map(([key, tipo]) => (
                  <option key={key} value={key}>
                    {tipo.icon} {tipo.label}
                  </option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                {editingUser ? 'Nova Senha (deixe vazio para manter)' : 'Senha'}
              </label>
              <input
                type="password"
                value={formData.senha}
                onChange={(e) => setFormData(prev => ({ ...prev, senha: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
                required={!editingUser}
              />
            </div>

            <div>
              <label className="flex items-center">
                <input
                  type="checkbox"
                  checked={formData.ativo}
                  onChange={(e) => setFormData(prev => ({ ...prev, ativo: e.target.checked }))}
                  className="mr-2"
                />
                <span className="text-sm font-medium text-gray-700">Usuário Ativo</span>
              </label>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Observações
              </label>
              <textarea
                value={formData.observacoes}
                onChange={(e) => setFormData(prev => ({ ...prev, observacoes: e.target.value }))}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
                rows="3"
              />
            </div>

            <div className="flex gap-3 pt-4">
              <button
                type="button"
                onClick={() => {
                  setShowModal(false);
                  setEditingUser(null);
                }}
                className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50"
              >
                Cancelar
              </button>
              <button
                type="submit"
                className="flex-1 px-4 py-2 bg-yellow-500 text-white rounded-md hover:bg-yellow-600"
              >
                {editingUser ? 'Atualizar' : 'Criar'}
              </button>
            </div>
          </form>
        </div>
      </div>
    );
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-yellow-500"></div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex justify-between items-center mb-6">
        <div>
          <h1 className="text-2xl font-bold text-gray-900 flex items-center">
            <Users className="h-8 w-8 text-yellow-500 mr-3" />
            Gestão de Usuários
          </h1>
          <p className="text-gray-600 mt-1">Gerencie usuários e permissões do sistema</p>
        </div>
        <button
          onClick={() => setShowModal(true)}
          className="bg-yellow-500 text-white px-4 py-2 rounded-lg hover:bg-yellow-600 flex items-center"
        >
          <Plus className="h-4 w-4 mr-2" />
          Novo Usuário
        </button>
      </div>

      {/* Filtros */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-1">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
              <input
                type="text"
                placeholder="Buscar por nome ou email..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500"
              />
            </div>
          </div>
          <div className="sm:w-48">
            <div className="relative">
              <Filter className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
              <select
                value={filterType}
                onChange={(e) => setFilterType(e.target.value)}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500 appearance-none"
              >
                <option value="todos">Todos</option>
                <option value="ativos">Ativos</option>
                <option value="inativos">Inativos</option>
                <option value="administrador">Administradores</option>
                <option value="gerente">Gerentes</option>
                <option value="operador">Operadores</option>
                <option value="consulta">Consulta</option>
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Lista de Usuários */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Usuário
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Tipo
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Último Login
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Ações
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {usuariosFiltrados.map((usuario) => (
                <tr key={usuario.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{usuario.nome}</div>
                      <div className="text-sm text-gray-500">{usuario.email}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${tiposUsuario[usuario.tipo_usuario].color}`}>
                      <span className="mr-1">{tiposUsuario[usuario.tipo_usuario].icon}</span>
                      {tiposUsuario[usuario.tipo_usuario].label}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                      usuario.ativo 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-red-100 text-red-800'
                    }`}>
                      {usuario.ativo ? '✅ Ativo' : '❌ Inativo'}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {usuario.ultimo_login ? formatarData(usuario.ultimo_login) : 'Nunca'}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <div className="flex space-x-2">
                      <button
                        onClick={() => {
                          // Ver detalhes
                          console.log('Ver detalhes:', usuario);
                        }}
                        className="text-blue-600 hover:text-blue-900"
                        title="Ver detalhes"
                      >
                        <Eye className="h-4 w-4" />
                      </button>
                      <button
                        onClick={() => {
                          setEditingUser(usuario);
                          setShowModal(true);
                        }}
                        className="text-yellow-600 hover:text-yellow-900"
                        title="Editar"
                      >
                        <Edit className="h-4 w-4" />
                      </button>
                      {usuario.tipo_usuario !== 'administrador' && (
                        <button
                          onClick={() => {
                            if (confirm('Tem certeza que deseja excluir este usuário?')) {
                              setUsuarios(prev => prev.filter(u => u.id !== usuario.id));
                            }
                          }}
                          className="text-red-600 hover:text-red-900"
                          title="Excluir"
                        >
                          <Trash2 className="h-4 w-4" />
                        </button>
                      )}
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {usuariosFiltrados.length === 0 && (
          <div className="text-center py-12">
            <Users className="mx-auto h-12 w-12 text-gray-400" />
            <h3 className="mt-2 text-sm font-medium text-gray-900">Nenhum usuário encontrado</h3>
            <p className="mt-1 text-sm text-gray-500">
              {searchTerm || filterType !== 'todos' 
                ? 'Tente ajustar os filtros de busca.'
                : 'Comece criando um novo usuário.'}
            </p>
          </div>
        )}
      </div>

      {/* Modal */}
      {showModal && <ModalUsuario />}
    </div>
  );
};

export default UsuariosModule;

